<?php
/**
 * Class WC_Email_New_Order file
 *
 * @package WooCommerce\Emails
 */

if (! defined('ABSPATH')) {
    exit;
}

if (! class_exists('WC_Email_Pre_Order_Alert')) :

    /**
     * New Order Email.
     *
     * An email sent to the admin when a new order is received/paid for.
     *
     * @class       WC_Email_Pre_Order_Alert
     * @extends     WC_Email
     */
    class WC_Email_Pre_Order_Alert extends WC_Email
    {

        /**
         * Constructor.
         */
        public function __construct()
        {
            $this->id             = 'new_preorder_alert';
            $this->title          = __('Pre-order Alert', 'preorders-for-woocommerce-pro');
            $this->description    = __('Pre-order alert emails are sent to chosen recipient(s) based on the alert settings.', 'preorders-for-woocommerce-pro');
            $this->template_html  = 'emails/preorder-alert.php';
            $this->template_plain = 'emails/plain/preorder-alert.php';
            
            
            // Call parent constructor.
            parent::__construct();

            // Other settings.
            $this->template_base  = WCPO_TEMPLATE_PATH;
            $this->recipient = $this->get_option('recipient', get_option('admin_email'));
        }
     

        /**
         * Get email subject.
         *
         * @since  3.1.0
         * @return string
         */
        public function get_default_subject()
        {
            return __('[{site_title}]: Pre-order Alert!', 'preorders-for-woocommerce-pro');
        }

        /**
         * Get email heading.
         *
         * @since  3.1.0
         * @return string
         */
        public function get_default_heading()
        {
            return __('Pre-Order Alert!', 'preorders-for-woocommerce-pro');
        }

        /**
         * Trigger the sending of this email.
         *
         */
        public function trigger()
        {
            // woohoo, send the email!
            $this->send($this->get_recipient(), $this->get_subject(), $this->get_content(), $this->get_headers(), $this->get_attachments());
        }

        /**
         * Get content html.
         *
         * @return string
         */
        public function get_content_html()
        {
            return wc_get_template_html(
                $this->template_html,
                array(
                  
                    'email_heading'      => $this->get_heading(),
                    'additional_content' => $this->get_additional_content(),
                    'sent_to_admin'      => true,
                    'plain_text'         => false,
                    'email'              => $this,
                ),
                '',
                $this->template_base
            );
        }

        /**
         * Get content plain.
         *
         * @return string
         */
        public function get_content_plain()
        {
            return wc_get_template_html(
                $this->template_plain,
                array(
                
                    'email_heading'      => $this->get_heading(),
                    'additional_content' => $this->get_additional_content(),
                    'sent_to_admin'      => true,
                    'plain_text'         => true,
                    'email'              => $this,
                ),
                '',
                $this->template_base
            );
        }

        /**
         * Default content to show below main email content.
         *
         * @since 3.7.0
         * @return string
         */
        public function get_default_additional_content()
        {
            return __('Email Notification for preorder.', 'preorders-for-woocommerce-pro');
        }

        /**
         * Initialise settings form fields.
         */
        public function init_form_fields()
        {
            /* translators: %s: list of placeholders */
            $placeholder_text  = sprintf(__('Available placeholders: %s', 'preorders-for-woocommerce-pro'), '<code>' . implode('</code>, <code>', array_keys($this->placeholders)) . '</code>');
            $this->form_fields = array(
                'enabled'            => array(
                    'title'   => __('Enable/Disable', 'preorders-for-woocommerce-pro'),
                    'type'    => 'checkbox',
                    'label'   => __('Enable this email notification', 'preorders-for-woocommerce-pro'),
                    'default' => 'yes',
                ),
                'recipient'          => array(
                    'title'       => __('Recipient(s)', 'preorders-for-woocommerce-pro'),
                    'type'        => 'text',
                    /* translators: %s: WP admin email */
                    'description' => sprintf(__('Enter recipients (comma separated) for this email. Defaults to %s.', 'preorders-for-woocommerce-pro'), '<code>' . esc_attr(get_option('admin_email')) . '</code>'),
                    'placeholder' => '',
                    'default'     => '',
                    'desc_tip'    => true,
                ),
                'subject'            => array(
                    'title'       => __('Subject', 'preorders-for-woocommerce-pro'),
                    'type'        => 'text',
                    'desc_tip'    => true,
                    'description' => $placeholder_text,
                    'placeholder' => $this->get_default_subject(),
                    'default'     => '',
                ),
                'heading'            => array(
                    'title'       => __('Email heading', 'preorders-for-woocommerce-pro'),
                    'type'        => 'text',
                    'desc_tip'    => true,
                    'description' => $placeholder_text,
                    'placeholder' => $this->get_default_heading(),
                    'default'     => '',
                ),
                'additional_content' => array(
                    'title'       => __('Additional content', 'preorders-for-woocommerce-pro'),
                    'description' => __('Text to appear below the main email content.', 'preorders-for-woocommerce-pro') . ' ' . $placeholder_text,
                    'css'         => 'width:400px; height: 75px;',
                    'placeholder' => __('N/A', 'preorders-for-woocommerce-pro'),
                    'type'        => 'textarea',
                    'default'     => $this->get_default_additional_content(),
                    'desc_tip'    => true,
                ),
                'email_type'         => array(
                    'title'       => __('Email type', 'preorders-for-woocommerce-pro'),
                    'type'        => 'select',
                    'description' => __('Choose which format of email to send.', 'preorders-for-woocommerce-pro'),
                    'default'     => 'html',
                    'class'       => 'email_type wc-enhanced-select',
                    'options'     => $this->get_email_type_options(),
                    'desc_tip'    => true,
                ),
            );
        }
    }

endif;

return new WC_Email_Pre_Order_Alert();
