<?php
/**
 * Class WC_Preorder_Ready file.
 *
 * @package WooCommerce\Emails
 */

if (! defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

if (! class_exists('WC_Preorder_Ready', false)) :

    /**
     * Customer Preorder  Email.
     *
     * An email sent to the customer when a new order is preorder for.

     * @package     WooCommerce\Classes\Emails
     * @extends     WC_Email
     */
    class WC_Preorder_Ready extends WC_Email
    {

        /**
         * Constructor.
         */
        public function __construct()
        {
            $this->id             = 'wc_preorder_available';
            $this->customer_email = true;
            $this->title          = __('Pre-order available today', 'preorders-for-woocommerce-pro');
            $this->description    = __('This e-mail is triggered when a pre-order is available.', 'preorders-for-woocommerce-pro');
            $this->template_html  = 'emails/customer-preorder-available.php';
            $this->template_plain = 'emails/plain/customer-preorder-available.php';
            $this->placeholders   = array(
                '{order_date}'   => '',
                '{order_number}' => '',
            );

            // Call parent constructor.
            parent::__construct();

            $this->template_base  = WCPO_TEMPLATE_PATH;
        }

        /**
         * Get email subject.
         *
         * @since  3.1.0
         * @return string
         */
        public function get_default_subject()
        {
            return __('[{site_title}]: Pre-Order Available #{order_number}', 'preorders-for-woocommerce-pro');
        }

        /**
         * Get email heading.
         *
         * @since  3.1.0
         * @return string
         */
        public function get_default_heading()
        {
            return __('Pre-Order Available #{order_number}', 'preorders-for-woocommerce-pro');
        }

        /**
         * Trigger the sending of this email.
         *
         * @param int            $order_id The order ID.
         * @param WC_Order|false $order Order object.
         */
        public function trigger($order_id, $order = false)
        {
            $this->setup_locale();

            if ($order_id && ! is_a($order, 'WC_Order')) {
                $order = wc_get_order($order_id);
            }

            if (is_a($order, 'WC_Order')) {
                $this->object                         = $order;
                $this->recipient                      = $this->object->get_billing_email();
                $this->placeholders['{order_date}']   = wc_format_datetime($this->object->get_date_created());
                $this->placeholders['{order_number}'] = $this->object->get_order_number();
            }

            if ($this->is_enabled() && $this->get_recipient()) {
                $this->send($this->get_recipient(), $this->get_subject(), $this->get_content(), $this->get_headers(), $this->get_attachments());
            }

            $this->restore_locale();
        }

        /**
         * Get content html.
         *
         * @return string
         */
        public function get_content_html()
        {
            return wc_get_template_html(
                $this->template_html,
                array(
                    'order'              => $this->object,
                    'email_heading'      => $this->get_heading(),
                    'additional_content' => $this->get_additional_content(),
                    'sent_to_admin'      => false,
                    'plain_text'         => false,
                    'email'              => $this,
                ),
                '',
                $this->template_base
            );
        }

        /**
         * Get content plain.
         *
         * @return string
         */
        public function get_content_plain()
        {
            return wc_get_template_html(
                $this->template_plain,
                array(
                    'order'              => $this->object,
                    'email_heading'      => $this->get_heading(),
                    'additional_content' => $this->get_additional_content(),
                    'sent_to_admin'      => false,
                    'plain_text'         => true,
                    'email'              => $this,
                ),
                '',
                $this->template_base
            );
        }

        /**
         * Default content to show below main email content.
         *
         * @since 3.7.0
         * @return string
         */
        public function get_default_additional_content()
        {
            return __('We look forward to fulfilling your pre-order soon.', 'preorders-for-woocommerce-pro');
        }
    }

endif;
return new WC_Preorder_Ready();
