<?php

namespace Woocommerce_Preorders;

class Sync
{
    public function __construct()
    {
        if (!wp_next_scheduled('check_for_released_preorders')) {
            wp_schedule_event(time(), WCPO_CRON_INTERVAL, 'check_for_released_preorders');
        }
        if (!wp_next_scheduled('send_preorder_email_alert')) {
            wp_schedule_event(time(), 'daily', 'send_preorder_email_alert');
        }
        if (!wp_next_scheduled('wc_send_pre_order_available_email')) {
            wp_schedule_event(time(), 'daily', 'wc_send_pre_order_available_email');
        }
        if (!wp_next_scheduled('wc_check_parent_order_status')) {
            wp_schedule_event(time(), 'twicedaily', 'wc_check_parent_order_status');
        }

        add_action('check_for_released_preorders', [$this,'checkForPreorders']);
        add_action('send_preorder_email_alert', [$this,'sendPreorderEmailAlert']);
        add_action('wc_send_pre_order_available_email', [$this, 'checkPreOrdersAvailable']);
        add_action('wc_check_parent_order_status', [$this, 'checkParentOrderPayment']);
    }
    /**
     * Preorder email alert for admin
     *
     * @return void
     */
    public function sendPreorderEmailAlert()
    {
        $args = [
            'status' => 'wc-pre-ordered'
        ];
        $preOrderedOrders = wc_get_orders($args);
        $notificationOrders = array();
        foreach ($preOrderedOrders as $order) {
            $preOrderDate = strtotime(get_post_meta($order->get_id(), '_preorder_date', true));

            $notifyTime = strtotime('-'.absint(get_option('wc_preorders_alert')).'day', $preOrderDate);
           
            $beginOfDay = strtotime('+'.absint(get_option('wc_preorders_alert')).'day', time());
            $beginOfDay = date('m/d/Y', $beginOfDay); // get date without time
            $beginOfDay = strtotime($beginOfDay); // get date as str format

            if ($notifyTime < time() && $order->get_status() == 'pre-ordered' && $beginOfDay == $preOrderDate) {
                $notificationOrders[] = $order->get_id();
            }
        }
        
       
        set_transient('temp_preorder_data', $notificationOrders, 60);

        if (!get_transient('temp_preorder_data')) {
            return;
        }

        add_action('preorder_alert_email_order', function () {?>
        <table class="td" width="100%" cellspacing="0" cellpadding="6" border="1"
            style="margin-bottom:15px; color: #636363; border: 1px solid #e5e5e5; vertical-align: middle; width: 100%; font-family: 'Helvetica Neue', Helvetica, Roboto, Arial, sans-serif;">
            <thead>
                <tr>
                    <th class="td" scope="col"
                        style="color: #636363; border: 1px solid #e5e5e5; vertical-align: middle; padding: 12px; text-align: left;">
                        <?php echo __('Orders', 'preorders-for-woocommerce-pro');?>
                    </th>
                    <th class="td" scope="col"
                        style="color: #636363; border: 1px solid #e5e5e5; vertical-align: middle; padding: 12px; text-align: left;">
                        <?php echo __('Total Items', 'preorders-for-woocommerce-pro');?>
                    </th>
                    <th class="td" scope="col"
                        style="color: #636363; border: 1px solid #e5e5e5; vertical-align: middle; padding: 12px; text-align: left;">
                        <?php echo __('Total Price', 'preorders-for-woocommerce-pro');?>
                    </th>
                </tr>
            </thead>
            <tbody>
                <?php foreach (get_transient('temp_preorder_data') as $key => $value) {?>
                <?php
                                    $order = wc_get_order($value);
                                    $diff = human_time_diff(time(), strtotime(get_post_meta($order->get_id(), '_preorder_date', true)));
                                ?>
                <tr>
                    <td class="td"
                        style="color: #636363; border: 1px solid #e5e5e5; padding: 12px; text-align: left; vertical-align: middle; font-family: 'Helvetica Neue', Helvetica, Roboto, Arial, sans-serif; word-wrap: break-word;">
                        <a href="<?php echo admin_url('post.php?post='.$value.'&action=edit') ; ?>"
                            target="_blank" style="font-weight: normal; text-decoration: underline; color: #96588a;"><?php echo __('Order #').$value; ?></a>
                        <small><?php echo sprintf(__('Available in %s', 'preorders-for-woocommerce-pro'), $diff); ?></small>
                    </td>
                    <td class="td"
                        style="color: #636363; border: 1px solid #e5e5e5; padding: 12px; text-align: left; vertical-align: middle; font-family: 'Helvetica Neue', Helvetica, Roboto, Arial, sans-serif; word-wrap: break-word;">
                        <?php echo esc_html($order->get_item_count()); ?>
                    </td>
                    <td class="td"
                        style="color: #636363; border: 1px solid #e5e5e5; padding: 12px; text-align: left; vertical-align: middle; font-family: 'Helvetica Neue', Helvetica, Roboto, Arial, sans-serif; word-wrap: break-word;">
                        <?php echo wc_price($order->get_total()); ?>
                    </td>
                </tr>
                <?php
                } ?>



            </tbody>
        </table>

        <?php
        });

        WC()->mailer()->get_emails()['WC_Pre_Order_Alert']->trigger();
    }

    /**
     * check preorders
     *
     * @return void
     */
    public function checkForPreOrders()
    {
        $args = [
            'status' => 'wc-pre-ordered'
        ];
        $status = get_option('wc_preorders_after_completed', 'wc-completed'); // set new status
        $preOrderedOrders = wc_get_orders($args);
        foreach ($preOrderedOrders as $order) {
            $preOrderDate = strtotime(get_post_meta($order->get_id(), '_preorder_date', true));

            if ($preOrderDate < time()) {
                $parentOrderId = $order->get_parent_id();
                if ($parentOrderId !== 0) {
                    $parentOrder = wc_get_order($parentOrderId);
                    if ($parentOrder->get_status() == 'completed') {
                        $order->update_status($status, '[WooCommerce Pre Orders] ');
                    }
                } else {
                    if ($order->get_status() == 'pre-ordered' && $order->payment_complete()) {
                        $order->update_status(apply_filters('change_order_status_on_preorder_date', $status), '[WooCommerce Pre Orders] ');
                    }
                }
            }
        }
    }
    
    /**
     * Check if the parent order is failed or cancel all preoders assign with that order
     * will be cancel.
     *
     * @return void
     */
    public function checkParentOrderPayment()
    {
        $args = [
            'post_type'         => 'shop_order',
            'post_status'       =>  'wc-pre-ordered',
            'posts_per_page'    => -1,
            
        ];
       
        $preOrderedOrders = get_posts($args);
        foreach ($preOrderedOrders as $order) {
            $preOrder = wc_get_order($order->ID);
          
            $parentOrder = wc_get_order($order->post_parent);
          
            if ('failed' == $parentOrder->get_status() || 'cancelled' == $parentOrder->get_status()) {
                $preOrder->update_status('cancelled', __('[Parent Order payment failed or cancelled]', 'preorders-for-woocommerce-pro'));
            }
        }
    }

    /**
     * heck the pre-orders available for today
     *
     * @return void
     */
    public function checkPreOrdersAvailable()
    {
        if (get_option('wc_preorder_customer_notification', 'yes') != 'yes') {
            return;
        } // check for notifictaion option
        
        $email_preorder_ready = WC()->mailer()->get_emails()['WC_Preorder_Ready'];
     
        $args = [
            'post_type'         => 'shop_order',
            'post_status'       =>  'wc-pre-ordered',
            'posts_per_page'    => -1,
            'meta_key'  => '_preorder_date',
            'meta_query' => array(
                array(
                    'key'     => '_preorder_date',
                    'compare' => '=',
                    'value' => date('Y-m-d'),
                    'type'    => 'DATE'
                ),
            ),
        ];
        $preOrderedOrders = get_posts($args);
    
        foreach ($preOrderedOrders as $order) {
            $email_preorder_ready->trigger($order->ID);
        }
    }
}
