<?php

namespace Woocommerce_Preorders;

class Checkout
{
    private $preordersMode;
    private $cart;
    private $emailIds;

    public function __construct()
    {
        $this->preordersMode = get_option('wc_preorders_mode');

        $this->cart = new Cart();

        if ('either' === $this->preordersMode) {
            add_filter('woocommerce_add_to_cart_validation', [$this->cart, 'allowOneTypeOnly'], 99, 2);
        }
       
        add_filter('woocommerce_billing_fields', [$this, 'addShippingDateField']);

        if (!\in_array(get_option('wc_preorders_mode'), ['whole', 'either'], true)) {
            add_action('woocommerce_thankyou', [$this, 'purgePreOrderedItems'], 100, 1);
        } else {
            add_action('woocommerce_thankyou', [$this, 'checkGeneratedOrderStatus'], 100, 1);
        }

        add_action('woocommerce_checkout_update_order_meta', [$this, 'managePreOrders'], 10, 2);
        

        // New order notification only for "Pending" Order status
        
        add_action('woocommerce_order_status_partially-paid', [$this, 'newPreOderNotification']);

        add_filter('woocommerce_available_payment_gateways', [$this,'conditionalPaymentGateways'], 10, 1);
        add_filter('woocommerce_payment_complete_order_status', [$this,'setPreroderStatus'], 10, 3);
        add_action('woocommerce_order_status_changed', [$this,'emailNotifications'], 10, 4);
        add_filter('woocommerce_email_enabled_new_order', [$this,'newOrderEmail'], 10, 2);
        add_filter('woocommerce_email_enabled_customer_processing_order', [$this,'processingOrderEmail'], 10, 2);
        add_filter('woocommerce_email_enabled_customer_on_hold_order', [$this,'onholdOrderEmail'], 10, 2);
        add_filter('woocommerce_email_enabled_customer_completed_order', [$this,'completedOrderEmail'], 10, 2);
        
        if (get_option('wc_preorders_avaiable_date_checkout_item') == 'yes') {
            add_filter('woocommerce_checkout_cart_item_quantity', [$this, 'PreorderDateInCheckoutItems'], 10, 2);
        }
        add_action('woocommerce_thankyou', [$this, 'sendOrderEmail'], 9999, 1);
        // We don't need this feature for now
        //add_filter('woocommerce_package_rates', [$this, 'manageShippingCosts'], 10, 2);
    }
    /**
     * Preroder date validator
     */
    private function orderDateValidator($preorder, $data, $orderId = '')
    {
        $oldestDate = str_replace([' 00:00:00','-'], ['',''], $this->cart->getOldestDate());
        $checkOutDate = str_replace('-', '', $data['preorder_date']);

        if (!empty($orderId)) {
            if ($checkOutDate <= $oldestDate) {
                update_post_meta($orderId, '_preorder_date', $this->cart->getOldestDate());
            } else {
                update_post_meta($orderId, '_preorder_date', esc_attr($data['preorder_date']));
            }
            return;
        }


        if ($checkOutDate < $oldestDate) {
            $preorder->update_meta_data('_preorder_date', $this->cart->getOldestDate());
        } else {
            $preorder->update_meta_data('_preorder_date', $data['preorder_date']);
        }
    }
    /**
     * Sends normal order and invoice email to the customer when the user arrives to the thank you page.
     */
    public function sendOrderEmail($orderId)
    {
        $orderObj = wc_get_order($orderId);
        $email_new_order = WC()->mailer()->get_emails()['WC_Email_New_Order'];
        $emailProcessingOrder = WC()->mailer()->get_emails()['WC_Email_Customer_Processing_Order'];
        $emailOnHoldOrder = WC()->mailer()->get_emails()['WC_Email_Customer_On_Hold_Order'];

      
        $hasPreorderedProductsOnly = count($orderObj->get_items()) === count($this->getPreorderedProducts($orderObj));
        $purgePreorderedItems = get_post_meta($orderObj->get_id(), '_purge_preordered_items', true);
        
        // We're only firing these emails if there's only a non-preordered product present.
        if (!$hasPreorderedProductsOnly && $purgePreorderedItems == 1) {
            $email_new_order->trigger($orderId);

            if ($orderObj->get_status() == 'on-hold') {
                $emailOnHoldOrder->trigger($orderId);
            } elseif ($orderObj->get_status() ==  'processing') {
                $emailProcessingOrder->trigger($orderId);
            }
        }
    }
    /**
    * Prevent 'preorder' new order email
    *
    * @param [type] $enabled
    * @param [type] $object
    * @return void
    */
    public function newOrderEmail($enabled, $object)
    {
        //  error_log('is_enabled: ' . current_filter());
        if ($object == null) {
            return $enabled;
        }
        $orderId = $object->get_order_number();
        $order = wc_get_order($orderId);
        if (!$order) {
            return $enabled;
        }
        if (count($this->getPreorderedProducts($order)) >= 1) {
            return false;
        }
        
        
        return $enabled;
    }
    public function processingOrderEmail($enabled, $object)
    {
        //  error_log('is_enabled: ' . current_filter());
        if ($object == null) {
            return $enabled;
        }
        $orderId = $object->get_order_number();
        $order = wc_get_order($orderId);
        $preorderDate = get_post_meta($orderId, '_preorder_date', true);
        if (!$order) {
            return $enabled;
        }

        if (count($this->getPreorderedProducts($order)) >= 1 && !empty($preorderDate)) {
            return $enabled;
        } // if the order is a preorder

        if (count($this->getPreorderedProducts($order)) >= 1) {
            return false;
        }
        
        return $enabled;
    }
    public function completedOrderEmail($enabled, $object)
    {
        //  error_log('is_enabled: ' . current_filter());
        if ($object == null) {
            return $enabled;
        }
        $orderId = $object->get_order_number();
        $order = wc_get_order($orderId);
        $preorderDate = get_post_meta($orderId, '_preorder_date', true);

        if (count($this->getPreorderedProducts($order)) >= 1 && !empty($preorderDate)) {
            return $enabled;
        } // if the order is a preorder
        
        if (count($this->getPreorderedProducts($order)) >= 1) {
            do_action('preorder_send_default_wc_email_'. current_filter());

            return false;
        }
        
        return $enabled;
    }
    public function onholdOrderEmail($enabled, $object)
    {
        //  error_log('is_enabled: ' . current_filter());
        if ($object == null) {
            return $enabled;
        }
        $orderId = $object->get_order_number();
        $order = wc_get_order($orderId);
        $preorderDate = get_post_meta($orderId, '_preorder_date', true);

        if (count($this->getPreorderedProducts($order)) >= 1 && !empty($preorderDate)) {
            return $enabled;
        } // if the order is a preorder

        
        if (count($this->getPreorderedProducts($order)) >= 1) {
            do_action('preorder_send_default_wc_email_'. current_filter());

            return false;
        }
        
        
        return $enabled;
    }

    /**
     * send preorder related emails
     *
     * @param [int] $order_id
     * @param [string] $old_status
     * @param [string] $new_status
     * @param [object] $order
     * @return void
     */
    public function emailNotifications($order_id, $old_status, $new_status, $order)
    {
        if ($old_status == 'pending' && $new_status == 'pre-ordered') {
          
            // Send "New Email" notification (to customer)
            WC()->mailer()->get_emails()['WC_New_Customer_Pre_Order_Email']->trigger($order_id);
            WC()->mailer()->get_emails()['WC_New_Pre_Order_Email']->trigger($order_id);
        }
    }
    /**
     * Set main order status 'pre-ordered' after payment complete
     *
     * @param [string] $status
     * @param [int] $order_id
     * @param [type] $order
     * @return status
     */
    public function setPreroderStatus($status, $order_id, $order)
    {
        if (get_post_meta($order_id, '_preorder_date', true)) {
            return 'pre-ordered';
        }

        return $status;
    }
    /**
     * Disbale Payment gateways
     *
     * @param [array] $availableGateways
     * @return void
     */
    public function conditionalPaymentGateways($availableGateways)
    {
        // Not in backend (admin)
        if (is_admin() || !is_checkout()) {
            return $availableGateways;
        }
        if (bootstrap::cartHasPreorderItems() && get_option('wc_preorders_disable_payment')) {
            foreach (get_option('wc_preorders_disable_payment') as $key => $gateway) {
                unset($availableGateways[$gateway]);
            }
        }
        return $availableGateways;
    }
    // Display a preorder date under cart item name in checkout
    public function PreorderDateInCheckoutItems($item_qty, $cart_item)
    {
        // Here below define your shipping class slug
        $isPreoder = get_post_meta($cart_item['product_id'], '_is_pre_order', true);
        $PreOrderDate = get_post_meta($cart_item['product_id'], '_pre_order_date', true);
        if ('yes' == $isPreoder && strtotime($PreOrderDate) > time()) {
            $timeFormat = date_i18n(get_option('date_format'), strtotime($PreOrderDate)) ;
            $humanTime = human_time_diff(time(), strtotime($PreOrderDate));
          

            $text =  Shop::replaceDateTxt(get_option('wc_preorders_avaiable_date_text'), $timeFormat, $humanTime, $timeFormat);

            $item_qty .= '<br />'.apply_filters('preorder_avaiable_date_text', $text);
        }
        return $item_qty;
    }
    
    /**
     * Send Pre-Order emails fired by an action - preorder_email at the end of the switch statement in the managePreOrders() method.
    */
    public function newPreOderNotification($order_id)
    {
        // Get an instance of the WC_Order object (same as before)
        $order = wc_get_order($order_id);
        $order_status  = $order->get_status(); // Get the order status


        if ($order_status == 'partially-paid' && get_post_meta($order_id, '_preorder_date', true)) {
            // fix compability issue with WooCommerce Deposit plugin by Webtomizer
            WC()->mailer()->get_emails()['WC_New_Customer_Pre_Order_Email']->trigger($order_id);
        } elseif ($order_status == 'pre-ordered') {
            // Send "New Email" notification to customer
            WC()->mailer()->get_emails()['WC_New_Customer_Pre_Order_Email']->trigger($order_id);
        }
        
        // Send "New Email" notification admin
        if (get_post_meta($order_id, 'sent_new_preorder_email', true) != 1) {
            WC()->mailer()->get_emails()['WC_New_Pre_Order_Email']->trigger($order_id);
        }

        update_post_meta($order_id, 'sent_new_preorder_email', 1);
    }


    public function addShippingDateField($fields)
    {
        if (!is_checkout() && !is_cart()) {
            return $fields;
        }
        if ('no' === get_option('wc_preorders_always_choose_date')) {
            $class = ['disabled-input', 'form-row-wide'];
        } else {
            $class = ['form-row-wide'];
        }
        global $woocommerce;
        $cart = $woocommerce->cart->get_cart();
        $this->cart->checkPreOrderProducts($cart);
        if (\count($this->cart->getPreOrderProducts()) > 0) {
            $oldestDate = str_replace(' 00:00:00', '', $this->cart->getOldestDate());
            $fields['preorder_date'] = [
                'label' => __('Pre order Date', 'preorders-for-woocommerce-pro'),
                'type' => 'text',
                'class' => $class,
                'description' => __('Please enter the date when you want to receive your order', 'preorders-for-woocommerce-pro'),
                // 'input_class'   => 'datepicker',
                'priority' => 35,
                'required' => true,
                'default' => $oldestDate,
                'custom_attributes' => ['data-pre_order_date' => $oldestDate],
            ];
        }

        return $fields;
    }

    public function manageShippingCosts($rates, $package)
    {
        $factor = 1;
        if ('individual' === $this->preordersMode) {
            /*
            * If we are on "individual" mode, then we will have to multiply it by the number of
            * orders that we are going to generate.
            */
            global $woocommerce;
            $cart = $woocommerce->cart->get_cart();
            $this->cart->checkPreOrderProducts($cart);
            if (\count($this->cart->getPreOrderProducts()) > 0) {
                $factor = 1 + \count($this->cart->getPreOrderProducts());
            }
        } elseif ('partial' === $this->preordersMode) {
            /*
            * If we are in partial mode and the "multiply shipping" option is enabled,
            * then we will have to multiply our shipping costs by 2
            */
            $factor = 2;
        }
        foreach ($rates as $id => $rate) {
            $rates[$id]->cost *= $factor;
        }

        return $rates;
    }
    public function managePreOrders($orderId, $data)
    {
        $order = wc_get_order($orderId);
        /*
         * Small hack: if all our products were set up as pre-ordered and the user chose
         * to treat the order as partial, then it will be the same as treating the order
         * as whole. Check our docs for more info.
         */
        if (('partial' === $this->preordersMode && \count($order->get_items()) === \count($this->cart->getPreOrderProducts()))) {
            $this->preordersMode = 'whole';
        }

        /*
         * Once an order is created, we will look for the different pre-ordering methods that we work with,
         * and add the required logic in each case. The pre-ordering methods are 3 so far:
         *
         * #1 - Treat the whole order as a pre-order
         * If you choose this mode, the customer will be able to select a shipping date, and all products will be shipped together at that specific day.
         * That day will be limited to the latest pre-order date available, for instance, if the order has 3 different products marked as pre-order ones:
         * Product A will be available on the 1st of September
         * Product B will be available on the 3rd of September
         * Product C will be available on the 1st of October
         * Then, the minimum shipping date will be the 1st of October (i.e: the minimum date when all products will be available).
         *
         * #2 - Generate two separate orders, one for pre-orders and one for in-stock products
         * If you choose this mode, the customer will get two different orders generated, one for pre-order products and one for in-stock ones. For instance, if your order looks like this:
         * Product A is in-stock
         * Product B will be available on the 1st of September
         * Product C is in-stock
         * Then you will get an order which will be processed now, with products A and C, and then another order which will be shipped on the 1st of September.
         *
         * #3 - Generate separate orders for each pre-order products
         * If you choose this mode, then the customer will get one order for all in-stock products, and then one order for each pre-order product which will be shipped in each specific pre-order date.
         *
         * Check our docs for more info.
         */
        if ('whole' !== $this->preordersMode) {
            $preorderedProducts = $this->getPreorderedProducts($order);

            /*
             * We need to generate two arrays with all the billing and shipping info
             * We only have access to the post data from our order, and that data
             * comes with some prefixes and in a different format than what WooCommerce understands.
             * In order to fix this, we will need to first look for the prefixes keys with the prefixes
             * billing_ and shipping_ from the posted data, and then strip those prefixes from the generated /
             * filtered array. In other words, we will first look for all the keys starting with a certain prefix
             * using the method filterFields, and then strip those prefixes with the method stripFieldsPrefix.
             * I really hope this makes sense.
             */
            $billingAddress = $this->getFilteredFields('billing_', $data);
            $shippingAddress = $this->getFilteredFields('shipping_', $data);
        }

        // Calculate shipping split factor
        $factor = $this->getSplitShippingFactor($order);

        switch ($this->preordersMode) {
            /*
             * Case #2: Generate two separate orders, one for pre-orders and one for in-stock products
             * We will create a new order and set its status as pre-ordered in case this option is enabled.
             * Once we created the new order and the payment was completed, we will re-edit the original order
             * and remove the pre-order items from there.
             */
            case 'partial':
                if (empty($preorderedProducts)) {
                    return;
                }
                $preorder = wc_create_order();

                foreach ($preorderedProducts as $product) {
                    $productId = 0 !== $product->get_variation_id() ? $product->get_variation_id() : $product->get_product_id();
                    $preorder->add_product(wc_get_product($productId), $product->get_quantity());
                }
                $preorder->set_address($billingAddress, 'billing');
                $preorder->set_address($shippingAddress, 'shipping');

                $shippingDataFirst = true;
                $shippingData = [];
                $shippingMetaData = [];
                foreach ($order->get_items('shipping') as $key => $value) {
                    if ($shippingDataFirst) {
                        $shippingData = $value->get_data();
                        $shippingMetaData = $value->get_meta_data()[0];
                        $shippingDataFirst = false;
                    }
                }
                
                $shippingMethodArray = $shippingData;
                $shippingMethod = ($shippingData) ? $shippingMethodArray : array();
                // check for shipping cost option
                // @since 1.0.10
                if ('yes' != get_option('wc_preorders_prevent_shipping_cost')) {
                    $shippingMethodCurated = new \WC_Shipping_Rate($shippingMethod['id'], $shippingMethod['method_title'], $shippingMethod['total'] / $factor, $shippingMethod['total_tax'] / $factor, $shippingMethod['method_id'], $shippingMethod['instance_id']);
                    ($shippingMetaData) ? $shippingMethodCurated->add_meta_data($shippingMetaData->key, $shippingMetaData->value) : '';
                    ($shippingData) ? $preorder->add_shipping($shippingMethodCurated) : '';
                }

                $preorder->calculate_totals();
                $preorder->set_status('wc-pre-ordered');
                $preorder->set_parent_id($orderId);
                $preorder->set_customer_id(get_current_user_id());

                $this->OrderDateValidator($preorder, $data);

                $preorder->save();
               

                update_post_meta($orderId, 'preorder_date', '');

                // store all preorder ids to later pass in an action callback
                $this->emailIds['preorderIds'][] = $preorder->get_id();

            break;
            /*
             * Case #3: generate one order for all in-stock products, and then one order for each pre-order product
             * which will be shipped in each specific pre-order date.
             */
            case 'individual':
                // TODO: we are repeating too much code between cases #2 and #3
                
                $i = 0;
                $order = wc_get_order($orderId);
                $physicalProduts = [];

                foreach ($preorderedProducts as $product) {
                    $preorder = wc_create_order();
                    $productId = 0 !== $product->get_variation_id() ? $product->get_variation_id() : $product->get_product_id();
                    $preorder->add_product(wc_get_product($productId), $product->get_quantity());
                    $preorder->set_address($billingAddress, 'billing');
                    $preorder->set_address($shippingAddress, 'shipping');
                    $preorder->set_customer_id(get_current_user_id());

                    $wproduct = wc_get_product($productId);
                    $physicalProduts[] = $wproduct->is_downloadable();
                    
                    $shippingData = [];
                    $shippingMetaData = [];
                    foreach ($order->get_items('shipping') as $key => $value) {
                        $shippingData[] = $value->get_data();
                        $shippingMetaData[] = $value->get_meta_data()[0];
                    }
                
                    $shippingData = $shippingData[$i];
                    $shippingMethodArray = $shippingData;
                    $shippingMethod = ($shippingData) ? $shippingMethodArray : array();

                    
                    // check for shipping cost option
                    // @since 1.0.10
                    if ('yes' != get_option('wc_preorders_prevent_shipping_cost') && in_array(false, $physicalProduts)) {
                        $shippingMethodCurated = new \WC_Shipping_Rate($shippingMethod['id'], $shippingMethod['method_title'], $shippingMethod['total'] / $factor, $shippingMethod['total_tax'] / $factor, $shippingMethod['method_id'], $shippingMethod['instance_id']);
                        ($shippingMetaData) ? $shippingMethodCurated->add_meta_data($shippingMetaData[$i]->key, $shippingMetaData[$i]->value) : '';

                        ($shippingData) ? $preorder->add_shipping($shippingMethodCurated) : '';
                    }
                   
                    $preorder->calculate_totals();
                    $preorder->set_parent_id($orderId);
                    $preorder->set_status('wc-pre-ordered');

                    /*
                     * Check if the customer is able to choose a shipping date. If so, set the pre-order release
                     * date for the chose date. Otherwise get that info from each product.
                     */
                    if ('yes' === get_option('wc_preorders_always_choose_date')) {
                        $this->OrderDateValidator($preorder, $data);
                    } else {
                        $shippingDate = get_post_meta($productId, '_pre_order_date', true);
                        $preorder->update_meta_data('_preorder_date', $shippingDate);
                    }

                    $preorder->update_meta_data('_payment_method', get_post_meta($orderId, '_payment_method', true));
                    $preorder->update_meta_data('_payment_method_title', get_post_meta($orderId, '_payment_method', true));
                    $preorder->update_meta_data('_created_via', get_post_meta($orderId, '_created_via', true));
                    $preorder->update_meta_data('_date_paid', get_post_meta($orderId, '_date_paid', true));
                    $preorder->update_meta_data('_paid_date', get_post_meta($orderId, '_paid_date', true));

                    $preorder->save();
                    
                    // store all preorder ids to later pass in an action callback
                    $this->emailIds['preorderIds'][] = $preorder->get_id();

                    $i++;
                }
                
                $items = (array) $order->get_items('shipping');
                $numItems = count($items);
                $i = 0;
                if (sizeof($items) > 0) {
                    // Loop through shipping items
                    foreach ($items as $item_id => $item) {
                        if (++$i != $numItems) {
                            $order->remove_item($item_id);
                        }
                    }
                    $order->calculate_totals();
                }

                if (count($order->get_items()) == count($this->getPreorderedProducts($order))) {
                    $shippingItems = (array) $order->get_shipping_methods();
                    $taxItems = (array) $order->get_taxes();
                    if (sizeof($shippingItems) > 0) {
                        // Loop through shipping items
                        foreach ($shippingItems as $item_id => $item) {
                            wc_delete_order_item($item_id);
                            wc_delete_order_item(array_key_first($taxItems));
                            $order->calculate_totals();
                        }
                    }
                    $order->set_status('trash');
                    $order->save();
                }
     
        
            break;
            default:
                /*
                * Case #1: treat the whole order as a pre-order
                * Check if the order is of type partial or individual, and if not set the whole order as pre-ordered
                */
                if (isset($data['preorder_date'])) {
                    $this->orderDateValidator($preorder, $data, $orderId);
                    // update_post_meta($orderId, '_preorder_date', esc_attr($data['preorder_date']));
                    $order->set_status('wc-pre-ordered');
                }

                /**
                 * Option number 4 is treated as a whole but only with preordered products, we're checking here
                 * if we're only having preordered products whether we're on the 1st or 4th option as
                 * both are treated as whole orders, the difference being having mixed products with the first option.
                 */
                if ($this->orderHasOnlyPreorderedProducts($order, $this->cart)) {
                    $this->emailIds['preorderIds'][] = $orderId;
                }

                break;
        }

        // main action firing emails.
        do_action('preorder_email', $this->emailIds);
    }

    public function orderHasOnlyPreorderedProducts($orderObj, $cartObj)
    {
        return count($orderObj->get_items()) === count($cartObj->getPreOrderProducts());
    }

    public function checkGeneratedOrderStatus($order_id)
    {
        $order = wc_get_order($order_id);
        $items = $order->get_data();
        $this->cart->checkPreOrderProducts($items);
        if (\count($this->cart->getPreOrderProducts()) > 0) {
            $order->update_status('pre-ordered');
        }
    }

    /**
     * Check generated order and look for pre-order items.
     * If we are using case #2, then we must first check if all items are pre-ordered, and if so just change the order status.
     * Otherwise, if we are using case #3, we will have to remove the parent order.
     * @since version 1.0.7
     * This function also purge the shipping data
     */
    public function purgePreOrderedItems($orderId)
    {
        $order = wc_get_order($orderId);

        if (count($order->get_items()) == count($this->getPreorderedProducts($order)) && get_option('wc_preorders_mode') == 'partial') {
            /**
             * if there have no regular product into the order
             * then change the order status to preorder
             * @since version 1.0.8
             */
        
            $order->set_status('wc-pre-ordered');
            $order->save();

            return;
        }
             
        $items = $order->get_items();
        foreach ($items as $key => $product) {
            if (('yes' === get_post_meta($product->get_product_id(), '_is_pre_order', true) && strtotime(get_post_meta($product->get_product_id(), '_pre_order_date', true)) > time()) || 'yes' === get_post_meta($product->get_variation_id(), '_is_pre_order', true) && strtotime(get_post_meta($product->get_variation_id(), '_pre_order_date', true)) > time()) {
                $order->remove_item($product->get_id());
                $order->update_meta_data('_purge_preordered_items', 1);
            }
        }

        if (empty($order->get_items())) {
            $shippingItems = (array) $order->get_shipping_methods();
            if (sizeof($shippingItems) > 0) {
                // Loop through shipping items
                foreach ($shippingItems as $item_id => $item) {
                    $order->remove_item($item_id); // Remove Item
                }
                $order->calculate_totals();
            }
        } elseif (empty($order->get_items()) && 'individual' == get_option('wc_preorders_mode')) {
            $shippingItems = (array) $order->get_shipping_methods();
            if (sizeof($shippingItems) > 0) {
                // Loop through shipping items
                foreach ($shippingItems as $item_id => $item) {
                    $order->remove_item($item_id);
                }
                $order->calculate_totals();
            }
        } elseif ('partial' == get_option('wc_preorders_mode')) {
            $items = (array) $order->get_items('shipping');
            $numItems = count($items);
            
            $i = 0;
            if (sizeof($items) > 0) {
                // Loop through shipping items
                foreach ($items as $item_id => $item) {
                    if (++$i != $numItems) {
                        $order->remove_item($item_id);
                    }
                }
            }
            
            $order->calculate_totals();
        }

        /*
        * Calculate taxes and totals
        */
        $order->update_taxes();
        $order->calculate_totals();

    
        if ($order->get_payment_method() != 'bacs' || $order->get_payment_method() !='cod') {
            $order->payment_complete();
        }
           
        $order->save();
    }


    public function checkWholeOrders($order_id)
    {
        if (get_post_meta($order_id, '_pre_order_date', true)) {
            $order = wc_get_order($order_id);
            $order->set_status('wc-pre-ordered');
            $order->save();
        }
    }

    private function getPreorderedProducts($order)
    {
        $preorderedProducts = [];
        foreach ($order->get_items() as $item) {
            $productId = 0 !== $item->get_variation_id() ? $item->get_variation_id() : $item->get_product_id();
            $isPreOrder = get_post_meta($productId, '_pre_order_date', true);
            if ($isPreOrder && strtotime($isPreOrder) > time()) {
                $preorderedProducts[] = $item;
            }
        }

        return $preorderedProducts;
    }

    // Get the shipping total and split it into the amount of orders generated
    private function getSplitShippingFactor($order)
    {
        if ('yes' !== get_option('wc_preorders_multiply_shipping')) {
            return 1;
        }

        // If we are working on partial mode, then we will split it in 2 halves
        if ('partial' === $this->preordersMode) {
            return 2;
            // Otherwise we will have to split it by the amount of orders that we have
        }
        if ('individual' === $this->preordersMode) {
            return 1 + \count($this->getPreorderedProducts($order));
        }

        return 1;
    }

    private function getFilteredFields($prefix, $fields)
    {
        return $this->stripFieldsPrefix($prefix, $this->filterFields($prefix, $fields));
    }

    private function stripFieldsPrefix($prefix, $fields)
    {
        return array_combine(
            array_map(
                function ($k) use ($prefix) {
                    return str_replace($prefix, '', $k);
                },
                array_keys($fields)
            ),
            array_values($fields)
        );
    }

    private function filterFields($prefix, $fields)
    {
        return array_filter($fields, function ($key) use ($prefix) {
            return 0 === strpos($key, $prefix);
        }, ARRAY_FILTER_USE_KEY);
    }
}
