<?php
namespace Blue2\Scruffy\Controllers;

use Blue2\Scruffy\Site;
use Blue2\Scruffy\Order\Order;
use Blue2\Scruffy\Order\OrderItem;
use Clegginabox\PDFMerger\PDFMerger;
use Knp\Snappy\Pdf;
use mysql_xdevapi\Exception;

class OrderDashboardController
{
    private $pluginDirectoryPath;

    public static $defaultArgs = [
        'search' => null,
        'date_filter' => null,
        'status' => null,
        'date_from' => null,
        'date_to' => null,
    ];

    public function __construct()
    {
        $this->registerActions();
        $this->registerFilters();

        $this->pluginDirectoryPath = plugin_dir_path(__FILE__).'../';
    }

    private function registerActions()
    {
        add_action('woocommerce_order_data_store_cpt_get_orders_query', [$this, 'searchShippedDate'], 10, 3);

        add_action('rest_api_init', [$this, 'addDashboardAPIEndpoints']);
        add_action('wp_enqueue_scripts', [$this, 'enqueueAssets']);
        add_action('woocommerce_settings_tabs_settings_tab_walkers_dashboard', [$this, 'dashboardTabContents']);
        add_action('woocommerce_update_options_settings_tab_walkers_dashboard', [$this, 'updateDashboardSettings']);
        add_action('init', [ $this, 'addDashboardEndpoints' ]);
    }

    public function addDashboardAPIEndpoints()
    {
        $routes = [
            '/get-orders' => 'getOrdersAjax',
            '/get-batch' => 'getBatchAjax',
            '/(?P<id>\d+)/cancel-order' => 'cancelOrder',
            '/(?P<id>\d+)/receipt-plain' => 'createReceiptPlain',
            '/(?P<id>\d+)/receipt-letterhead' => 'createReceiptLetterhead',
            '/(?P<id>\d+)/gift-message' => 'createGiftMessage',
            '/(?P<id>\d+)/print-batch' => 'printBatch',
            '/(?P<id>\d+)/create-tracking' => 'createTracking',
            '/(?P<id>\d+)/add-tracking' => 'addTracking',
            '/(?P<id>\d+)/refund-order' => 'refundOrder',
            '/get-unprocessed-orders' =>'getUnprocessedOrdersAjax',
            '/process-batch' => 'createBatch'];

        foreach ($routes as $route => $callback) {
            register_rest_route('casa-candles/v1', $route, [
                    'methods' => 'GET, POST',
                    'callback' => [$this, $callback],
                ]);
        }
    }

    public function addDashboardEndpoints()
    {
        add_rewrite_rule('^order-dashboard/processing/?$', 'index.php?pagename=order-dashboard&od-page=processing', 'top');
        add_rewrite_rule('^order-dashboard/order/([0-9]+)/?$', 'index.php?pagename=order-dashboard&orderId=$matches[1]', 'top');
        add_rewrite_rule('^order-dashboard/batch/([0-9]+)/?$', 'index.php?pagename=order-dashboard&batchId=$matches[1]', 'top');
        flush_rewrite_rules();
    }

    public function addDashboardVars($vars)
    {
        $vars[] = 'od-page';
        $vars[] = 'batchId';
        $vars[] = 'orderId';

        return $vars;
    }

    private function registerFilters()
    {
        add_filter('woocommerce_settings_tabs_array', [$this, 'addDashboardTab'], 50);
        add_filter('query_vars', [ $this, 'addDashboardVars']);
        add_filter('wf_ups_filter_label_from_packages', [$this, 'multiPackageSupport'], 10, 3);
        add_filter('wf_ups_shipment_data', [$this, 'multiShippingSupport'], 10, 2);
        add_filter('ph_ups_address_customization', [$this, 'updatePackageAddress'], 10, 5);
    }

    public function addDashboardTab($settingsTabs)
    {
        $settingsTabs['settings_tab_walkers_dashboard'] = __('Order Dashboard', 'woocommerce-settings-walkers-order-dashboard');
        return $settingsTabs;
    }

    public function dashboardTabContents()
    {
        woocommerce_admin_fields(self::getDashboardTabFields());
    }

    public function updateDashboardSettings()
    {
        woocommerce_update_options(self::getDashboardTabFields());
    }

    public static function getDashboardTabFields()
    {
        $settings = array(
            'section_title' => array(
                'name'     => __('Batch Processing settings', 'walkers'),
                'type'     => 'title',
                'desc'     => '',
                'id'       => 'wc_settings_tab_order_section_title'
            ),
            'title' => array(
                'name' => __('Batch Size', 'walkers'),
                'type' => 'number',
                'desc' => __('The maximum number of orders in a batch', 'walkers'),
                'id'   => 'wc_settings_order_dashboard_batch_size'
            ),
            'section_end' => array(
                 'type' => 'sectionend',
                 'id' => 'wc_settings_tab_order_section_end'
            )
        );
        return apply_filters('wc_settings_tab_order_dashboard_settings', $settings);
    }

    public function createOrderDashboard()
    {
        ob_start();
        if (current_user_can('access_order_dashboard')) {
            get_template_part('/templates/order_dashboard/dashboard');
        } else {
            get_template_part('/templates/order_dashboard/login');
        }
        $content = ob_get_contents();
        ob_end_clean();
        return $content;
    }

    public function searchShippedDate($wp_query_args, $queryVars, $dataStore)
    {
        $filters = self::getFilters();

        if ($filters['date_filter'] == 'shipped') {
            $wp_query_args = array_merge($wp_query_args, ['meta_query' => $queryVars['meta_query']]);
        }

        return $wp_query_args;
    }

    public static function searchIdOrMeta($where = '')
    {
        $filters = self::getFilters();

        if ($filters['order_search'] && $filters['order_search'] !== '') {
            global $wpdb;

            $sanitised_search = $wpdb->esc_like($filters['order_search']);
            $search_term = " '%" . $sanitised_search . "%'";

            $where .= " AND ($wpdb->posts.ID = '{$sanitised_search}' OR
            (opm.meta_key = '_billing_address_index' AND opm.meta_value LIKE ".$search_term.")
            )";
        }

        return $where;
    }
    public static function joinPostMeta($join = '')
    {
        $filters = self::getFilters();

        if ($filters['order_search'] && $filters['order_search'] !== '') {
            global $wpdb;
            $join .= " LEFT JOIN $wpdb->postmeta opm ON $wpdb->posts.ID = opm.post_id ";
        }

        return $join;
    }

    public static function groupByPost($groupby)
    {
        global $wpdb;
        $groupby = "{$wpdb->posts}.ID";

        return $groupby;
    }

    public static function getFilters()
    {
        $from = filter_input(INPUT_GET, 'date_from', FILTER_SANITIZE_STRING);
        $to = filter_input(INPUT_GET, 'date_to', FILTER_SANITIZE_STRING);

        $filters = [
            'order_search' => filter_input(INPUT_GET, 'order_search', FILTER_SANITIZE_STRING),
            'date_filter' => filter_input(INPUT_GET, 'date_filter', FILTER_SANITIZE_STRING),
            'status' => filter_input(INPUT_GET, 'status', FILTER_SANITIZE_STRING),
            'date_from' => $from ? new \DateTime($from.' 00:00') : null,
            'date_to' => $to ? new \DateTime($to.' 23:59') : null,
        ];

        return $filters;
    }

    public static function showOrderDashboard()
    {

        if (get_query_var('batchId') != '') {
            get_template_part('/templates/order_dashboard/partials/batch');
        } elseif (get_query_var('orderId') != '') {
            get_template_part('/templates/order_dashboard/partials/single-order');
        } elseif (get_query_var('od-page') != '') {
            get_template_part('/templates/order_dashboard/partials/order-'.get_query_var('od-page'));
        } else {
            get_template_part('/templates/order_dashboard/partials/order-table');
        }
    }

    public function getUserRoles()
    {
        $roles = [];
        if (is_user_logged_in()) {
            $user = wp_get_current_user();
            $roles = $user->roles;
        }
        return $roles;
    }

    public static function getOrders($data)
    {
        $dateQuery = [];
        $metaQuery = $data['filters']['meta_query'] ?? [];

        $filters = $data['filters'] ?? [];
        $filters = array_merge(self::$defaultArgs, $filters);
        $order = $data['order'] ?? 'DESC';

        if ($filters['date_filter'] == 'ordered') {
            if (isset($filters['date_from'])) {
                $dateQuery['after'] = $filters['date_from']->format('Y-m-d H:i');
            }
            if (isset($filters['date_to'])) {
                $dateQuery['before'] = $filters['date_to']->format('Y-m-d H:i');
            }
        } else {
            $metaQuery['relation'] = 'AND';
            if (isset($filters['date_from'])) {
                $metaQuery[] = [
                    'key' => '_completed_date',
                    'compare' => '>=',
                    'type' => 'DATETIME',
                    'value' => $filters['date_from']->format('Y-m-d H:i')];
            }
            if (isset($filters['date_to'])) {
                $metaQuery[] = [
                    'key' => '_completed_date',
                    'compare' => '<=',
                    'type' => 'DATETIME',
                    'value' => $filters['date_to']->format('Y-m-d H:i')];
            }
            //_completed_date
        }

        if (isset($filters['batchId'])) {
            $metaQuery[] = [
                'key' => 'batch',
                'compare' => '=',
                'value' => $filters['batchId']];
        }

        switch ($filters['status']) {
            case 'unprocessed':
                $status = ['wc-pending'];
            break;
            case 'processing':
                $status = ['wc-processing'];
            break;
            case 'shipped':
                $status = ['wc-completed'];
            break;
            case 'cancelled':
                $status = ['wc-cancelled'];
            break;
            default: $status = ['wc-pending', 'wc-processing', 'wc-on-hold', 'wc-completed', 'wc-cancelled', 'wc-refunded', 'wc-failed'];
            break;
        }

        $args = [
                'post_type' => 'shop_order',
                'posts_per_page' => $data['limit'],
                'offset' => $data['page'],
                'orderby' => 'id',
                'order' => $order,
                'date_query' => [$dateQuery,'inclusive' => true],
                'meta_query' => $metaQuery,
                'post_status' => $status
            ];

        add_filter('posts_where', [__CLASS__, 'searchIdOrMeta']);
        add_filter('posts_join', [__CLASS__, 'joinPostMeta']);
        add_filter('posts_groupby', [__CLASS__, 'groupByPost']);
        $query = new \WP_Query($args);
        remove_filter('posts_where', [__CLASS__, 'searchIdOrMeta']);
        remove_filter('posts_join', [__CLASS__, 'joinPostMeta']);
        remove_filter('posts_groupby', [__CLASS__, 'groupByPost']);

        return (object) [
                'orders'        => $query->posts,
                'total'         => $query->found_posts,
                'max_num_pages' => $query->max_num_pages
            ];
    }

    public function cancelOrder($request)
    {
        $orderNumber = $request->get_params()['id'];
        if (current_user_can('access_order_dashboard')) {
            $order = wc_get_order($orderNumber);
            $order->update_status('cancelled', __('Order cancelled from dashboard.', 'woocommerce'));
            do_action('woocommerce_cancelled_order', $order->get_id());
            return rest_ensure_response(['success' => true]);
        }
        return rest_ensure_response(['success' => false]);
    }

    private function createPDF($contents, $gift = false, $filename = false)
    {
        if ($_SERVER['HTTP_HOST'] == 'casa-world.test' || $_SERVER['HTTP_HOST'] == 'walkers-us.test') {
            $snappy = new Pdf(get_template_directory() . '/vendor\wemersonjanuario\wkhtmltopdf-windows\bin\64bit\wkhtmltopdf.exe');
        } else {
            $snappy = new Pdf(get_template_directory() . '/vendor/h4cc/wkhtmltopdf-amd64/bin/wkhtmltopdf-amd64');
        }

        $snappy->setOption('margin-top', '0.5cm');
        $snappy->setOption('margin-left', '0.5cm');
        $snappy->setOption('margin-right', '0.5cm');
        $snappy->setOption('margin-bottom', '2.0cm');
        $snappy->setOption('page-size', 'A4');

        if ($filename) {
            $snappy->generateFromHtml($contents, $filename, [], true);
        } else {
            echo $snappy->getOutputFromHtml($contents);
        }
    }

    public function createReceiptPlain($request)
    {
        $orderId = $request->get_params()['id'];

        if (current_user_can('access_order_dashboard')) {
            ob_start();
            include(locate_template('/templates/order_dashboard/receipt-plain.php'));
            $contents = ob_get_clean();

            header('Content-type: application/pdf');
            $this->createPDF($contents);
        }
    }

    public function createReceiptLetterhead($request)
    {
        $orderId = $request->get_params()['id'];

        if (current_user_can('access_order_dashboard')) {
            ob_start();
            include(locate_template('/templates/order_dashboard/receipt-letterhead.php'));
            $contents = ob_get_clean();

            header('Content-type: application/pdf');
            $this->createPDF($contents);
        }
    }
    public function createGiftMessage($request)
    {
        $orderId = $request->get_params()['id'];

        if (current_user_can('access_order_dashboard')) {
            ob_start();
            include(locate_template('/templates/order_dashboard/gift-message.php'));
            $contents = ob_get_clean();

            header('Content-type: application/pdf');
            $this->createPDF($contents, true);
        }
    }

    public function printBatch($request)
    {
        if (current_user_can('access_order_dashboard')) {
            $batchId = $request->get_params()['id'];
            $orders = get_field('orders', $batchId);
            $brands = [];
            $pickingList2 = [];

            if ($orders) {
                foreach ($orders as $order) {
                    //$orderId = $order->ID;
                    $orderId = $order;
                    $wooOrder = wc_get_order($orderId);
                    $order = new Order($wooOrder);

                    foreach ($order->getOrderItems() as $wooOrderItem) {
                        $orderItem = new OrderItem($wooOrderItem, '', $orderId);
                        $brand = $orderItem->getBrand();

                        $identifier = $wooOrderItem->get_variation_id() > 0 ? $wooOrderItem->get_variation_id() : $wooOrderItem->get_product_id();

                        if (!in_array($brand->name, $brands)) {
                            $brands[$brand->slug] = $brand->name;
                        }

                        if (!isset($pickingList2[$brand->slug])) {
                            $pickingList2[$brand->slug] = [];
                        }
                        if (!isset($pickingList2[$brand->slug][$identifier])) {
                            $type = $wooOrderItem->get_variation_id() ? array_pop(wc_get_product($wooOrderItem->get_variation_id())->get_attributes()) : 'single';
                            $pickingList2[$brand->slug][$identifier] = ['sku' => $orderItem->getSKU(),
                                'product' => $orderItem->getOrderName(), 'type' => $type, 'quantity' => 0,
                                'brand' => $brand->slug];
                        }

                        $pickingList2[$brand->slug][$identifier]['quantity'] += $orderItem->getQuantity();
                    }
                }

                //Render pick list
                $pdfMerger = new PDFMerger;
                ob_start();

                include(locate_template('/templates/order_dashboard/pick-list.php'));
                $contents = ob_get_clean();
                $picklistFilename = 'picking.pdf';

                $this->createPDF($contents, false, $picklistFilename);
                $pdfMerger->addPDF($picklistFilename, 'all');

                header('Content-type: application/pdf');
                echo $pdfMerger->merge('string', 'test.pdf');

                //Remove created docs
                unlink($picklistFilename);
            }
        }
        //return rest_ensure_response(['success' => false]);
    }
    public function printGiftMessages($request)
    {
        if (current_user_can('access_order_dashboard')) {
            $batchId = $request->get_params()['id'];
            $batch = get_post($batchId);

            $orders = get_field('orders', $batchId);

            $pickingList = [];
            $foundGift = false;

            ob_start();
            if ($orders) {
                foreach ($orders as $order) {
                    $orderId = $order->ID;
                    $wooOrder = wc_get_order($orderId);
                    $order = new Order($wooOrder);

                    if (get_post_meta($orderId, 'is_gift', true)) {
                        $foundGift = true;
                        include(locate_template('/templates/order_dashboard/gift-message.php'));
                    }
                }

                $contents = ob_get_clean();
                //Show pick list

                if ($foundGift) {
                    header('Content-type: application/pdf');
                    $this->createPDF($contents, true);
                } else {
                    header('Location: /order-dashboard/batch/'.$batchId);
                    die;
                }
            }
        }
        //return rest_ensure_response(['success' => false]);
    }

    public function refundOrder($request)
    {
        $orderNumber = $request->get_params()['id'];
        $refundTotal = number_format($request->get_params()['refund_total'], 2);

        if (current_user_can('access_order_dashboard')) {
            $order = wc_get_order($orderNumber);

            $refund = wc_create_refund(['amount' => $refundTotal,'order_id' => $order->get_id()]);
            if (is_wp_error($refund)) {
                return rest_ensure_response(['success' => false]);
            }

            $order->add_order_note(__('Refund issued for '.$refundTotal));
            return rest_ensure_response(['success' => true]);
        }
        return rest_ensure_response(['success' => false]);
    }

    public function createTracking($request)
    {
        $orderNumber = $request->get_params()['id'];

        if (current_user_can('access_order_dashboard')) {
            if (Site::isUSSite()) {
                add_action('wf_after_package_generation', 'wf_auto_genarate_label_fedex', 2, 2);
                wf_automatic_package_and_label_generation_fedex($orderNumber);
                remove_action('wf_after_package_generation', 'wf_auto_genarate_label_fedex', 2);
            } else {
                add_action('wf_after_package_generation', 'wf_auto_genarate_label_ups', 2, 2);
                wf_automatic_package_and_label_generation_ups($orderNumber);
                remove_action('wf_after_package_generation', 'wf_auto_genarate_label_ups', 2);
            }

            //set order completed
            $order = wc_get_order($orderNumber);
            $order->update_status('completed');
        }
        header('Location: /order-dashboard/order/'.$orderNumber);
        die;
    }

    public function addTracking($request)
    {
        $orderNumber = $request->get_params()['id'];
        $trackingNumber = $request->get_params()['tracking'];

        if (current_user_can('access_order_dashboard')) {
            $key = 'ups_shipment_ids';
            if (Site::isUSSite()) {
                $key = 'wf_woo_fedex_shipmentId';
            }
            update_post_meta($orderNumber, $key, $trackingNumber);

            //set order completed
            $order = wc_get_order($orderNumber);
            $order->update_status('completed');
        }
        header('Location: /order-dashboard/order/'.$orderNumber);
        die;
    }

    public function getOrdersAjax()
    {
        if (current_user_can('access_order_dashboard')) {
            $limit = 25;
            $start = $_GET['start'] ?? 0;
            $page = $start +1;
            $filters = self::getFilters();

            $orders = $this->getOrders(['limit' => $limit,'page' => $start, 'filters' => $filters]);
            $orderData = $this->getOrderDataAsJson($orders->orders);

            return rest_ensure_response(['data' => $orderData, 'pos'=> $start, 'total_count' => $orders->total]);
        }
        return rest_ensure_response([]);
    }

    public function getUnprocessedOrders($page = 0)
    {
        $limit = get_option('wc_settings_order_dashboard_batch_size', 50);
        $filters = self::$defaultArgs;

        $filters['status'] = 'processing';
        $filters['meta_query'] = [
                ['relation'=> 'OR',[
                'key'     => 'batch',
                'compare' => 'NOT EXISTS',
                ],[
                    'key'     => 'batch',
                    'compare' => '=',
                    'value' => ''
                    ]]
                ];

        return $this->getOrders(['limit' => $limit, 'page' => $page, 'filters' => $filters, 'order' => 'ASC']);
    }

    public function getUnprocessedOrdersAjax()
    {
        if (current_user_can('access_order_dashboard')) {
            $start = $_GET['start'] ?? 0;
            $page = $start +1;

            $orders = $this->getUnprocessedOrders($start);
            $orderData = $this->getOrderDataAsJson($orders->orders);

            return rest_ensure_response(['data' => $orderData, 'pos'=> $start, 'total_count' => count($orderData), 'all_total' => $orders->total]);
        }
        return rest_ensure_response([]);
    }

    public function getOrderDataAsJson($orders)
    {
        $jsonOrders = [];
        foreach ($orders as $singleOrder) {
            $order = new Order(wc_get_order($singleOrder));
            $jsonOrders[] = $order->newOrderData();
        }

        return $jsonOrders;
    }

    public function enqueueAssets()
    {
        if (is_page('order-dashboard')) {
            wp_enqueue_script('webix-js', '//cdn.webix.com/edge/webix.js', [], '1', true);
            wp_enqueue_style('webix-css', '//cdn.webix.com/edge/webix.css', []);

            wp_enqueue_script('order-dashboard-js', get_template_directory_uri() . '/assets/scripts/order-dashboard.js', [], '1', true);
            wp_localize_script('order-dashboard-js', 'wpApiSettings', array(
                'root' => esc_url_raw(rest_url()),
                'nonce' => wp_create_nonce('wp_rest')
            ));

            if (get_query_var('batchId') != '') {
                wp_enqueue_script('order-dashboard-batch', get_template_directory_uri() . '/assets/scripts/batch.js', [], '1', true);
                wp_localize_script('order-dashboard-batch', 'wpApiSettings', array(
                    'root' => esc_url_raw(rest_url()),
                    'nonce' => wp_create_nonce('wp_rest')
                ));
            } elseif (get_query_var('od-page') != '') {
                wp_enqueue_script('order-dashboard-processing', get_template_directory_uri() . '/assets/scripts/processing.js', [], '1', true);
                wp_localize_script('order-dashboard-processing', 'wpApiSettings', array(
                    'root' => esc_url_raw(rest_url()),
                    'nonce' => wp_create_nonce('wp_rest')
                ));
            } else {
                wp_enqueue_script('order-dashboard-table', get_template_directory_uri() . '/assets/scripts/webix-scripts.js', [], '1', true);
                wp_localize_script('order-dashboard-table', 'wpApiSettings', array(
                    'root' => esc_url_raw(rest_url()),
                    'nonce' => wp_create_nonce('wp_rest')
                ));
            }
        }
    }

    public static function getPreviousBatches($totalOnly = false)
    {
        $batchArgs = [
                'post_type' => 'batch',
                'post_status' => 'any',
                'posts_per_page' => -1,
            ];
        $batches = new \WP_Query($batchArgs);
        return $totalOnly ? $batches->found_posts : $batches;
    }

    public function createBatch()
    {
        if (current_user_can('access_order_dashboard')) {
            $batches = self::getPreviousBatches(true);
            $orders = $this->getUnprocessedOrders();
            $nextBatch = $batches +1;

            $batchId = wp_insert_post(['post_type' => 'batch', 'post_title' => 'Batch '.$nextBatch]);
            update_field('batch_id', $nextBatch, $batchId);

            $batchOrders = [];

            foreach ($orders->orders as $order) {
                update_field('batch', $batchId, $order->ID);
                $batchOrders[] = $order->ID;
            }

            update_field('orders', $batchOrders, $batchId);

            header('Location: /order-dashboard/batch/'.$batchId);
            die;

            return rest_ensure_response(['success' => true]);
        }
        return rest_ensure_response(['success' => false]);
    }

    public function getBatchAjax()
    {
        if (current_user_can('access_order_dashboard')) {
            $limit = 250;
            $start = 0;
            $page = $start +1;
            $filters = self::getFilters();

            $batchId = $_GET['batchId'];

            $filters = array_merge($filters, ['batchId' => $batchId]);

            $orders = $this->getOrders(['limit' => $limit,'page' => $start, 'filters' => $filters]);
            $orderData = $this->getOrderDataAsJson($orders->orders);

            return rest_ensure_response(['data' => $orderData, 'pos'=> $start, 'total_count' => $orders->total]);
        }
        return rest_ensure_response([]);
    }

    public static function getShippingAddresses($order)
    {
        global $wcmca_order_model, $wcmca_option_model;
        $shippingAddresses = [];

        if ($wcmca_option_model->shipping_per_product()) {
            foreach ($order->get_items() as $orderItem) {
                $shippingAddress = $wcmca_order_model->get_formatted_item_shipping_address($orderItem);
                $hashString = strtolower($orderItem->get_meta('_wcmca_billing_address_1').$orderItem->get_meta('_wcmca_billing_postcode'));
                $hashString = preg_replace("/[^A-Za-z0-9 ]/", '', $hashString);
                $addressHash = wp_hash($hashString);
                $shippingAddresses[$addressHash] = $shippingAddress;
            }
        }

        return $shippingAddresses;
    }

    public static function getShippingPackages($order)
    {
        global $wcmca_order_model, $wcmca_option_model;
        $shippingPackages = [];

        if ($wcmca_option_model->shipping_per_product()) {
            foreach ($order->get_items() as $orderItem) {
                $shippingAddress = $wcmca_order_model->get_formatted_item_shipping_address($orderItem, false);
                $hashString = strtolower($orderItem->get_meta('_wcmca_billing_address_1').$orderItem->get_meta('_wcmca_billing_postcode'));
                $hashString = preg_replace("/[^A-Za-z0-9 ]/", '', $hashString);
                $addressHash = wp_hash($hashString);
                if (!isset($shippingPackages[$addressHash])) {
                    $shippingPackages[$addressHash] = ['address' => $shippingAddress, 'items' => [$orderItem]];
                } else {
                    $shippingPackages[$addressHash]['items'][] = $orderItem;
                }
            }
        } else {
            if ($order->get_formatted_shipping_address()) {
                $address =  $order->get_formatted_shipping_address();
            } else {
                $address =  $order->get_formatted_billing_address();
            }
            $shippingPackages [['address' => $address, 'items' => $order->get_items()]];
        }
        return $shippingPackages;
    }

    public function multiPackageSupport($packages, $shippingAddress, $orderId)
    {
        $order = new \WC_Order($orderId);
        $shippingPackages = self::getShippingPackages($order);

        if (count($shippingPackages) > 0) {
            $newPackages = [];

            foreach ($shippingPackages as $shippingPackage) {
                $items = [];
                foreach ($shippingPackage['items'] as $orderItem) {
                    $address = $this->getItemShippingAddress($orderItem);
                    $productId = $orderItem->get_product_id();
                    $productVariationId = $orderItem->get_variation_id();
                    $productId = $productVariationId > 0 ? $productVariationId : $productId;
                    $product = wc_get_product($productId);

                    if (!isset($items[$productId])) {
                        $items[$productId] = ['data' => $product, 'quantity' => 1];
                    } else {
                        $items[$productId]['quantity']++;
                    }
                }

                $newPackages[] = [
                    'contents' => $items,
                    'destination' => $address
                ];
            }

            $packages = $newPackages;
        }
        return $packages;
    }

    public function getItemShippingAddress($item)
    {
        $type = 'shipping';
        $value = "";
        $address = array();
        $notes_field = "";
        if (is_object($item) && get_class($item) == 'WC_Order_Item_Meta') {
            $meta_data = $item->meta;
        } else {
            $meta_data = version_compare(WC_VERSION, '2.7', '<') ? $item["item_meta"]  : $item->get_meta_data();
        }

        foreach ($meta_data as $key => $single_meta) {
            if (!is_object($single_meta)) {
                $type = $key == '_wcmca_type' ? $single_meta[0] : $type;
                if (strpos($key, '_wcmca_shipping_') !== false) {
                    $address[str_replace('_wcmca_shipping_', "", $key)] = $single_meta[0];
                }
                if (strpos($key, '_wcmca_billing_') !== false) {
                    $address[str_replace('_wcmca_billing_', "", $key)] = $single_meta[0];
                }
            } else { // > 3.0
                $type = $single_meta->key == '_wcmca_type' ? $single_meta->value : $type;
                if (strpos($single_meta->key, '_wcmca_shipping_') !== false) {
                    $address[str_replace('_wcmca_shipping_', "", $single_meta->key)] = $single_meta->value;
                }
                if (strpos($single_meta->key, '_wcmca_billing_') !== false) {
                    $address[str_replace('_wcmca_billing_', "", $single_meta->key)] = $single_meta->value;
                }
            }
        }

        $address['address'] = $address['address_1'];
        return $address;
    }

    public function multiShippingSupport($shipments, $order)
    {
        $packages = $this->multiPackageSupport(null, null, $order->get_id());
        if (count($packages) > 0) {
            $shipments = [];
            $ups_admin_class 	= new \WF_Shipping_UPS_Admin();
            $shipping_service_data	= $ups_admin_class->wf_get_shipping_service_data($order);
            $default_service_type 	= $shipping_service_data['shipping_service'];

            $ship_packages = get_post_meta($order->get_id(), '_wf_ups_stored_packages', true);

            foreach ($ship_packages as $key => $ship_package) {
                $shipments[]	=	array(
                                    'shipping_service'	=>	$default_service_type,
                                    'packages'			=>	[$ship_package],
                                    'address' => $packages[$key]['destination']
                                );
            }
        }
        return $shipments;
    }

    public function updatePackageAddress($to_address, $shipment, $ship_from_address, $orderId, $recipient)
    {
        if (isset($shipment['address'])) {
            $originalAddress = $to_address;
            $to_address = $shipment['address'];
            $to_address['name'] = $to_address['first_name'].' '.$to_address['last_name'];
            $to_address['email'] = $to_address['email'] ?? $originalAddress['email'];
            $to_address['company'] = $to_address['company'] ?? $originalAddress['company'];
            $to_address['phone'] = $to_address['phone'] ?? $originalAddress['phone'];
            $to_address['address_2'] = $to_address['address_2'] ?? $originalAddress['address_2'];
        }

        return $to_address;
    }

    public static function getShippingCosts($postId)
    {
        global $wp;
        $costResponse = get_post_meta($postId, 'wf_ups_generate_packages_rates_response', true);
        $storedPackages = get_post_meta($postId, '_wf_ups_stored_packages', true);
        $attemptedCalc = get_post_meta($postId, '_walkers_attempted_calc', true);

        if ($costResponse == '' && $storedPackages != '') {
            $weight = 0;
            $length = 0;
            $width = 0;
            $height = 0;

            foreach ($storedPackages as $storedPackage) {
                $weight += $storedPackage['Package']['PackageWeight']['Weight'];
                $length += $storedPackage['Package']['Dimensions']['Length'];
                $width += $storedPackage['Package']['Dimensions']['Width'];
                $height += $storedPackage['Package']['Dimensions']['Height'];
            }

            if ($weight > 0 && $length > 0 && $width > 0 && $height > 0 && $attemptedCalc == '') {
                update_post_meta($postId, '_walkers_attempted_calc', true);
                $_GET['wf_ups_generate_packages_rates'] = base64_encode($postId);
                $_GET['insurance'] = '';
                $_GET['weight'] = $weight;
                $_GET['length'] = $length;
                $_GET['width'] = $width;
                $_GET['height'] = $height;

                $url = home_url(add_query_arg(array(), $wp->request));

                $location = add_filter('wp_redirect', function () use ($url) {
                    return $url;
                });
                define('WP_ADMIN', true);
                $ups_admin_class = new \WF_Shipping_UPS_Admin();
                $ups_admin_class->wf_ups_generate_packages_rates();
            } else {
                return 'N/A';
            }
        } elseif (is_array($costResponse)) {
            $firstCost = array_pop($costResponse);
            return $firstCost['cost'];
        }
        return 'N/A';
    }

    public static function isSecure()
    {
        $isSecure = false;
        if (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] == 'on') {
            $isSecure = true;
        } elseif (!empty($_SERVER['HTTP_X_FORWARDED_PROTO']) && $_SERVER['HTTP_X_FORWARDED_PROTO'] == 'https' || !empty($_SERVER['HTTP_X_FORWARDED_SSL']) && $_SERVER['HTTP_X_FORWARDED_SSL'] == 'on') {
            $isSecure = true;
        }
        return $isSecure;
    }
}
