import React, { useState } from 'react';
import { connect } from 'react-redux';
import { addTicketToCart, removeTicketFromCart } from '../../redux/actions';
import { getTotalSeatsFromCart } from '../../redux/selectors';
import Loader from 'react-loader-spinner';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';

const TicketList = ({
    tickets,
    totalPassengers,
    inCart,
    addTicketToCart,
    removeTicketFromCart
}) => {
    const [opened, setOpened] = useState(false);
    const openClass = opened ? 'open' : '';

    return (
        <div className="ticket-list">
            <div className="ticket-input" onClick={() => setOpened(!opened)}>
                {totalPassengers} Passengers
                <button onClick={() => setOpened(!opened)}>
                    <FontAwesomeIcon
                        icon={['far', 'chevron-down']}
                        rotation={opened ? 180 : null}
                    />
                </button>
            </div>
            <div className={`ticket-window ${openClass}`}>
                <div className="container">
                    <h2>Passengers</h2>
                    {tickets.length === 0 ? (
                        <Loader type="Oval" width={50} height={50} color="#088584" />
                    ) : (
                        tickets.map(ticket => {
                            const { id, title } = ticket;
                            const count = inCart[id] ? inCart[id].quantity : 0;
                            return (
                                <div className="ticket" key={id}>
                                    <p className="title">{title}</p>
                                    <button onClick={() => removeTicketFromCart(ticket)}>
                                        <FontAwesomeIcon icon={['fas', 'minus-circle']} />
                                    </button>
                                    <span>{count}</span>
                                    <button onClick={() => addTicketToCart(ticket)}>
                                        <FontAwesomeIcon icon={['fas', 'plus-circle']} />
                                    </button>
                                </div>
                            );
                        })
                    )}

                    <div className="info">
                        <b>Important information</b>
                        <ol>
                            <li>
                                A family ticket covers two adults with two children or one adult
                                with three children
                            </li>
                            <li>
                                For Wheelchair users please call 01382 221609 to make your
                                booking. 48 hours advance notice is required before date of travel.
                            </li>
                            <li>
                                For Groups of 7 or more, please visit the Travel Centre in
                                Commercial Street or call 01382 221609
                            </li>
                        </ol>
                    </div>
                    <button className="primary" onClick={() => setOpened(false)}>
                        Done
                    </button>
                </div>
            </div>
        </div>
    );
};

const mapStateToProps = state => {
    return {
        tickets: state.tickets.allTickets,
        inCart: state.cart.inCart,
        totalPassengers: getTotalSeatsFromCart(state)
    };
};

export default connect(
    mapStateToProps,
    { addTicketToCart, removeTicketFromCart }
)(TicketList);
