<?php

namespace XploreBooking\Controller;

class BookingController
{
    const SEATS = 29;

    public function __construct()
    {
        $this->registerActions();
    }

    private function registerActions()
    {
        add_action('rest_api_init', function () {
            register_rest_route('xplore-booking/v1', '/add-new-booking/', [
                'methods' => 'POST',
                'callback' => [$this, 'addNewBooking'],
            ]);

            register_rest_route('xplore-booking/v1', '/set-booking-notes/', [
                'methods' => 'POST',
                'callback' => [$this, 'addNotes'],
            ]);

            register_rest_route('xplore-booking/v1', '/update-payment-status/', [
                'methods' => 'POST',
                'callback' => [$this, 'updatePaymentStatus'],
            ]);

            register_rest_route('xplore-booking/v1', '/get-confirmation-text/', [
                'methods' => 'GET',
                'callback' => [$this, 'getConfirmationText'],
            ]);

            register_rest_route('xplore-booking/v1', '/cancel-unconfirmed-bookings/', [
                'methods' => 'GET',
                'callback' => [$this, 'cancelUnconfirmedBookings'],
            ]);

            register_rest_route('xplore-booking/v1', '/validate-recaptcha/', [
                'methods' => 'POST',
                'callback' => [$this, 'validateRecaptcha'],
            ]);
        });

        add_action('template_redirect', [$this, 'confirmationRedirect']);

        add_filter('acf/update_value/name=outbound_journey', [$this, 'outboundJourneyUpdated'], 10, 3);

        add_action('save_post', [$this, 'bookingAmended'], 10, 3);

        add_action('admin_notices', [$this, 'adminNotices']);

        add_action('admin_notices', [$this, 'notAmended']);

        add_action('pre_get_posts', [$this, 'postFilter']);

        add_filter('rest_pre_dispatch', [$this, 'rest_basic_auth_check_errors'], 10, 3);
    }

    public static function cancelUnconfirmedBookings()
    {
        $args = [
            'post_type' => 'booking',
            'posts_per_page' => -1,
            'meta_query' => [
                [
                    'key' => 'payment_reference',
                    'value' => '',
                    'compare' => '=',
                ],

            ],
        ];

        $pendingBookings = new \WP_Query($args);
        if ($pendingBookings->have_posts()) {
            while ($pendingBookings->have_posts()) {
                $pendingBookings->the_post();
                $now = strtotime('-2 days');
                $date = strtotime(get_the_date());
                if ($date < $now) {
                    $tickets = get_field('tickets');
                    $totalSeats = 0;
                    foreach ($tickets as $ticket) {
                        $realTicket = get_page_by_title($ticket['ticket_type'], null, 'ticket');
                        $seatsPerTicket = get_field('seats', $realTicket->ID) * $ticket['quantity'];
                        $totalSeats += $seatsPerTicket;
                    }
                    $outboundJourney = get_field('outbound_journey');
                    $inboundJourney = get_field('inbound_journey');

                    $outboundSeats = get_field('seats', $outboundJourney) + $totalSeats;
                    $inboundSeats = get_field('seats', $inboundJourney) + $totalSeats;

                    update_field('seats', $outboundSeats, $outboundJourney);
                    update_field('seats', $inboundSeats, $inboundJourney);

                    wp_delete_post(get_the_ID());
                }
            }
        }

        wp_reset_query();
    }

    public function rest_basic_auth_check_errors($result, $server, $request)
    {
        $nonce = $request->get_header('X-WP-Nonce');
        if (!$nonce) {
            return new \WP_REST_Response('Invalid request', 400);
        }
    }

    public function validateRecaptcha($request)
    {
        $response = isset($request['recaptchaToken']) ? $request['recaptchaToken'] : null;
        $args = [
            'secret' => '6LdRLWQcAAAAAJMUDzSfT8dyCrQ3u7Os34kCvkis',
            'response' => $response,
        ];

        //open connection
        $ch = curl_init();

        //set the url, number of POST vars, POST data
        curl_setopt($ch, CURLOPT_URL, 'https://www.google.com/recaptcha/api/siteverify');
        curl_setopt($ch, CURLOPT_POST, count($args));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HEADER, false);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($args));

        //execute post
        $result = curl_exec($ch);

        //close connection
        curl_close($ch);

        $result = json_decode($result, true);

        $valid = $result['success'];

        return new \WP_REST_Response($result);
    }

    public function confirmationRedirect()
    {
        if (is_404() && isset($_POST['name']) && isset($_POST['transStatus'])) {
            self::updatePaymentStatus();
            $bookingId = $_POST['cartId'];
            status_header(200);
            echo '
            <!DOCTYPE html>
            <html>
            <head>
                <meta name="viewport" content="width=device-width, initial-scale=1">
                <link rel="profile" href="http://gmpg.org/xfn/11">
                <META HTTP-EQUIV="Refresh" CONTENT="3; URL=http://booking.xploremore.co.uk/booking/?complete=' . $bookingId . '">
            </head>
            <WPDISPLAY FILE="header.html">
            <body>
                <p>Redirecting to site</p>
            </body>
            <WPDISPLAY FILE="footer.html">
            </html>';
            exit;
        }
    }

    public static function updatePaymentStatus()
    {
        $bookingId = $_POST['cartId'];
        $amount = $_POST['amount'];

        $realAmount = get_field('price', $bookingId);
        $paymentReference = $_POST['transId'];

        $status = $_POST['transStatus'] == 'Y' ? 'paid' : 'failed';

        switch ($status) {
            case 'paid':
                if ($amount != $realAmount) {
                    die;
                }
                update_field('payment_reference', $paymentReference, $bookingId);
                update_field('booking_status', 'Completed', $bookingId);
                wp_update_post(['ID' => $bookingId]);
                self::emailUserBookingDetails($status, $bookingId);

                //Check if there has been a seating overlap
                $seats = get_field('total_seats', $bookingId);

                $outboundJourney = get_field('outbound_journey', $bookingId);
                $journeySeats = get_field('seats', $outboundJourney[0]);
                if ($journeySeats - $seats >= 0) {
                    update_field('seats', $journeySeats - $seats, $outboundJourney[0]);
                } else {
                    self::emailOverBookingDetails($bookingId);
                }

                //Check if there has been a seating overlap
                $inboundJourney = get_field('inbound_journey', $bookingId);
                if ($inboundJourney) {
                    $journeySeats = get_field('seats', $inboundJourney[0]);
                    if ($journeySeats - $seats >= 0) {
                        update_field('seats', $journeySeats - $seats, $inboundJourney[0]);
                    } else {
                        self::emailOverBookingDetails($bookingId);
                    }
                }

                break;
            case 'failed':
                update_field('payment_reference', $paymentReference, $bookingId);
                update_field('booking_status', 'Failed', $bookingId);
                self::emailUserBookingDetails($status, $bookingId);
                break;
            default:
                update_field('booking_status', 'Pending', $bookingId);
                wp_update_post(['ID' => $bookingId]);
                break;
        }
    }

    public static function emailUserBookingDetails($status, $bookingId)
    {
        $to = get_field('email', $bookingId);
        $name = get_field('name', $bookingId);
        $surname = get_field('surname', $bookingId);
        $email = get_field('email', $bookingId);
        $paymentReference = get_field('payment_reference', $bookingId);

        switch ($status) {
            case 'paid':
                $emailText = 'successful booking';
                // no break
            case 'bookingAmended':
                $emailText = 'booking confirmed';
        }

        switch ($status) {
            case 'paid':
            case 'bookingAmended':
                $subject = 'Edinburgh Airport Xpress - ' . $emailText;
                $tickets = get_field('tickets', $bookingId);
                $outboundJourneyId = get_field('outbound_journey', $bookingId);
                $outboundJourney = get_the_title($outboundJourneyId[0]);
                $outboundReference = get_field('outbound_reference', $bookingId);

                $inboundJourneyId = get_field('inbound_journey', $bookingId);
                $inboundJourney = get_the_title($inboundJourneyId[0]);
                $inboundReference = get_field('inbound_reference', $bookingId);
                $addOns = get_field('add_ons', $paymentReference);

                $body = '<img src="' . get_field('logo', 'options') . '" style="max-width: 200px; height: auto; margin-bottom: 15px;"> <br>';
                $body .= '<p>Dear ' . $name . ' ' . $surname . '</p>';
                $body .= '<p>Thank you for booking the Edinburgh Airport Express with Xplore More </p>';
                $body .= '<p><strong>This confirmation is your coach ticket and must be shown to the driver when boarding the vehicle. All members of your party must be present at the time of boarding.</strong></p><br>';
                $body .= '<p><strong>Please ensure that you arrive at the coach stance at least 10 minutes prior to the time of departure as after this time your seat may be released to other customers</strong></p>';
                $body .= '<p><strong>Booking Name: </strong>' . $name . ' ' . $surname . '.</p>';
                $body .= '<p><strong>Email Address: </strong>' . $email . '</p>';
                $body .= '<p><strong>Payment Reference:</strong> ' . $paymentReference . '</p>';
                $body .= '<p style="font-weight: 600; font-size:20px;">Tickets:</p>';

                foreach ($tickets as $ticket) {
                    $body .= '<p style="font-weight: 600; font-size:20px;">' . $ticket['ticket_type'] . ', Quantity: ' . $ticket['quantity'] . '</p><br>';
                }

                $extras = explode(',', get_field('add_ons', $bookingId));

                if ($extras[0] !== 'no') {
                    $body .= '<p style="font-weight: 600;">Extra luggage: ' . $extras[1] . '</p>';
                }
                if ($extras[2] !== 'no') {
                    $body .= '<p style="font-weight: 600;">Oversized baggage details: ' . $extras[3] . '</p>';
                }

                $body .= self::getFormattedJourneyText($outboundJourney, $outboundReference);
                if ($inboundJourney) {
                    $body .= self::getFormattedJourneyText($inboundJourney, $inboundReference);
                }
                $body .= "<p>
                    <strong>We hope you have an enjoyable journey to/from Edinburgh Airport</strong>
                </p>";
                $body .= "<p>Please check your entire booking carefully and if you have any queries or questions regarding any of these details please call 01382 221609/07557753057 as soon as possible.</p>";
                $body .= "<p>Phone lines are open Monday to Friday 10.00  – 5.00 and Saturday 10.00 – 2.00.</p>";
                $body .= "<p><strong>Scottish Concessionary Card Holders and Students</strong> <br/>
                    <strong>All Scottish Concessionary Card Holders</strong> must ensure that they present their cards to the driver on boarding the Coach, where they will be scanned on the ticket machine and the relevant concessionary ticket/s issued.   Please Note:  If Concessionary Card/s are not available for presentation to the driver at the timeof boarding, the relevant standard return fare for the journey being made will be payable. No refunds will be available on production of concession cards at a later date.</p>";
                $body .= "<p><strong>Students</strong>
                All students must show a current and valid matriculation card (or letter of acceptance) to the driver prior to boarding the coach.   If you do not have a Student Card/Letter of Acceptance your booking will be invalid and you will be required to pay the relevant standard fare for your journey.</p>";
                $body .= "<p>
                            <strong>Baggage Allowance</strong> <br />
                            Individuals are entitled to a baggage allowance which is to be stored in the hold of the Coach. The baggage allowance is either:
                            <ul>
                              <li>
                              2 medium sized suitcases or rucksacks each measuring no more than 70cmsx45cmsx30cms and weighing in total no more than 23 kgs, or
                              </li>
                              <li>
                              1 large suitcase or rucksack measuring no more than 75cmsx50cmsx32cms and weighing no more than 23kg
                              </li>

                            </ul>
                          </p>
                          <p>
                          In addition, one small piece of luggage can be taken on board the Coach measuring no more than 45cmsx35cmsx20cms.  With the exception of briefcases, no hard shelled cases, wheeled bags or hard framed rucksacks/backpacks are permitted on board for safety reasons and mustbe stowed in the luggage hold.                          </p>

                          <p>
                          Individual luggage allowances may be ‘pooled’ if they are travelling in a group or as a family but they must travel on the same ticket.                        </p>";
                $body .= "<p>
                            <strong>Excess or Outsized luggage</strong> <br />
                            Additional pieces of hold luggage or ‘outsized’ items (such as Golf bags, large musical instruments, ski’s etc) may be carried subject always to space availability.  A charge of £10 per item is payable to the driver.
                        </p>
                        <p>
                        A maximum of three chargeable items of luggage per person may be taken in the hold .    If insufficient space is available for excess luggage or ‘outsized’ items, the next available servicemust be taken. 
                        </p>
                        <p>
                        Luggage is carried at the owners risk.   The company’s liability for loss or damage to luggage is limited.
                        </p>";
                $body .= "<p>
                            <strong>Booking Queries/Service Information</strong> <br />
                            Booking queries:
                            <ul>
                              <li>
                              Telephone: 01382 221609 <i>Monday to Friday 1000 – 1700   Saturday 1000 – 1400</i>
                              </li>
                            </ul>
                          </p>";
                $body .= "<p>For information relating to service delivery outwith the above hours:</p>";
                $body .= "<p>
                            <strong>Terms & Conditions</strong> <br />
                            Terms and Conditions can be found on the xploremore.co.uk website or by contacting our Travel Centre in Commercial Street on 01382 221609.
                        </p>";
                $body .= '<p><strong>Lost or Stolen Tickets</strong> - cannot be replaced, however, if a replacement ticket has been purchased and the original ticket is subsequently found,  forward both the original and replacement tickets to our Customer Services where a refund will be made in accordance with our refund policy.

                        </p>';
                $body .= '<p><strong>Refunds</strong> can only be made by calling or visiting our Travel Centre.   Tickets cancelled 24 hours prior to time of departure are subject to a cancellation fee.

                        </p>';
                break;
            case 'failed':
                $subject = '<p>Unsuccessful Booking for ' . $paymentReference . '</p>';
                $body = '<p>Unfortunately payment was not succesful for reference: ' . $paymentReference . '</p>';
                $body .= '<p>Please try again or contact <a href="tel:01382 340006">01382 340006</a> or <a href="mailto:XploreMore@nationalexpress.com">XploreMore@nationalexpress.com</a></p>';
                break;
        }

        $headers = ['Content-Type: text/html; charset=UTF-8'];
        //$headers[] = 'Cc: ed.airxpress@nationalexpress.com';

        wp_mail($to, $subject, $body, $headers);
    }

    public static function getFormattedJourneyText($journeyTitle, $journeyRef)
    {
        $explodedOutbound = explode(' ', $journeyTitle);
        return "Journey: &nbsp;&nbsp;&nbsp;&nbsp;$explodedOutbound[0] $explodedOutbound[1] $explodedOutbound[2]	&nbsp;	&nbsp;	&nbsp;	&nbsp;Time: $explodedOutbound[3]	&nbsp;	&nbsp;	&nbsp;	&nbsp; Date: $explodedOutbound[4] $explodedOutbound[5] $explodedOutbound[6]
                <br> <span style=\"font-weight: 600; font-size:20px;\">Reference: &nbsp;&nbsp; $journeyRef</span> <br>";
    }

    public static function emailOverBookingDetails($bookingId)
    {
        $to = get_field('email', $bookingId);
        $name = get_field('name', $bookingId);
        $surname = get_field('surname', $bookingId);
        $paymentReference = get_field('payment_reference', $bookingId);

        $subject = 'Edinburgh Airport Xpress - Overbooking alert for ' . $paymentReference;
        $tickets = get_field('tickets', $bookingId);
        $outboundJourneyId = get_field('outbound_journey', $bookingId);
        $outboundJourney = get_the_title($outboundJourneyId[0]);
        $outboundReference = get_field('outbound_reference', $bookingId);

        $inboundJourneyId = get_field('inbound_journey', $bookingId);
        $inboundJourney = get_the_title($inboundJourneyId[0]);
        $inboundReference = get_field('inbound_reference', $bookingId);
        $addOns = get_field('add_ons', $paymentReference);

        $body = '<img src="' . get_field('logo', 'options') . '" style="max-width: 200px; height: auto; margin-bottom: 15px;"> <br>';
        $body .= '<p>The following order has been placed and has overbooked one or more of the journeys. Please review and contact the customer.</p>';
        $body .= '<p><strong>Booking Name: </strong>' . $name . ' ' . $surname . '.</p>';
        $body .= '<p><strong>Payment Reference:</strong> ' . $paymentReference . '</p>';
        $body .= '<p style="font-weight: 600; font-size:20px;">Tickets:</p>';

        foreach ($tickets as $ticket) {
            $body .= '<p style="font-weight: 600; font-size:20px;">' . $ticket['ticket_type'] . ', Quantity: ' . $ticket['quantity'] . '</p><br>';
        }

        $extras = explode(',', get_field('add_ons', $bookingId));

        if ($extras[0] !== 'no') {
            $body .= '<p style="font-weight: 600;">Extra luggage: ' . $extras[1] . '</p>';
        }
        if ($extras[2] !== 'no') {
            $body .= '<p style="font-weight: 600;">Oversized baggage details: ' . $extras[3] . '</p>';
        }

        $body .= self::getFormattedJourneyText($outboundJourney, $outboundReference);
        if ($inboundJourney) {
            $body .= self::getFormattedJourneyText($inboundJourney, $inboundReference);
        }

        $headers = ['Content-Type: text/html; charset=UTF-8'];
        //$headers[] = 'Cc: ed.airxpress@nationalexpress.com';

        wp_mail($to, $subject, $body, $headers);
    }

    public function checkJourneySeats($bookingId, $journeyDirection, $seats)
    {
        $journey = get_field($journeyDirection, $bookingId);
        if ($journey) {
            $journeySeats = get_field('seats', $journey[0]);
            if ($journeySeats - $seats >= 0) {
                update_field('seats', $journeySeats - $seats, $journey[0]);
                acf_flush_value_cache($journey[0], 'seats');
            } else {
                self::emailOverBookingDetails($bookingId);
            }
        }
    }

    public function getConfirmationText($request)
    {
        $bookingId = $request['bookingId'];
        if ($bookingId == null) {
            return new \WP_REST_Response('No id was sent', 400);
        }

        $outboundJourneyId = get_field('outbound_journey', $bookingId);
        $outboundRef = get_field('outbound_reference', $bookingId);
        $response[] = self::getFormattedJourneyText(get_the_title($outboundJourneyId[0]), $outboundRef);

        $inboundJourneyId = get_field('inbound_journey', $bookingId);
        if ($inboundJourneyId) {
            $inboundRef = get_field('inbound_reference', $bookingId);
            $response[] = self::getFormattedJourneyText(get_the_title($inboundJourneyId[0]), $inboundRef);
        }

        return $response;
    }

    public function addNewBooking($request)
    {
        $requiredFields = [
            'firstName',
            'surname',
            'email',
            'phoneNumber',
            'outbound',
            'seats',
        ];
        //Check required fields are set, and return error if not
        foreach ($requiredFields as $field) {
            if (!isset($request[$field]) || $request[$field] === '') {
                return new \WP_REST_Response('Required fields are missing.', 400);
            }
        }
        $name = $request['firstName'];
        $surname = $request['surname'];
        $email = $request['email'];
        $phone = $request['phoneNumber'];
        $seats = $request['seats'];
        $price = $request['price'];

        $outboundSlug = $request['outbound'];
        $outboundReference = explode('-', $outboundSlug);
        $outboundDateString = $outboundReference[4] . '/' . $outboundReference[5] . '/' . $outboundReference[6];
        $outboundDate = \DateTime::createFromFormat('d/m/Y', $outboundDateString);

        //Get difference in outbound date and today, for comparison of advance ticket pricing
        $now = new \DateTime("now");
        //Get interval in total days
        $interval = $now->diff($outboundDate)->format("%a");
        //Format outbound date for post
        $outboundDate = $outboundDate->format('Ymd');
        $outboundTime = substr($outboundSlug, 20, 4);
        //Check if this is a return journey with both directions set
        $includeInbound = isset($request['inbound']) && $request['inbound'] !== "false" && $request['inbound'] !== false;

        //Check that the price from front end matches backend
        $priceCheck = $this->checkCorrectPrice($request, (int)$interval >= 14, $includeInbound);
        if ($priceCheck !== true) {
            return $priceCheck;
        }

        //Format the time
        $outboundTime = substr_replace($outboundTime, ":", 2, 0);
        $outboundTime = substr_replace($outboundTime, ":00", 5, 0);

        $outboundReference = strtoupper($outboundReference[0] . '-' . $outboundReference[2] . '-' . $outboundReference[3] . '-' . mt_rand(00000, 99999));

        // If the slug is wrong here, don't add the journey.
        if (strpos($request['outbound'], 'dundee-to-edinburgh') === false && strpos($request['outbound'], 'edinburgh-to-dundee') === false) {
            die;
        }

        $outboundJourney = self::updateOrInsertPostJourney($outboundSlug, $seats);

        $inboundJourney = null;

        if ($includeInbound) {
            $inboundSlug = $request['inbound'];
            $inboundReference = explode('-', $inboundSlug);
            $inboundDateString = $inboundReference[4] . '/' . $inboundReference[5] . '/' . $inboundReference[6];
            $inboundDate = \DateTime::createFromFormat('d/m/Y', $inboundDateString);
            $inboundDate = $inboundDate->format('Ymd');

            $inboundReference = strtoupper($inboundReference[0] . '-' . $inboundReference[2] . '-' . $inboundReference[3] . '-' . mt_rand(00000, 99999));
            $inboundJourney = self::updateOrInsertPostJourney($inboundSlug, $seats);
        }
        $addOns = [];
        foreach ($request['addons'] as $key => $addon) {
            $addOns[] = $key . ': <strong>' . $addon . '</strong>';
        }
        $addOns = implode("\n", $addOns);

        $cartItem = $request['inCart'];
        $tickets = [];
        foreach ($cartItem as $ticket) {
            $tickets[] = [
                'ticket_type' => $ticket['title'],
                'quantity' => $ticket['quantity'],
            ];
        }


        if ((isset($inboundJourney) && $inboundJourney === 'no-seats') || $outboundJourney === 'no-seats') {
            return new \WP_REST_Response('There are not enough seats available on the selected journey', 400);
        } else {
            $args = [
                'post_type' => 'booking',
                'post_title' => $name . ' ' . $surname . ' ' . date("d-m-Y H:i"),
                'post_status' => 'publish',
            ];

            $newBooking = wp_insert_post($args);
            update_field('tickets', $tickets, $newBooking);
            update_field('outbound_journey', [$outboundJourney], $newBooking);
            update_field('outbound_reference', $outboundReference, $newBooking);
            update_field('departure_time', $outboundTime, $newBooking);
            if ($includeInbound) {
                update_field('inbound_journey', [$inboundJourney], $newBooking);
                update_field('inbound_reference', $inboundReference, $newBooking);
            }
            update_field('add_ons', $addOns, $newBooking);
            update_field('name', $name, $newBooking);
            update_field('surname', $surname, $newBooking);
            update_field('email', $email, $newBooking);
            update_field('phone_number', $phone, $newBooking);
            update_field('price', $price, $newBooking);
            update_field('total_seats', $seats, $newBooking);
            update_field('outbound_booking_time', $outboundDate, $newBooking);

            if ($includeInbound) {
                update_field('inbound_booking_time', $inboundDate, $newBooking);
            }
            return $newBooking;
        }
    }

    /**
     * @param $request
     * @param $isAdvance
     * @param $includeInbound
     * @return bool|\WP_REST_Response
     */
    private function checkCorrectPrice($request, $isAdvance, $includeInbound)
    {
        //Get state of entire cart
        $cartItem = $request['inCart'];

        $realPrice = 0;
        $totalSeats = 0;
        //Verify front end and backend price match
        foreach ($cartItem as $ticket) {
            $realTicket = get_page_by_title($ticket['title'], null, 'ticket');
            if ($ticket['seats'] != get_field('seats', $realTicket->ID)) {
                return new \WP_REST_Response('Incorrect seats for ticket type.', 400);
            }
            //Use CMS value to calculate correct price
            //Check if interval is greater or equal to 14 days. If so, use advanced ticket pricing for comparison.
            if ($isAdvance) {
                $realPrice += $includeInbound ? get_field('advanced_return', $realTicket->ID) * $ticket['quantity'] : get_field('advanced_single', $realTicket->ID) * $ticket['quantity'];
            } else {
                $realPrice += $includeInbound ? get_field('return_cost', $realTicket->ID) * $ticket['quantity'] : get_field('single_cost', $realTicket->ID) * $ticket['quantity'];
            }

            $totalSeats += $ticket['seats'] * $ticket['quantity'];
        }
        //Add Booking Fee
        $bookingFee = min(6, $totalSeats);
        $realPrice += $bookingFee;
        //Compare tickets vs price
        if ($request['price'] != $realPrice) {
            return new \WP_REST_Response('Incorrect price for number of tickets.', 400);
        }

        return true;
    }

    public static function updateOrInsertPostJourney($slug, $seats)
    {
        $newJourneyBooking = '';

        $postTitle = ucwords(str_replace('-', ' ', $slug));
        $postParts = explode('-', $slug);

        $date = $postParts['4'] . '-' . $postParts['5'] . '-' . $postParts['6'];
        $date = date('Ymd', strtotime($date));

        $jourArgs = [
            'name' => $postParts[0] . '-' . $postParts[1] . '-' . $postParts[2] . '-' . $postParts[3],
            'post_type' => 'journey',
            'posts_per_page' => 1
        ];

        $seatsQuery = new \WP_Query($jourArgs);

        if ($seatsQuery->have_posts()) {
            while ($seatsQuery->have_posts()) {
                $seatsQuery->the_post();
                $totalSeats = get_field('seats');
                $id = get_the_ID();
                $type = wp_get_object_terms($id, 'journey-type');
                $journeyType = $type[0]->term_id;
                $departTime = get_field('departure_time');
            }
        } else {
            $totalSeats = self::SEATS;
        }

        $args = [
            'name' => $slug,
            'post_type' => 'journey-date',
            'posts_per_page' => 1,
        ];

        $query = new \WP_Query($args);

        if (!$query->have_posts()) {
            $newJourneyArgs = [
                'post_title' => $postTitle,
                'post_type' => 'journey-date',
                'post_status' => 'publish',
            ];

            $newJourneyBooking = wp_insert_post($newJourneyArgs);
            //setting total seats
            update_field('total_seats', $totalSeats, $newJourneyBooking);
            update_field('date', $date, $newJourneyBooking);
            update_field('departure_time', $departTime, $newJourneyBooking);
            wp_set_post_terms(get_the_ID(), [$journeyType], 'journey-type');
            //remaining seats
            update_field('seats', $totalSeats - $seats, $newJourneyBooking);
        } else {
            while ($query->have_posts()) {
                $query->the_post();
                wp_update_post(get_the_ID());
                $seatsLeft = get_field('seats', get_the_ID());
                if ($seatsLeft >= $seats) {
                    update_field('seats', ($seatsLeft - $seats), get_the_ID());
                    update_field('departure_time', $departTime, get_the_ID());
                    wp_set_post_terms(get_the_ID(), [$journeyType], 'journey-type');
                    $newJourneyBooking = get_the_ID();
                } else {
                    $newJourneyBooking = 'no-seats';
                }
            }
        }
        return $newJourneyBooking;
    }

    public function addNotes($request)
    {
        $bookingId = $request['bookingId'];
        $notes = $request['notes'];

        update_field('agent_payment_details', $notes, $bookingId);
        update_field('booking_status', 'Completed', $bookingId);
        update_field('payment_reference', 'Cash', $bookingId);

        $seats = get_field('total_seats', $bookingId);

        $this->checkJourneySeats($bookingId, 'outbound_journey', $seats);
        $this->checkJourneySeats($bookingId, 'inbound_journey', $seats);

        self::emailUserBookingDetails('paid', $bookingId);
    }

    public function outboundJourneyUpdated($value, $post_id, $field)
    {
        if (is_admin() && $value[0] != get_field('outbound_journey', $post_id)[0]) {
            // Get the ticket type
            // Calculate no. of seats
            $tickets = get_field('tickets', $post_id);
            $seats = AdminController::getPeoplePerTicket($tickets);

            // Get the remaining seats on the new journey
            $remainingSeats = get_field('seats', $value[0]);

            // Get the seats on the current/"old" journey
            $seatsOnOld = get_field('seats', get_field('outbound_journey', $post_id)[0]);

            $_GET['didNotUpdate'] = true;
            // if there enough seats update the booking
            if ($seats <= $remainingSeats) {
                // New journey's new no. of seats
                $remainingSeats = $remainingSeats - $seats;
                // Old journey's new no. of seats
                $seatsOnOld = $seatsOnOld + $seats;

                // Subtract the number of seats from the new journey
                update_field('seats', $remainingSeats, $value[0]);

                // Add the number of seats back on to the journey these bookings are being removed from
                $seatsOnOld = ($seatsOnOld > self::SEATS) ? self::SEATS : $seatsOnOld; // Don't think this will be needed but...
                update_field('seats', $seatsOnOld, get_field('outbound_journey', $post_id)[0]);
                return $value;
            } else {
                add_filter('redirect_post_location', [$this, 'addQueryString'], 99);
                return "";
            }
        }
        return $value;
    }

    public function addQueryString($location)
    {
        remove_filter('redirect_post_location', [$this, 'addQueryString'], 99);
        return add_query_arg(['notUpdated' => 'ID'], $location);
    }

    public function bookingAmended($post_id, $post, $update)
    {
        if (!is_admin()) {
            return;
        }

        if ($post->post_type === 'booking') {
            add_filter('redirect_post_location', [$this, 'add_notice_query_var'], 99);
        }
    }

    public function add_notice_query_var($location)
    {
        remove_filter('redirect_post_location', [$this, 'add_notice_query_var'], 99);
        self::emailUserBookingDetails('bookingAmended', get_the_ID());
        return add_query_arg(['updated' => get_the_ID()], $location);
    }

    public function adminNotices()
    {
        if (!isset($_GET['updated'])) {
            return;
        } ?>
        <div class='updated'>
            <p><a href="<?= home_url() ?>/booking/?complete=<?= $_GET['updated'] ?>&updated=1" target="_blank">View
                    Confirmation</a></p>
        </div>
        <?php
    }

    public function notAmended()
    {
        if (isset($_GET['notUpdated'])) {
        ?>
            <div class='error'>
                <p>Booking Journey not changed, not enough seats on the bus!</p>
            </div>
<?php
        }
    }

    public function postFilter($query)
    {
        if (is_admin() && $query->get('post_type') == "booking" && isset($_GET['customFilter']) && isset($_GET['journeyID'])) {
            $metaArgs = [
                'relation' => 'OR',
                [
                    'key' => 'inbound_journey',
                    'value' => '"' . $_GET['journeyID'] . '"',
                    'compare' => 'LIKE',
                ],
                [
                    'key' => 'outbound_journey',
                    'value' => '"' . $_GET['journeyID'] . '"',
                    'compare' => 'LIKE',
                ],
            ];
            $query->set('meta_query', $metaArgs);
        }
        return $query;
    }
}
