<?php

namespace XploreBooking;

use XploreBooking\Controller\JourneyController;
use XploreBooking\Controller\TicketController;

class XploreBookingPlugin
{
    private $pluginDirectoryPath;
    private $pluginDirectoryURL;
    private $journeyController;
    private $bookingController;
    private $ticketController;
    private $importController;

    public function __construct($pluginDirectoryPath, $pluginDirectoryURL)
    {
        $this->site = new \Blue2\Common\Site();
        $this->pluginDirectoryPath = $pluginDirectoryPath;
        $this->pluginDirectoryURL = $pluginDirectoryURL;

        $this->setupControllers();
        $this->registerShortCodes();
        $this->registerActions();
        $this->registerFilters();

        add_theme_support('soil-google-analytics', 'UA-22555137-13');
        add_role('agent', 'Agent', ['read' => true]);
        add_action('template_redirect', [$this, 'templateRedirect']);
        if (function_exists('acf_add_options_page')) {
            acf_add_options_page([
                'page_title' => 'Site Options'
            ]);
        }
        add_action('admin_head', [$this, 'adminStyles']);
    }

    public function adminStyles() {
        $postType = get_post_type();
        if(!is_admin() || ($postType !== 'booking' && $postType !== 'journey-date')) {
            return;
        } ?>

        <style>
            #filter-by-date {
                display: none;
            }
        </style>
        <?php
    }

    public function templateRedirect()
    {
        if (is_404() && !isset($_POST['name'])) {
            wp_redirect(home_url('/booking/'));
            exit();
        }
    }

    private function setupControllers()
    {
        $this->journeyController = new Controller\JourneyController();
        $this->bookingController = new Controller\BookingController();
        $this->ticketController = new Controller\TicketController();
        $this->ticketController = new Controller\AdminController();
        $this->importController = new Controller\ImportController($this->bookingController);
    }

    private function registerShortCodes()
    {
    }

    private function registerActions()
    {
        add_action('wp_enqueue_scripts', [$this, 'enqueueAssets']);
        add_action('template_redirect', [$this, 'redirectPages']);
        add_action('init', [$this, 'registerPostTypes']);

        add_action('rest_api_init', function () {
            if (is_user_logged_in()) {
                register_rest_route('xplore-booking/v1', '/verify-login/', [
                    'methods' => 'POST',
                    'callback' => function () {
                        return true;
                    },
                ]);
            } else {
                register_rest_route('xplore-booking/v1', '/verify-login/', [
                    'methods' => 'POST',
                    'callback' => function () {
                        return false;
                    },
                ]);
            }

            register_rest_route('xplore-booking/v1', '/get-policy/', [
                'methods' => 'GET',
                'callback' => function () {
                    return get_field('refund_policy', 'options');
                },
            ]);
        });
    }

    private function registerFilters()
    {
        add_filter('page_template', [$this, 'assignTemplates']);
        add_filter('acf/settings/save_json', [$this, 'acfJsonSaveLocation']);
        add_filter('acf/settings/load_json', [$this, 'acfJsonLoadLocation']);
    }

    public function registerPostTypes()
    {
        Post\Journey::registerPostType();
        Post\JourneyDate::registerPostType();
        Post\Booking::registerPostType();
        Post\Ticket::registerPostType();
    }

    public function pluginActivation()
    {
        $this->createPages();
        $this->createTickets();
        $this->createJourneys();
    }

    private function createPages()
    {
        $pageDefinitions = [
            'booking' => [
                'post_title' => 'Booking',
            ],
            'confirmation' => [
                'post_title' => 'Confirmation',
            ],
        ];
        foreach ($pageDefinitions as $slug => $pageArgs) {
            // Check that the page doesn't exist already
            $query = new \WP_Query('pagename=' . $slug);
            if (!$query->have_posts()) {
                // Add the page using the data from the array above
                $defaultArgs = [
                    'post_name' => $slug,
                    'post_status' => 'publish',
                    'post_type' => 'page',
                    'ping_status' => 'closed',
                    'comment_status' => 'closed',
                ];
                $args = array_merge($pageArgs, $defaultArgs);
                wp_insert_post($args);
            }
        }
    }

    private function createTickets()
    {
        $ticketTypes = TicketController::ticketTypes();

        foreach ($ticketTypes as $slug => $pageArgs) {
            $query = new \WP_Query('pagename=' . $slug);
            if (!$query->have_posts()) {
                $defaultArgs = [
                    'post_name' => $slug,
                    'post_status' => 'publish',
                    'post_type' => 'ticket',
                    'ping_status' => 'closed',
                    'comment_status' => 'closed',
                ];
                $args = array_merge($pageArgs, $defaultArgs);
                $postID = wp_insert_post($args);
                update_field('single_cost', $pageArgs['single_cost'], $postID);
                update_field('return_cost', $pageArgs['return_cost'], $postID);
                update_field('advanced_single', $pageArgs['advanced_single'], $postID);
                update_field('advanced_return', $pageArgs['advanced_return'], $postID);
                update_field('seats', $pageArgs['seats'], $postID);
                update_field('staff_ticket', $pageArgs['staff_ticket'], $postID);
            }
        }
    }

    private function createJourneys()
    {
        $pageDefinitions = JourneyController::journeyTimes();

        foreach ($pageDefinitions as $slug => $pageArgs) {
            // Check that the page doesn't exist already
            $query = new \WP_Query('pagename=' . $slug);
            if (!$query->have_posts()) {
                // Add the page using the data from the array above
                $defaultArgs = [
                    'post_name' => $slug,
                    'post_status' => 'publish',
                    'post_type' => 'journey',
                    'ping_status' => 'closed',
                    'comment_status' => 'closed',
                ];
                $args = array_merge($pageArgs, $defaultArgs);
                $postID = wp_insert_post($args);
                update_field('departure_time', $pageArgs['departure_time'], $postID);
                update_field('arrival_time', $pageArgs['arrival_time'], $postID);
            }
        }
    }

    public function enqueueAssets()
    {
        wp_enqueue_script('xplore-booking-app', $this->pluginDirectoryURL . '/dist/main.js', [], '1.1.1', true);
        wp_localize_script( 'xplore-booking-app', 'wpApiSettings', array(
            'nonce' => wp_create_nonce( 'wp_rest' )
        ));
        wp_enqueue_style('xplore-booking', $this->pluginDirectoryURL . '/dist/main.css', [], '1');
    }

    public function redirectPages()
    {
        preg_match('(\bbooking\/\w)', $_SERVER['REQUEST_URI'], $matches);
        if ($_SERVER['REQUEST_URI'] === '/' || !empty($matches)) {
            wp_redirect(home_url('/booking/'));
            die;
        }
    }

    public function assignTemplates($page_template)
    {
        if (is_page('booking')) {
            $page_template = $this->pluginDirectoryPath . '/templates/template-xplore-booking.php';
        }
        return $page_template;
    }

    public function checkValidUserType()
    {
        return !empty(array_intersect(['administrator'], $this->getUserRoles()));
    }

    public function getUserRoles()
    {
        $roles = '';
        if (is_user_logged_in()) {
            $user = wp_get_current_user();
            $roles = $user->roles;
        }
        return $roles;
    }

    public function acfJsonSaveLocation()
    {
        $path = $this->pluginDirectoryPath . '/acf-json';
        return $path;
    }

    public function acfJsonLoadLocation($paths)
    {
        unset($paths[0]);
        $paths[] = $this->pluginDirectoryPath . '/acf-json';
        return $paths;
    }
}
