<?php

namespace XploreBooking\Controller;

class BookingController
{

    const SEATS = 29;

    public function __construct()
    {
        $this->registerActions();
    }

    private function registerActions()
    {
        add_action('rest_api_init', function () {
            register_rest_route('xplore-booking/v1', '/add-new-booking/', [
                'methods' => 'POST',
                'callback' => [$this, 'addNewBooking'],
            ]);

            register_rest_route('xplore-booking/v1', '/set-booking-notes/', [
                'methods' => 'POST',
                'callback' => [$this, 'addNotes'],
            ]);

            register_rest_route('xplore-booking/v1', '/update-payment-status/', [
                'methods' => 'POST',
                'callback' => [$this, 'updatePaymentStatus'],
            ]);

            register_rest_route('xplore-booking/v1', '/get-confirmation-text/', [
                'methods' => 'GET',
                'callback' => [$this, 'getConfirmationText'],
            ]);
        });

        add_action('template_redirect',[$this, 'confirmationRedirect']);

        add_action('save_post', [$this, 'bookingAmended'], 10, 3);
        add_action( 'admin_notices', [$this, 'adminNotices'] );
    }

    public function confirmationRedirect() {
        if (is_404() && isset($_POST['name']) && isset($_POST['transStatus'])) {
            self::updatePaymentStatus();
            $bookingId = $_POST['cartId'];
            status_header(200);
            echo '
            <!DOCTYPE html>
            <html>
            <head>
                <meta name="viewport" content="width=device-width, initial-scale=1">
                <link rel="profile" href="http://gmpg.org/xfn/11">
                <META HTTP-EQUIV="Refresh" CONTENT="3; URL=http://booking.xploremore.co.uk/booking/?complete=' . $bookingId . '">
            </head>
            <WPDISPLAY FILE="header.html">
            <body>
                <p>Redirecting to site</p>
            </body>
            <WPDISPLAY FILE="footer.html">
            </html>';
            exit;
        }
    }

    public static function updatePaymentStatus()
    {
        $status = $_POST['transStatus'] == 'Y' ? 'paid' : 'failed';
        $bookingId = $_POST['cartId'];
        $paymentReference = $_POST['transId'];
        switch ($status) {
            case 'paid':
                update_field('payment_reference', $paymentReference, $bookingId);
                update_field('booking_status', 'Completed', $bookingId);
                wp_update_post(['ID' => $bookingId]);
                self::emailUserBookingDetails($status, $bookingId);
                break;
            case 'failed':
                update_field('payment_reference', $paymentReference, $bookingId);
                update_field('booking_status', 'Failed', $bookingId);
                self::emailUserBookingDetails($status, $bookingId);
                break;
            default :
                update_field('booking_status', 'Pending', $bookingId);
                wp_update_post(['ID' => $bookingId]);
                break;
        }
    }

    public function getConfirmationText($request) {
        $bookingId = $request['bookingId'];
        if($bookingId == null) {
            return new \WP_REST_Response('No id was sent', 400);
        }

        $outboundJourneyId = get_field('outbound_journey', $bookingId)[0];
        $outboundRef = get_field('outbound_reference', $bookingId);
        $response[] = self::getFormattedJourneyText(get_the_title($outboundJourneyId), $outboundRef);

        $inboundJourneyId = get_field('inbound_journey', $bookingId)[0];
        if($inboundJourneyId) {
            $inboundRef = get_field('inbound_reference', $bookingId);
            $response[] = self::getFormattedJourneyText(get_the_title($inboundJourneyId), $inboundRef);
        }
        $email = get_field('email', $bookingId);
        $response[] = "An email will be sent to $email";

        return $response;
    }

    public static function getFormattedJourneyText($journeyTitle, $journeyRef) {
        $explodedOutbound = explode(' ', $journeyTitle);
        return "Journey: &nbsp;&nbsp;&nbsp;&nbsp;$explodedOutbound[0] $explodedOutbound[1] $explodedOutbound[2]	&nbsp;	&nbsp;	&nbsp;	&nbsp;Time: $explodedOutbound[3]	&nbsp;	&nbsp;	&nbsp;	&nbsp; Date: $explodedOutbound[4] $explodedOutbound[5] $explodedOutbound[6]
                <br> <span style=\"font-weight: 600; font-size:20px;\">Reference: &nbsp;&nbsp; $journeyRef</span> <br>";
    }

    public static function emailUserBookingDetails($status, $bookingId)
    {
        $to = get_field('email', $bookingId);
        $name = get_field('name', $bookingId);
        $surname = get_field('surname', $bookingId);
        $paymentReference = get_field('payment_reference', $bookingId);
        switch ($status) {
            case 'paid':
            case 'bookingAmended':
                $subject = 'Edinburgh Airport Xpress - successful booking for ' . $paymentReference;
                $tickets = get_field('tickets', $bookingId);
                $outboundJourneyId = get_field('outbound_journey', $bookingId)[0];
                $outboundJourney = get_the_title($outboundJourneyId);
                $outboundReference = get_field('outbound_reference', $bookingId);

                $inboundJourneyId = get_field('inbound_journey', $bookingId)[0];
                $inboundJourney = get_the_title($inboundJourneyId);
                $inboundReference = get_field('inbound_reference', $bookingId);
                $addOns = get_field('add_ons', $paymentReference);

                $body = '<img src="' . get_field('logo', 'options') . '" style="max-width: 200px; height: auto; margin-bottom: 15px;"> <br>';
                $body .= '<p>Dear ' . $name . ' ' . $surname . '</p>';
                $body .= '<p>Thank you for booking the Xplore More Edinburgh Airport Xpress Service from/to Dundee.   Below you will find your booking and journey details for you/your party - please check these carefully.</p>';
                $body .= '<p><strong>This confirmation is your coach ticket and must be shown to the driver when boarding the vehicle, and all members of your party must be present at the time of boarding.</strong></p><br>';
                $body .= '<p><strong>Please ensure that you arrive at the coach stance at least 10 minutes prior to the time of departure as after this time your seat may be released to other customers</strong></p>';
                $body .= '<p><strong>Booking Name: </strong>' . $name . ' ' . $surname . '.</p>';
                $body .= '<p><strong>Payment Reference:</strong> ' . $paymentReference . '</p>';
                $body .= '<p style="font-weight: 600; font-size:20px;">Tickets:</p>';

                foreach ($tickets as $ticket) {
                    $body .= '<p style="font-weight: 600; font-size:20px;">' . $ticket['ticket_type'] . ', Quantity: ' . $ticket['quantity'] . '</p><br>';
                }

                $extras = explode(',', get_field('add_ons', $bookingId));

                if($extras[0] !== 'no') {
                    $body .= '<p style="font-weight: 600;">Extra luggage: ' . $extras[1].'</p>';
                }
                if($extras[2] !== 'no') {
                    $body .= '<p style="font-weight: 600;">Oversized baggage details: ' . $extras[3].'</p>';
                }

                $body .= self::getFormattedJourneyText($outboundJourney, $outboundReference);
                if ($inboundJourney) {
                    $body .= self::getFormattedJourneyText($inboundJourney, $inboundReference);
                }
                $body .= '<p>
                            <strong>Scottish Concessionary Card Holders</strong> <br />
                            All Scottish Concessionary Card Holders must ensure that they present their cards to
                            the driver and scan them on the ticket machines to obtain their concessionary
                            tickets. If you are unable to present your concessionary card you will be asked to
                            pay the relevant standard fare for your journey.
                        </p>';
                $body .= '<p>
                            <strong>Students</strong> <br />
                            Students should ensure that they have a current and valid matriculation card or
                            letter of acceptance at a college/university available to show to the driver prior
                            to boarding the coach.
                        </p>';
                $body .= '<p>
                        If you have any queries or questions regarding your booking please call 01382 201121
                        during office hours Monday to Friday 9am - 5pm.
                    </p>
                    <p>
                        Outwith these hours please call the National Express helpline
                        number on 0871 781 8181 and they will be more than happy to assist you in your
                        enquiry
                    </p>';
                $body .= '<p><strong>Have an enjoyable trip.</strong></p>';
                break;
            case 'failed':
                $subject = '<p>Unsuccessful Booking for ' . $paymentReference . '</p>';
                $body = '<p>Unfortunately payment was not succesful for reference: ' . $paymentReference . '</p>';
                $body .= '<p>Please try again or contact <a href="tel:01382 340006">01382 340006</a> or <a href="mailto:XploreMore@nationalexpress.com">XploreMore@nationalexpress.com</a></p>';
                break;
        }

        $headers = ['Content-Type: text/html; charset=UTF-8'];
        $headers[] = 'Cc: ed.airxpress@nationalexpress.com';

        wp_mail($to, $subject, $body, $headers);
    }

    public function addNewBooking($request)
    {
        $requiredFields = [
            'firstName',
            'surname',
            'email',
            'phoneNumber',
            'outbound',
            'seats'
        ];
        foreach($requiredFields as $field) {
            if(!isset($request[$field]) || $request[$field] === '') {
                return new \WP_REST_Response('Required fields are missing.', 400);
            }
        }

        $name = $request['firstName'];
        $surname = $request['surname'];
        $email = $request['email'];
        $phone = $request['phoneNumber'];
        $seats = $request['seats'];
        $price = $request['price'];

        $outboundSlug = $request['outbound'];
        $outboundReference = explode('-', $outboundSlug);
        $outboundDateString = $outboundReference[4].'/'.$outboundReference[5].'/'.$outboundReference[6];
        $outboundDate = \DateTime::createFromFormat('d/m/Y', $outboundDateString);
        $outboundDate = $outboundDate->format('Ymd');

        $outboundReference = strtoupper($outboundReference[0] . '-' . $outboundReference[2] . '-' . $outboundReference[3] . '-' . mt_rand(00000, 99999));
        $outboundJourney = self::updateOrInsertPostJourney($outboundSlug, $seats);

        if(isset($request['inbound']) && $request['inbound'] !== false) {
            $inboundSlug = $request['inbound'];
            $inboundReference = explode('-', $inboundSlug);
            $inboundDateString = $inboundReference[4].'/'.$inboundReference[5].'/'.$inboundReference[6];
            $inboundDate = \DateTime::createFromFormat('d/m/Y', $inboundDateString);
            $inboundDate = $inboundDate->format('Ymd');


            $inboundReference = strtoupper($inboundReference[0].'-'.$inboundReference[2].'-'.$inboundReference[3].'-'.mt_rand(00000, 99999));
            $inboundJourney = self::updateOrInsertPostJourney($inboundSlug, $seats);
        }

        $addOns = implode(",",$request['addons']);

        $cartItem = $request['inCart'];
        $tickets = [];
        foreach($cartItem as $ticket) {
            $tickets[] = [
                'ticket_type' => $ticket['title'],
                'quantity' => $ticket['quantity']
            ];
        }


        if ( (isset($inboundJourney) && $inboundJourney === 'no-seats')  || $outboundJourney === 'no-seats') {
            return new \WP_REST_Response('There are not enough seats available on the selected journey', 400);
        } else {
            $args = [
                'post_type' => 'booking',
                'post_title' => $name . ' ' . $surname . ' ' . date("d-m-Y H:i"),
                'post_status' => 'publish'
            ];

            $newBooking = wp_insert_post($args);
            update_field('tickets', $tickets, $newBooking);
            update_field('outbound_journey', [$outboundJourney], $newBooking);
            update_field('outbound_reference', $outboundReference, $newBooking);
            update_field('inbound_journey', [$inboundJourney], $newBooking);
            update_field('inbound_reference', $inboundReference, $newBooking);
            update_field('add_ons', $addOns, $newBooking);
            update_field('name', $name, $newBooking);
            update_field('surname', $surname, $newBooking);
            update_field('email', $email, $newBooking);
            update_field('phone_number', $phone, $newBooking);
            update_field('price', $price, $newBooking);
            update_field('outbound_booking_time', $outboundDate, $newBooking);
            update_field('inbound_booking_time', $inboundDate, $newBooking);

            return $newBooking;
        }
    }

    public function addNotes($request) {
        $bookingId = $request['bookingId'];
        $notes = $request['notes'];

        update_field('agent_payment_details', $notes, $bookingId);
        update_field('booking_status', 'Completed', $bookingId);
        update_field('payment_reference', 'Cash', $bookingId);

        self::emailUserBookingDetails('paid', $bookingId);
    }

    public static function updateOrInsertPostJourney($slug, $seats)
    {
        $newJourneyBooking = '';

        $postTitle = ucwords(str_replace('-', ' ', $slug));
        $postParts = explode('-', $slug);

        $date = $postParts['4'] . '-' . $postParts['5'] . '-' . $postParts['6'];
        $date = date('Ymd', strtotime($date));

        $args = [
            'name' => $slug,
            'post_type' => 'journey-date',
            'posts_per_page' => 1
        ];

        $query = new \WP_Query($args);

        if (!$query->have_posts()) {

            $newJourneyArgs = [
                'post_title' => $postTitle,
                'post_type' => 'journey-date',
                'post_status' => 'publish'
            ];

            $newJourneyBooking = wp_insert_post($newJourneyArgs);
            update_field('date', $date, $newJourneyBooking);
            update_field('seats', (self::SEATS - $seats), $newJourneyBooking);
        } else {
            while ($query->have_posts()) {
                $query->the_post();
                wp_update_post(get_the_ID());
                $seatsLeft = get_field('seats', get_the_ID());
                if ($seatsLeft >= $seats) {
                    update_field('seats', ($seatsLeft - $seats), get_the_ID());
                    $newJourneyBooking = get_the_ID();
                } else {
                    $newJourneyBooking = 'no-seats';
                }
            }
        }
        return $newJourneyBooking;
    }

    function bookingAmended( $post_id, $post, $update ) {
        if ( $post->post_type === 'booking') {
            add_filter( 'redirect_post_location', [$this, 'add_notice_query_var'], 99 );
        }
    }

    public function add_notice_query_var( $location ) {
        remove_filter( 'redirect_post_location', [$this, 'add_notice_query_var'], 99 );
        self::emailUserBookingDetails('bookingAmended', get_the_ID());
        return add_query_arg(['updated' => get_the_ID()], $location );
    }

    public function adminNotices()
    {
        if ( ! isset( $_GET['updated'] ) ) {
            return;
          }
          ?>
          <div class='updated'>
             <p><a href="<?=home_url()?>/booking/?complete=<?=$_GET['updated']?>" target="_blank">View Confirmation</a></p>
          </div>
          <?php 
    }
}
