<?php


namespace XploreBooking\Controller;


use XploreBooking\Post\Booking;

class ImportController
{

    private $bookingController;

    /**
     * ImportController constructor.
     * @param BookingController $bookingController
     */
    public function __construct($bookingController)
    {
        $this->bookingController = $bookingController;
        add_action('init', [$this, 'handleFileUpload']);
    }

    public function handleFileUpload()
    {
        if (!is_admin()) return;

        echo "<pre>";
        if (isset($_POST['importSubmit']) && isset($_FILES['bookingCsv'])) {
            $csv = $this->loadCsv();
            foreach($csv as $booking) {
                $outbound = $this->getJourneySlug($booking, 'Outbound');

                if($outbound == null) {
                    //Bus is single, but departs from edinburgh. Stored in Return col even though it's not a return journey
                    $outbound = $this->getJourneySlug($booking, 'Return');
                    $inbound = null;
                } else {
                    //Not null, so inbound can be checked as normal
                    $inbound = $this->getJourneySlug($booking, 'Return');
                }

                $splitName = explode(' ', $booking['Name']);
                $newBooking = [
                    'firstName' => $splitName[0],
                    'surname' => $splitName[1] ?: '',
                    'email' => $booking['Email'],
                    'phoneNumber' => $booking['Tel No.'],
                    'seats' => $booking['Total Persons'],
                    'price' => $booking['Total'],
                    'outbound' => $outbound,
                    'inbound' => $inbound
                ];
                $result = $this->bookingController->addNewBooking($newBooking);
                if(!$result instanceof  \WP_REST_Response) {
                    $this->updateWithDefaults($booking, $result);
                    $this->updateTickets($booking, $result);
                } else {
                    var_dump('Required fields were missing');
                }
            }
        }
        echo "</pre>";
    }

    private function getJourneySlug($booking, $direction)
    {
        $result = null;
        if($booking[$direction . ' Location'] === 'Dundee') {
            $result .= 'dundee-to-edinburgh-';
        } else if($booking[$direction . ' Location'] === 'Edinburgh Airport') {
            $result .= 'edinburgh-to-dundee-';
        }

        if($result !== null) {
            $time = preg_replace('/[^0-9.]/', '', $booking[$direction . ' Bus Time']);
            $result .= $time . '-';
            $departDate = \DateTime::createFromFormat('d/m/Y', $booking[$direction . ' Date']);
            $result .= $departDate->format('d-m-Y');
        }
        return $result;
    }

    /**
     * @return array
     */
    private function loadCsv()
    {
        $csv = array_map('str_getcsv', file($_FILES['bookingCsv']['tmp_name']));
        array_walk($csv, function (&$a) use ($csv) {
            $a = array_combine($csv[0], $a);
        });
        array_shift($csv);
        return $csv;
    }

    /**
     * @param array $booking
     * @param int $bookingId
     */
    private function updateWithDefaults($booking, $bookingId)
    {
        update_field('agent_payment_details', $booking['Notes'], $bookingId);
        update_field('payment_reference', 'Imported', $bookingId);
        update_field('outbound_reference', 'PRE', $bookingId);
        update_field('inbound_reference', 'PRE', $bookingId);
        update_field('booking_status', 'Completed', $bookingId);
    }

    /**
     * @param array $booking
     * @param int $bookingId
     */
    public function updateTickets($booking, $bookingId)
    {
        $tickets = [
            'Adults' => $booking['Adults'],
            'Child' => $booking['Child'],
            'Student' => $booking['Student'],
            'Concession' => $booking['Concession'],
            'Staff' => $booking['Staff'],
        ];

        foreach($tickets as $type => $value) {
            if($value != 0 && $value !== '') {
                add_row('tickets', ['ticket_type' => $type, 'quantity' => $value], $bookingId);
            }
        }
    }
}
