import * as yup from 'yup';

const schema = yup.object().shape({
    name: yup.string().required(),
    surname: yup.string().required(),
    email: yup
        .string()
        .email()
        .required(),
    phoneNumber: yup.number().required()
});

export const getTotalSeatsFromCart = store => {
    const { inCart } = store.cart;
    let seats = 0;
    for (const [key, value] of Object.entries(inCart)) {
        seats += parseInt(value.seats) * parseInt(value.quantity);
    }
    return seats;
};

export const getBookingCost = store => {
    const totalSeats = getTotalSeatsFromCart(store);
    const { bookingFee } = store.cart;
    const bookingCost = totalSeats > 6 ? 6 * bookingFee : bookingFee * totalSeats;
    return bookingCost;
};
export const getTotalCostFromCart = store => {
    const { inCart, journeyType, bookingFee, advanced } = store.cart;
    let priceIndex;
    if (advanced) {
        priceIndex = journeyType === 'SINGLE' ? 'advanced_single' : 'advanced_return';
    } else {
        priceIndex = journeyType === 'SINGLE' ? 'single_cost' : 'return_cost';
    }
    let cost = 0;
    for (let [key, value] of Object.entries(inCart)) {
        cost += value[priceIndex] * value.quantity;
    }

    const bookingCost = getBookingCost(store);
    cost = cost + bookingCost;
    return cost;
};

export const checkValidStep = store => {
    const { step } = store.route;
    let valid = {
        status: true,
        message: ''
    };

    switch (step) {
        case 1: {
            if (Object.keys(store.cart.inCart).length === 0) {
                valid.status = false;
                valid.message = 'At least one ticket must be added';
            }
            if (store.cart.journeyType === 'RETURN' && store.cart.returnDate === null) {
                valid.status = false;
                valid.message = 'A return date must be selected';
            }
            break;
        }
        case 2: {
            const { outbound, inbound, journeyType } = store.cart;
            if (outbound === null) {
                valid.status = false;
                valid.message = 'An outbound journey must be selected';
            }
            if (journeyType === 'RETURN' && inbound === null) {
                valid.status = false;
                valid.message = 'An inbound journey must be selected';
            }
            break;
        }
        case 4: {
            const validSchema = schema.isValidSync(store.user);
            if (!validSchema) {
                valid.status = false;
                valid.message =
                    'Please check you have filled in all the details with valid information';
            }
            const { agreeToTerms } = store.cart;
            if (!agreeToTerms) {
                valid.status = false;
                valid.message = 'You must accept the terms and conditions before continuing';
            }
            break;
        }
        default:
            return valid;
    }

    return valid;
};
