import React, { Component } from 'react';
import axios from 'axios';
import Loader from 'react-loader-spinner';
import { AVAILABLE_JOURNEYS } from './../config/endpoints';
import { connect } from 'react-redux';
import { getTotalSeatsFromCart } from '../redux/selectors';
import { addJourney } from '../redux/actions';
import moment from 'moment';
import NoResults from '../components/ChooseTickets/NoResults';
import posed from 'react-pose';
import { toast } from 'react-toastify';
import { confirmAlert } from 'react-confirm-alert';
import 'react-confirm-alert/src/react-confirm-alert.css';

const Container = posed.div({
    enter: { staggerChildren: 50 }
});

const AnimatedDiv = posed.div({
    enter: { x: 0, opacity: 1 },
    exit: { x: 25, opacity: 0 }
});

class ChooseJourney extends Component {
    constructor(props) {
        super(props);

        this.state = {
            journeys: null,
            loading: true,
            showReturn: false
        };
    }

    componentDidMount() {
        const { journeyType, depart, departureDate, returnDate, totalSeats } = this.props;
        const departFormatted = this.formatDate(departureDate);
        const returnFormatted = this.formatDate(returnDate);
        const body = { journeyType, depart, departFormatted, returnFormatted, totalSeats };

        axios
            .get(AVAILABLE_JOURNEYS, {
                params: body
            })
            .then(response => {
                if (response.status !== 200) {
                    console.warn(response);
                    toast.error('An error occured fetching journeys');
                    return;
                }
                this.setState({
                    journeys: JSON.parse(response.data),
                    loading: false
                });
            })
            .catch(error => {
                toast.error('An error occured fetching journeys');
                console.error('Fetch Journeys: ', error);
                this.setState({ loading: false });
            });
    }

    formatDate = date => {
        return moment(date).format('DD-MM-YYYY');
    };

    journeyList = (journeys, direction) => {
        const { outbound, inbound, departureDate, returnDate, journeyType } = this.props;
        const now = moment();
        const inFuture = this.state.showReturn
            ? moment(returnDate).isAfter(now, 'day')
            : moment(departureDate).isAfter(now, 'day');
        let toastMessage;
        if (this.state.showReturn) {
            toastMessage = 'Inbound journey has been added to your booking.';
        } else if (journeyType === 'RETURN') {
            toastMessage =
                'Outbound journey has been added to your booking. Please select the return journey';
        } else {
            toastMessage = 'Outbound journey has been added to your booking.';
        }

        let items = [];
        const comparison = direction === 'outbound' ? outbound : inbound;
        for (let [key, value] of Object.entries(journeys)) {
            const selected = comparison && value.id === comparison.id;
            const departure = moment(value.departure_time, 'H:m');
            // Skip if the departure date is today, and the time has past
            if (inFuture || (!inFuture && departure >= now)) {
                items.push(
                    <AnimatedDiv className="journey" key={key}>
                        <h2 className="time">
                            {value.departure_time} - {value.arrival_time}
                        </h2>
                        <hr />
                        <div className="internal">
                            <div className="info">
                                <p className="duration">1hr 20mins Direct</p>
                                <p>{this.headingText()}</p>
                            </div>
                            <button
                                className="dark"
                                disabled={selected}
                                onClick={() =>
                                    this.selectJourneyHandler(value, key, direction, toastMessage)
                                }>
                                {selected ? `Added` : `Add Journey`}
                            </button>
                        </div>
                    </AnimatedDiv>
                );
            }
        }
        return items;
    };

    selectJourneyHandler = (value, key, direction, toastMessage) => {
        this.props.addJourney({ ...value, slug: key }, direction);
        if (this.state.journeys.inbound && !this.state.showReturn) {
            confirmAlert({
                title: 'Outbound Journey Added',
                message: 'Please now select your inbound journey',
                buttons: [
                    {
                        label: 'Continue'
                    }
                ]
            });
            this.setState({ showReturn: true });
        } else {
            toast.success(toastMessage);
        }

        window.scrollTo({ top: 0, behavior: 'smooth' });
    };

    headingText = () => {
        const direction =
            this.props.depart === 'DUNDEE' && !this.state.showReturn
                ? 'Dundee Union Street to Edinburgh Airport'
                : 'Edinburgh Airport to Dundee Union Street';
        return direction;
    };

    render() {
        return (
            <div className="choose-journey">
                <h1>
                    {this.headingText()} <br />
                    {this.state.showReturn ? '(Inbound Journey)' : '(Outbound)'}
                </h1>
                {this.state.loading && (
                    <Loader type="Oval" width={50} height={50} color="#088584" />
                )}
                {!this.state.loading && this.state.journeys === null && <NoResults />}

                {this.state.journeys && (
                    <Container>
                        {this.state.journeys.outbound && !this.state.showReturn && (
                            <>{this.journeyList(this.state.journeys.outbound, 'outbound')}</>
                        )}
                        {this.state.journeys.inbound && this.state.showReturn && (
                            <>{this.journeyList(this.state.journeys.inbound, 'inbound')}</>
                        )}
                    </Container>
                )}
            </div>
        );
    }
}

const mapStateToProps = state => {
    const { journeyType, depart, departureDate, returnDate, outbound, inbound } = state.cart;
    const totalSeats = getTotalSeatsFromCart(state);
    return {
        journeyType,
        depart,
        departureDate,
        returnDate,
        totalSeats,
        outbound,
        inbound
    };
};

export default connect(
    mapStateToProps,
    { addJourney }
)(ChooseJourney);
