import React, { useState } from 'react';
import { connect } from 'react-redux';
import { addTicketToCart, removeTicketFromCart } from '../../redux/actions';
import { getTotalSeatsFromCart } from '../../redux/selectors';
import Loader from 'react-loader-spinner';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';

const TicketList = ({
    tickets,
    totalPassengers,
    inCart,
    addTicketToCart,
    removeTicketFromCart
}) => {
    const [opened, setOpened] = useState(false);
    const openClass = opened ? 'open' : '';

    return (
        <div className="ticket-list">
            <div className="ticket-input" onClick={() => setOpened(!opened)}>
                {totalPassengers} Passengers
                <button onClick={() => setOpened(!opened)}>
                    <FontAwesomeIcon
                        icon={['far', 'chevron-down']}
                        rotation={opened ? 180 : null}
                    />
                </button>
            </div>
            <div className={`ticket-window ${openClass}`}>
                <div className="container">
                    <h2>Passengers</h2>
                    {tickets.length === 0 ? (
                        <Loader type="Oval" width={50} height={50} color="#088584" />
                    ) : (
                        tickets.map(ticket => {
                            const { id, title } = ticket;
                            const count = inCart[id] ? inCart[id].quantity : 0;
                            return (
                                <div className="ticket" key={id}>
                                    <p className="title">{title}</p>
                                    <button onClick={() => removeTicketFromCart(ticket)}>
                                        <FontAwesomeIcon icon={['fas', 'minus-circle']} />
                                    </button>
                                    <span>{count}</span>
                                    <button onClick={() => addTicketToCart(ticket)}>
                                        <FontAwesomeIcon icon={['fas', 'plus-circle']} />
                                    </button>
                                </div>
                            );
                        })
                    )}

                    <div className="info">
                        <b>Important information</b>
                        <ol>
                            <li>
                                For wheelchair - number to call 01382 340006/7 and requires 48 hours
                                advance notice
                            </li>
                            <li>For 7+, visit Travel Centre and call 01382 340006/7</li>
                            <li>
                                A family ticket covers two adults with two children or one adult
                                with three children
                            </li>
                        </ol>
                    </div>
                    <button className="primary" onClick={() => setOpened(false)}>
                        Done
                    </button>
                </div>
            </div>
        </div>
    );
};

const mapStateToProps = state => {
    return {
        tickets: state.tickets.allTickets,
        inCart: state.cart.inCart,
        totalPassengers: getTotalSeatsFromCart(state)
    };
};

export default connect(
    mapStateToProps,
    { addTicketToCart, removeTicketFromCart }
)(TicketList);
