<?php

namespace XploreBooking;

class XploreBookingPlugin
{

    private $pluginDirectoryPath;
    private $pluginDirectoryURL;

    private $journeyController;
    private $orderController;
    private $ticketController;

    public function __construct($pluginDirectoryPath, $pluginDirectoryURL)
    {
        $this->pluginDirectoryPath = $pluginDirectoryPath;
        $this->pluginDirectoryURL = $pluginDirectoryURL;

        $this->setupControllers();
        $this->registerShortCodes();
        $this->registerActions();
        $this->registerFilters();
        $this->registerPostType();
    }

    private function setupControllers()
    {
        $this->journeyController = new Controller\JourneyController();
        $this->bookingController = new Controller\BookingController();
        $this->ticketController = new Controller\TicketController();
    }

    private function registerShortCodes()
    {

    }

    private function registerActions()
    {
        add_action('wp_enqueue_scripts', [$this, 'enqueueAssets']);
    }

    private function registerFilters()
    {
        add_filter('acf/settings/save_json', [$this, 'acfJsonSaveLocation']);
        add_filter('acf/settings/load_json', [$this, 'acfJsonLoadLocation']);
    }

    public function addJquery()
    {
        wp_deregister_script('jquery');
        wp_enqueue_script('jquery', 'https://ajax.googleapis.com/ajax/libs/jquery/3.1.1/jquery.min.js', [], null, true);
    }

    public function pluginActivation()
    {
        $this->createTickets();
    }

    private function createTickets()
    {
        $pageDefinitions = [
            'adults' => [
                'post_title' => 'Adults',
                'single_cost' => 16,
                'return_cost' => 22,
                'advanced_single' => 12,
                'advanced_return' => 18,
                'seats' => 1
            ],
            'child' => [
                'post_title' => 'Child',
                'single_cost' => 8,
                'return_cost' => 12,
                'advanced_single' => 6,
                'advanced_return' => 10,
                'seats' => 1
            ],
            'family' => [
                'post_title' => 'Family',
                'single_cost' => 40,
                'return_cost' => 55,
                'advanced_single' => 30,
                'advanced_return' => 45,
                'seats' => 4
            ],
            'student' => [
                'post_title' => 'Student',
                'single_cost' => 12,
                'return_cost' => 18,
                'advanced_single' => 10,
                'advanced_return' => 16,
                'seats' => 1
            ],
            'concession' => [
                'post_title' => 'Concession',
                'v' => 1,
                'return_cost' => 1,
                'advanced_single' => 1,
                'advanced_return' => 1,
                'seats' => 1
            ],
            'group-of-4' => [
                'post_title' => 'Group of 4',
                'single_cost' => 50,
                'return_cost' => 75,
                'advanced_single' => 35,
                'advanced_return' => 60,
                'seats' => 4
            ],
            'group-of-5' => [
                'post_title' => 'Group of 5',
                'single_cost' => 55,
                'return_cost' => 85,
                'advanced_single' => 45,
                'advanced_return' => 70,
                'seats' => 5
            ],
            'group-of-6' => [
                'post_title' => 'Group of 6',
                'single_cost' => 65,
                'return_cost' => 99,
                'advanced_single' => 50,
                'advanced_return' => 80,
                'seats' => 6
            ],
        ];

        foreach ($pageDefinitions as $slug => $pageArgs) {
            // Check that the page doesn't exist already
            $query = new \WP_Query('pagename=' . $slug);
            if (!$query->have_posts()) {
                // Add the page using the data from the array above
                $defaultArgs = [
                    'post_name' => $slug,
                    'post_status' => 'publish',
                    'post_type' => 'ticket',
                    'ping_status' => 'closed',
                    'comment_status' => 'closed',
                ];
                $args = array_merge($pageArgs, $defaultArgs);
                $postID = wp_insert_post($args);
                update_field('single_cost', $pageArgs['single_cost'], $postID);
                update_field('return_cost', $pageArgs['return_cost'], $postID);
                update_field('advanced_single', $pageArgs['advanced_single'], $postID);
                update_field('advanced_return', $pageArgs['advanced_return'], $postID);
                update_field('seats', $pageArgs['seats'], $postID);
            }
        }
    }

    public function enqueueAssets()
    {
        wp_enqueue_script('xplore-booking', $this->pluginDirectoryURL . '/dist/main.js', [], '1', true);
        wp_enqueue_style('xplore-booking', $this->pluginDirectoryURL . '/dist/main.css', [], '1');
        if (is_page(['click-collect-dashboard', 'click-collect-location', 'click-collect-login'])) {
            wp_enqueue_style('click-collect-css', $this->pluginDirectoryURL . '/assets/styles/main.css', [], '1', false);
        }
    }

    function clickCollectTemplates($page_template)
    {
        //TODO: REPLACE WITH CORRECT
//        if (is_page(['click-collect-login', 'click-collect-location', 'click-collect-dashboard'])) {
//            $page_template = $this->pluginDirectoryPath . '/templates/template-click-collect.php';
//        }
//        return $page_template;
    }

    public function checkValidUserType()
    {
        return !empty(array_intersect(['administrator'], $this->getUserRoles()));
    }

    public function getUserRoles()
    {
        $roles = '';
        if (is_user_logged_in()) {
            $user = wp_get_current_user();
            $roles = $user->roles;
        }
        return $roles;
    }

    public function registerPostType()
    {
        Post\Journey::registerPostType();
        Post\Booking::registerPostType();
        Post\Ticket::registerPostType();
    }

    public function acfJsonSaveLocation()
    {
        $path = get_stylesheet_directory() . '/acf-json';
        return $path;
    }

    public function acfJsonLoadLocation($paths)
    {
        unset($paths[0]);
        $paths[] = get_stylesheet_directory() . '/acf-json';
        return $paths;
    }

}
