<?php
/**
 * WooCommerce Local Pickup Plus
 *
 * This source file is subject to the GNU General Public License v3.0
 * that is bundled with this package in the file license.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@skyverge.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade WooCommerce Local Pickup Plus to newer
 * versions in the future. If you wish to customize WooCommerce Local Pickup Plus for your
 * needs please refer to http://docs.woocommerce.com/document/local-pickup-plus/
 *
 * @author      SkyVerge
 * @copyright   Copyright (c) 2012-2020, SkyVerge, Inc.
 * @license     http://www.gnu.org/licenses/gpl-3.0.html GNU General Public License v3.0
 */

namespace SkyVerge\WooCommerce\Local_Pickup_Plus\Fields;

defined( 'ABSPATH' ) or exit;

use SkyVerge\WooCommerce\Local_Pickup_Plus\Data_Store\Package_Pickup_Data;
use SkyVerge\WooCommerce\PluginFramework\v5_5_0 as Framework;

/**
 * Field component to attach pickup data for items to be picked up at checkout.
 *
 * @since 2.0.0
 */
class Package_Pickup_Location_Field extends Pickup_Location_Field {


    /** @var int|string key index of current package this field is associated to */
    private $package_id;


    /**
     * Field constructor.
     *
     * @since 2.0.0
     *
     * @param int|string $package_id the package key index
     */
    public function __construct( $package_id ) {

        $this->object_type = 'package';
        $this->package_id  = $package_id;
        $this->data_store  = new Package_Pickup_Data( $package_id );
    }


    /**
     * Get the ID of the package for this field.
     *
     * @since 2.0.0
     *
     * @return int|string
     */
    private function get_package_id() {
        return $this->package_id;
    }


    /**
     * Get the current package for this field.
     *
     * TODO remove this method by version 3.0.0 or November 2020 {DM 2019-11-22}
     *
     * @since 2.0.0
     * @deprecated since 2.7.0
     *
     * @return array associative array
     */
    public function get_package() {

        wc_deprecated_function( __METHOD__, '2.7.0', '\SkyVerge\WooCommerce\Local_Pickup_Plus\Data_Store\Package_Pickup_Data::get_package' );

        return $this->get_data_store()->get_package();
    }


    /**
     * Returns the current product object, if there is only a single one in package.
     *
     * @since 2.2.0
     *
     * @return \WC_Product|null
     */
    private function get_single_product() {

        $product = null;
        $package = $this->data_store->get_package();

        if ( ! empty( $package['contents'] ) && 1 === count( $package['contents'] ) ) {
            $content = current( $package['contents'] );
            $product = isset( $content['data'] ) && $content['data'] instanceof \WC_Product ? $content['data'] : null;
        }

        return $product;
    }


    /**
     * Gets the pickup location select HTML.
     *
     * @since 2.1.0
     *
     * @return string
     */
    protected function get_pickup_location_html() {

        $shipping_method = wc_local_pickup_plus_shipping_method();
        $chosen_location = $this->data_store->get_pickup_location();

        ob_start(); ?>

        <?php if ( $shipping_method->is_per_order_selection_enabled() ) : ?>

            <?php echo $this->get_location_select_html( $this->get_package_id(), $chosen_location, $this->get_single_product() ); ?>

        <?php elseif ( $chosen_location ) : ?>

            <?php // record the chosen pickup location ID ?>

            <input
                    type="hidden"
                    name="_shipping_method_pickup_location_id[<?php echo esc_attr( $this->get_package_id() ); ?>]"
                    value="<?php echo esc_attr( $chosen_location->get_id() ); ?>"
                    data-package-id="<?php echo esc_attr( $this->get_package_id() ); ?>"
            />

        <?php endif; ?>

        <?php if ( $chosen_location ) : ?>

            <?php // display pickup location name, address & description ?>

            <div class="pickup-location-address">

                <?php if ( is_cart() && $shipping_method->is_per_item_selection_enabled() ) : ?>
                    <?php /* translators: Placeholder: %s - the name of the pickup location */
                    echo sprintf( __( 'Pickup Location: %s', 'woocommerce-shipping-local-pickup-plus' ), esc_html( $chosen_location->get_name() ) ) . '<br />'; ?>
                <?php endif; ?>

                <?php $address = $chosen_location->get_address()->get_formatted_html( true ); ?>
                <?php echo ! empty( $address ) ? wp_kses_post( $address . '<br />' ) : ''; ?>
                <?php $description = $chosen_location->get_description(); ?>
                <?php echo ! empty( $description ) ? wp_kses_post( $description . '<br />' ) : ''; ?>
            </div>

        <?php elseif ( is_checkout() ) : ?>

            <?php // the customer has previously selected items for pickup without specifying a location ?>

            <em><?php esc_html_e( 'Please choose a pickup location', 'woocommerce-shipping-local-pickup-plus' ); ?></em>

        <?php endif; ?>

        <?php

        return ob_get_clean();
    }


    /**
     * Gets the pickup appointment form HTML.
     *
     * TODO remove this method by version 3.0.0 or November 2020 {DM 2019-11-22}
     *
     * @since 2.1.0
     * @deprecated since 2.7.0
     *
     * @return string
     */
    protected function get_pickup_appointments_html() {

        wc_deprecated_function( __METHOD__, '2.7.0', 'Package_Pickup_Appointment_Field::get_html' );

        return '';
    }


    /**
     * Returns the default handling toggle.
     *
     * This is available on both cart totals and checkout review if "per order" pickup mode and automatic grouping are enabled in settings.
     *
     * TODO remove this method by version 3.0.0 or November 2020 {DM 2019-11-22}
     *
     * @since 2.2.0
     * @deprecated since 2.7.0
     *
     * @return string HTML
     */
    public function get_package_handling_toggle_html()
    {

        $toggle            = '';
        $local_pickup_plus = wc_local_pickup_plus_shipping_method();

        $package          = $this->get_package();
        $package_contents = isset($package['contents']) && is_array($package['contents']) ? $package['contents'] : array();
        $default_handling = $local_pickup_plus->get_default_handling();
        $total_items      = 0;
        $can_be_picked_up = 0;
        $can_be_shipped   = 0;


        foreach ($package_contents as $item) {

            $product  = isset($item['data'])     ? $item['data']           : null;
            $quantity = isset($item['quantity']) ? (int) $item['quantity'] : 1;

            if ($product instanceof \WC_Product) {

                if (wc_local_pickup_plus_product_can_be_picked_up($product)) {

                    $can_be_picked_up += $quantity;

                    if (!wc_local_pickup_plus_product_must_be_picked_up($product)) {
                        $can_be_shipped += $quantity;
                    }
                } elseif ($product->needs_shipping() && !wc_local_pickup_plus_product_must_be_picked_up($product)) {

                    $can_be_shipped += $quantity;
                }

                $total_items += $quantity;
            }
        }

        // only show if toggling is a logical possibility
        if (($can_be_picked_up > 0 && 'ship'   === $default_handling )
            || ($can_be_shipped   > 0 && 'pickup' === $default_handling)
        ) {

            // do not show the toggle if toggling from pickup to shipping but there are no available shipping methods
            if (
                isset($package['rates'])
                &&   1 === count($package['rates'])
                &&   $local_pickup_plus->get_method_id() === key($package['rates'])
                && !wc_local_pickup_plus()->get_packages_instance()->package_can_be_shipped($package)
            ) {

                $show_toggle      = true;
                $default_handling = 'pickup';

                wc_local_pickup_plus()->get_session_instance()->set_default_handling('pickup');
            }

            if ($show_toggle) {

                $ship_visibility   = 'pickup' === $default_handling ? 'style="display: none;"' : '';
                $ship_info         = _n('Free delivery over £30 (or £2.50 otherwise)<br>This order will be delivered. <br> Click <a href="https://thescottishbutcher.com/delivery/">here</a> to see where we deliver and when.', 'Free delivery over £30 (or £2.50 otherwise)<br>This order will be delivered. <br> Click <a href="https://thescottishbutcher.com/delivery/">here</a> to see where we deliver and when.', $total_items, 'woocommerce-shipping-local-pickup-plus');
                $ship_label        = _n('Click if you want to pickup this order', 'Click if you want to pickup this order', $total_items, 'woocommerce-shipping-local-pickup-plus');
                $ship_toggle       = '<a href="#" class="toggle-default-handling pickup">' . $ship_label . '.</a>';

                $pickup_visibility = 'ship' === $default_handling ? 'style="display: none;"' : '';
                $pickup_info       = _n('This item is for pickup.', 'These items are for pick up.', $total_items, 'woocommerce-shipping-local-pickup-plus');
                $pickup_label      = _n('Click here if you would like your order delivered', 'Click here if you would like your order delivered', $total_items, 'woocommerce-shipping-local-pickup-plus');
                $pickup_toggle     = '<a href="#" class="toggle-default-handling ship">' . $pickup_label . '.</a>';


                ?>
                <p id="wc-local-pickup-plus-toggle-default-handling">
						<span <?php echo $pickup_visibility; ?>><?php
                            printf('%1$s <br /> %2$s', $pickup_info, $pickup_toggle); ?></span>
                    <span <?php echo $ship_visibility; ?>><?php
                        printf('%1$s <br /> %2$s', $ship_info, $ship_toggle); ?></span>
                    <?php echo is_cart() ? '<br />' : ''; ?>
                </p>
                <?php

            }
        }


    }


    /**
     * Get the field HTML.
     *
     * @since 2.0.0
     *
     * @return string HTML
     */
    public function get_html() {

        $field_html      = '';
        $shipping_method = wc_local_pickup_plus_shipping_method();

        ob_start();

        ?>
        <div
                id="pickup-location-field-for-<?php echo esc_attr( $this->get_package_id() ); ?>"
                class="pickup-location-field pickup-location-field-<?php echo sanitize_html_class( $shipping_method->pickup_selection_mode() ); ?> pickup-location-<?php echo sanitize_html_class( $this->get_object_type() ); ?>-field"
                data-pickup-object-id="<?php echo esc_attr( $this->get_package_id() ); ?>">

            <?php // display the selected location, or location select field ?>
            <?php echo $this->get_pickup_location_html(); ?>

            <?php
            echo $this->get_package_handling_toggle_html(); ?>


        </div>
        <?php

        $field_html .= ob_get_clean();

        /**
         * Filter the package pickup location field HTML.
         *
         * @since 2.0.0
         *
         * @param string $field_html input field HTML
         * @param int|string $package_id the current package identifier
         * @param array $package the current package array
         */
        return apply_filters( 'wc_local_pickup_plus_get_pickup_location_package_field_html', $field_html, $this->get_package_id(), $this->data_store->get_package() );
    }


    /**
     * Gets the pickup location data.
     *
     * TODO remove this method by version 3.0.0 or November 2020 {DM 2019-11-22}
     *
     * @since 2.1.0
     * @deprecated since 2.7.0
     *
     * @param string $piece specific data to get. Defaults to getting all available data.
     * @return array|string
     */
    protected function get_pickup_data( $piece = '' ) {

        wc_deprecated_function( __METHOD__, '2.7.0', '\SkyVerge\WooCommerce\Local_Pickup_Plus\Data_Store\Package_Pickup_Data::get_pickup_data' );

        return $this->get_data_store()->get_pickup_data( $piece );
    }


    /**
     * Sets the pickup location data.
     *
     * TODO remove this method by version 3.0.0 or November 2020 {DM 2019-11-22}
     *
     * @since 2.1.0
     * @deprecated since 2.7.0
     *
     * @param array $pickup_data pickup data
     */
    protected function set_pickup_data( array $pickup_data ) {

        wc_deprecated_function( __METHOD__, '2.7.0', '\SkyVerge\WooCommerce\Local_Pickup_Plus\Data_Store\Package_Pickup_Data::set_pickup_data' );

        return $this->get_data_store()->set_pickup_data( $pickup_data );
    }


    /**
     * Deletes the pickup location data.
     *
     * TODO remove this method by version 3.0.0 or November 2020 {DM 2019-11-22}
     *
     * @since 2.1.0
     * @deprecated since 2.7.0
     */
    protected function delete_pickup_data() {

        wc_deprecated_function( __METHOD__, '2.7.0', '\SkyVerge\WooCommerce\Local_Pickup_Plus\Data_Store\Package_Pickup_Data::delete_pickup_data' );

        return $this->get_data_store()->delete_pickup_data();
    }


}

class_alias( Package_Pickup_Location_Field::class, 'WC_Local_Pickup_Plus_Pickup_Location_Package_Field' );
