const webpack = require('webpack');
// This is used to clean the dist folder at the start of the build
const CleanWebpackPlugin = require('clean-webpack-plugin');
// This is used to copy image files from source to dist)
const CopyWebpackPlugin = require('copy-webpack-plugin');
// This is used to extract css into seperate files
const MiniCssExtractPlugin = require('mini-css-extract-plugin');
// This is used to auto-reload browsers
const BrowserSyncPlugin = require('browser-sync-webpack-plugin');

module.exports = () => {
    // Files to read as entry point
    const entry = ['./assets/scripts/app.js', './assets/styles/main.scss'];
    // The output build folder
    const distPath = __dirname + '/../dist'; // eslint-disable-line
    // Url that browser sync should be replicating
    const devUrl = 'virosafe-solutions.test';

    const themeName = 'virosafe-solutions';

    return {
        entry,
        output: {
            path: distPath,
            publicPath: `/content/themes/${themeName}/dist/`, // Replace with theme name
            filename: 'site.js',
            chunkFilename: 'scripts/[name].js',
        },
        module: {
            rules: [
                {
                    test: /\.js$/,
                    exclude: /node_modules\/(?!(dom7|ssr-window|swiper)\/).*/,
                    use: {
                        loader: 'babel-loader', // Transpiles JavaScript
                    },
                },
                {
                    test: /\.(jpe?g|png|gif|svg)$/,
                    use: [
                        {
                            loader: 'file-loader',
                            options: {
                                name: 'images/[name].[ext]'
                            }
                        }
                    ],
                },
                {
                    test: /\.(sa|sc|c)ss$/,
                    use: [
                        MiniCssExtractPlugin.loader, // Extracts css into a separate file
                        {
                            loader: 'css-loader', // translates CSS into CommonJS modules
                            options: {
                                sourceMap: true,
                            },
                        },
                        'resolve-url-loader', // Rewrites relative paths to match the path in the output folder
                        {
                            loader: 'postcss-loader', // Run post css actions
                            options: {
                                sourceMap: true,
                                plugins: [
                                    require('autoprefixer')({
                                        grid: true,
                                        browsers: ['last 2 versions', 'ie 6-8', 'Firefox > 20']
                                    })
                                ],
                            },
                        },
                        {
                            loader: 'sass-loader', // compiles SASS to CSS
                            options: {
                                sourceMap: true,
                            },
                        },
                    ],
                },
                {
                    test: /\.(ttf|eot|woff2?)$/,
                    use: {
                        loader: 'file-loader',
                        options: {
                            name: '[name].[ext]',
                            outputPath: 'fonts/',
                            publicPath: 'fonts/',
                        },
                    },
                },
            ],
        },
        externals: {
            jquery: 'jQuery',
        },
        plugins: [
            // Cleans the dist path
            new CleanWebpackPlugin([distPath], {
                verbose: true,
                allowExternal: true,
            }),
            // Extracts css into file
            new MiniCssExtractPlugin({
                filename: 'site.css',
                chunkFilename: 'styles/[chunkhash].css',
            }),
            // Copies images to dist folder
            new CopyWebpackPlugin([
                {
                    from: 'assets/images/',
                    to: 'images',
                    ignore: ['*.gitignore'],
                },
            ]),
            // Automatically provide modules without having to import in src files
            new webpack.ProvidePlugin({
                $: 'jquery',
                jQuery: 'jquery',
                'window.jQuery': 'jquery',
                Popper: ['popper.js', 'default'],
            }),
            // Start a browsersync session
            new BrowserSyncPlugin({
                proxy: devUrl,
            }),
        ],
    };
};
