<?php

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across the
 * admin area of the site.
 *
 * @link       https://blue2.co.uk
 * @since      1.0.0
 *
 * @package    Blue2_Monitor
 * @subpackage Blue2_Monitor/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization and admin-specific hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Blue2_Monitor
 * @subpackage Blue2_Monitor/includes
 * @author     Blue2 Digital Ltd <ross@blue2.co.uk>
 */
class Blue2_Monitor
{
	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      Blue2_Monitor_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $plugin_name    The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;

	/**
	 * The URL of the Site Monitor API.
	 *
	 * @since 1.0.0
	 * @access protected
	 * @var string
	 */
	protected $api_url;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area of 
	 * the site.
	 *
	 * @since    1.0.0
	 */
	public function __construct()
	{
		if (defined('BLUE2_MONITOR_VERSION')) {
			$this->version = BLUE2_MONITOR_VERSION;
		} else {
			$this->version = '1.0.0';
		}

		if (defined('BLUE2_MONITOR_API_URL')) {
			$this->api_url = BLUE2_MONITOR_API_URL;
		} else {
			$this->api_url = 'https://site-monitor.blue2.co.uk/api/';
		}

		$this->plugin_name = 'blue2-monitor';

		$this->load_dependencies();
		$this->define_admin_hooks();
	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * Include the following files that make up the plugin:
	 *
	 * - Blue2_Monitor_Loader. Orchestrates the hooks of the plugin.
	 * - Blue2_Monitor_i18n. Defines internationalization functionality.
	 * - Blue2_Monitor_Admin. Defines all hooks for the admin area.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function load_dependencies()
	{
		/**
		 * The class responsible for orchestrating the actions and filters of the
		 * core plugin.
		 */
		require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-blue2-monitor-loader.php';

		/**
		 * The class responsible for defining all actions that occur in the admin area.
		 */
		require_once plugin_dir_path(dirname(__FILE__)) . 'admin/class-blue2-monitor-admin.php';

		$this->loader = new Blue2_Monitor_Loader();
	}

	/**
	 * Register all of the hooks related to the admin area functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_admin_hooks()
	{
		$plugin_admin = new Blue2_Monitor_Admin($this->get_plugin_name(), $this->get_version());

		// Add settings link to plugin page
		$plugin_basename = plugin_basename(plugin_dir_path(dirname(__FILE__)) . $this->plugin_name . '.php');
		$this->loader->add_filter('plugin_action_links_' . $plugin_basename, $plugin_admin, 'add_action_links');

		// Enqueue styles and scripts
		$this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'enqueue_styles');
		$this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts');

		// Add plugin to settings menu
		$this->loader->add_action('admin_menu', $plugin_admin, 'settings_menu');

		// AJAX call to manually ping the server
		$this->loader->add_action('wp_ajax_blue2_monitor_ping_server', $this, 'manually_ping_server');

		// Hook to ping the server
		$this->loader->add_action('blue2_monitor_ping_server', $this, 'ping_server');

		// Hook to save the API Token
		$this->loader->add_action('admin_post_b2_monitor_save_api_token', $this, 'save_api_token');

		// Hook to start a session when viewing the plugin's settings page
		$this->loader->add_action('admin_init', $this, 'start_session_when_viewing_plugin_page');
	}

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since    1.0.0
	 */
	public function run()
	{
		$this->loader->run();
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     1.0.0
	 * @return    string    The name of the plugin.
	 */
	public function get_plugin_name(): string
	{
		return $this->plugin_name;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @since     1.0.0
	 * @return    Blue2_Monitor_Loader    Orchestrates the hooks of the plugin.
	 */
	public function get_loader(): Blue2_Monitor_Loader
	{
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     1.0.0
	 * @return    string    The version number of the plugin.
	 */
	public function get_version(): string
	{
		return $this->version;
	}

	/**
	 * Manually ping the server which is activated by an AJAX call.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	public function manually_ping_server(): void
	{
		try {
			$response = $this->ping_server();
		} catch (Exception $e) {
			wp_send_json_error($e->getMessage());

			die();
		}

		if ($response) {
			wp_send_json_success($response);
		} else {
			wp_send_json_error('No API Token set');
		}

		die();
	}

	/**
	 * Ping the server with the current PHP and WordPress versions.
	 * Also send a list of plugins and their versions.
	 *
	 * @since 1.0.0
	 */
	public function ping_server()
	{
		if (!get_option('b2_monitor_api_token')) {
			throw new Exception('No API Token set');
		}

		global $wp_version;

		$updatePlugins = get_site_transient('update_plugins');
		$updatePluginsSorted = [];

		if (!empty($updatePlugins->response)) {
			foreach ($updatePlugins->response as $plugin) {
				$updatePluginsSorted[] = $plugin->slug;
			}
		}

		$plugins = [];
		$installedPlugins = get_option('active_plugins');

		foreach (get_plugins() as $key => $plugin) {
			if (in_array($key, $installedPlugins)) {
				$plugins[] = [
					'name' => $plugin['Name'],
					'slug' => $plugin['TextDomain'],
					'version' => $plugin['Version'],
					'update_available' => in_array($plugin['TextDomain'], $updatePluginsSorted)
				];
			}
		}

		$php_version = phpversion();

		$payloadData = [
			'php_version' => $php_version,
			'wordpress_version' => $wp_version,
			'plugins' => $plugins
		];

		return $this->call_api('phone-home', get_option('b2_monitor_api_token'), $payloadData);
	}

	/**
	 * Hook called when the API Token is saved.
	 * This will validate the token and save it to the database.
	 * It will also schedule the daily ping to the server.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	public function save_api_token()
	{
		if (!current_user_can('manage_options')) {
			wp_die('Unauthorized user');
		}

		if (!wp_verify_nonce($_POST['_wpnonce'], 'b2_monitor_save_api_token')) {
			wp_die('Nonce verification failed');
		}

		if (!isset($_POST['api_token'])) {
			wp_die('No API Token provided');
		}

		$apiToken = sanitize_text_field($_POST['api_token']);

		try {
			$this->call_api('validate-token', $apiToken, null);
		} catch (Exception $e) {
			session_start();
			$_SESSION['b2_monitor_response'] = [
				'success' => false,
				'message' => 'Invalid API Token!'
			];

			wp_redirect(admin_url('options-general.php?page=blue2-monitor'));
			exit;
		}

		update_option('b2_monitor_api_token', $apiToken);

		if (!wp_next_scheduled('blue2_monitor_ping_server')) {
			$tomorrow = strtotime('tomorrow 01:00');
			wp_schedule_event($tomorrow, 'daily', 'blue2_monitor_ping_server');
		}

		session_start();
		$_SESSION['b2_monitor_response'] = [
			'success' => true,
			'message' => 'API Token saved successfully!'
		];

		wp_redirect(admin_url('options-general.php?page=blue2-monitor'));
		exit;
	}

	/**
	 * Start a session when viewing the plugin page.
	 * This is so we can display a message after saving the API Token.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	public function start_session_when_viewing_plugin_page(): void
	{
		if (isset($_GET['page']) && $_GET['page'] === 'blue2-monitor') {
			session_start();
		}
	}

	/**
	 * Centralised function for calling the Site Monitor API.
	 *
	 * @since 1.0.0
	 * @access private
	 * @param string $url
	 * @param string $apiToken
	 * @param array|null $payload
	 * @return void
	 */
	private function call_api(string $url, string $apiToken, ?array $payload)
	{
		if (!in_array($url, ['phone-home', 'validate-token'])) {
			throw new Exception('Invalid API URL');
		}

		$ch = curl_init();

		curl_setopt($ch, CURLOPT_URL, $this->api_url . $url);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
		curl_setopt($ch, CURLOPT_POST, 1);

		if ($payload) {
			curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
		}

		curl_setopt($ch, CURLOPT_HTTPHEADER, [
			'Accept: application/json',
			'Content-Type: application/json',
			'Authorization: Bearer ' . $apiToken
		]);

		$result = curl_exec($ch);
		if (curl_errno($ch)) {
			$response = curl_error($ch);
		} else {
			$response = $result;
		}

		$responseCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

		curl_close($ch);

		if ($responseCode === 401) {
			throw new Exception('API Unauthenticated');
		}

		return $response;
	}
}
