/* global ajaxObject */

class CourseFilter {
    pageCount = 1;

    constructor() {
        this.activeFilters = {};
        this.isTagArchive = false;
        this.tag = '';
        this.checkPreviousURL();
        this.attachOnChangeEvents();
        this.onKeyPressEvents();
        this.removeActiveFilterButton();
        this.clearAllFilters();
        this.createClearAllFilterButton();
        this.setActiveFilterOptionsOnLoad();
        this.checkPagination();
        const nextPageButton = document.getElementById('next-page-button');
        if (nextPageButton !== null) {
            nextPageButton.onclick = this.nextPage;
        }
        const prevPageButton = document.getElementById('prev-page-button');
        if (prevPageButton !== null) {
            prevPageButton.onclick = this.prevPage;
        }

        if ($('body').hasClass('tag')) {
            this.isTagArchive = true;
            this.tag = $('.custom-tag-archive').data('tag');
        }

        window.course = this;
        console.log(this.activeFilters);
    }

    setActiveFilterOptionsOnLoad() {
        let i;
        const getParameters = location.search.replace('?', '').split('&');
        if (getParameters[0] === '') { return; };
        const paramValues = getParameters.map(function(val) {
            return val.split('=');
        });

        for (i = 0; i < paramValues.length; ++i) {
            const getName = paramValues[i][0];
            const getValue = paramValues[i][1];
            this.activeFilters[getName] = getValue;
        }
    }

    attachOnChangeEvents() {
        const self = this;
        $('.filter-select').on('change', function() {
            const name = $(this).attr('name');

            $(`#${name}`).remove();
            const val = $(this).val();
            const label = $(this).data('label');
            self.pageCount = 1;
            self.activeFilters[name] = val;
            self.createClearAllFilterButton();
            self.generateActiveFilterButton(val, name, label);
            self.sendFilterUpdates();
            self.checkPagination();
            self.updateURLstring();
        });

        $('.filter-radio').on('change', function() {
            const name = $(this).attr('name');
            const val = $(this).val();
            $('.radio-item').removeClass('checked-radio');
            $(this).parent('.radio-item')
                .addClass('checked-radio');
            self.pageCount = 1;
            self.activeFilters[name] = val;
            self.sendFilterUpdates();
            self.checkPagination();
            self.updateURLstring();
        });
    };

    updateURLstring() {
        const urlArray = [];
        console.log(this.activeFilters);
        $.each(this.activeFilters, function(index, value) {
            urlArray.push(`${index}=${value}`);
        });

        const urlString = `?${urlArray.join('&')}`;

        if ($('.post-type-archive-course').hasClass('paged')) {
            window.location = `/courses/${urlString}`;
        } else if ($('.tag').hasClass('paged')) {
            window.location = `/tag/${this.tag}/${urlString}`;
        } else {
            window.history.replaceState(null, null, urlString);
        }
    }

    onKeyPressEvents() {
        const self = this;
        let timeout = null;
        $('.keyword-search').on('input',function(el) {
            clearTimeout(timeout);
            timeout = setTimeout(function() {
                const name = $(el.target).attr('name');
                const val = $(el.target).val();
                self.activeFilters[name] = val;
                self.sendFilterUpdates();
                self.checkPagination();
                self.updateURLstring();
            }, 500);
        });
    }

    createClearAllFilterButton() {
        if (typeof this.activeFilters !== 'undefined' && this.activeFilters.length > 0) {
            $('.clear-all').addClass('active');
        } else {
            $('.clear-all').removeClass('active');
        }
    }

    generateActiveFilterButton(val, name, label) {
        const filterValue = val;
        const filterName = name;
        const filterLabel = label;
        $('.active-course-filters ul').prepend(`<li class='filter-property' id='${filterName}'>${filterLabel}<div data-value='${filterValue}' data-name='${filterName}' class='clear-filter' aria-label='Clear filter'><i class='fas fa-times'></i></div></li>`);
    }

    removeActiveFilterButton() {
        const self = this;
        $('body').on('click', '.clear-filter', function() {
            const name = $(this).data('name');
            $(`#disabled-${name}`).prop('checked', true);
            $(this).parent('li')
                .remove();
            delete self.activeFilters[name];
            self.sendFilterUpdates();
            self.updateURLstring();
        });
    }

    clearAllFilters() {
        $('.clear-all').on('click', function(e) {
            e.preventDefault();
            $('.filter-property').remove();
            this.activeFilters = [];
            window.location.href = '/courses/';
        });
    }

    prevPage = () => {
        if (this.pageCount > 1) {
            this.pageCount--;
            this.sendFilterUpdates();
            this.checkPagination();
        }
    };

    nextPage = () => {
        this.pageCount++;
        this.sendFilterUpdates();
        this.checkPagination();
    };

    checkPagination() {
        if (this.pageCount === 1) {
            $('#prev-page-button').hide();
        } else if (this.pageCount >= 2) {
            $('#prev-page-button').show();
        }
    }

    sendFilterUpdates() {
        $.ajax({
            type: 'POST',
            url: ajaxObject.ajax_url,
            data: {
                action: 'filter_courses',
                filters: this.activeFilters,
                pages: this.pageCount,
                isTag: this.isTagArchive,
                tag: this.tag,
            },
            beforeSend() {
                $('.course-item-group').hide();
                $('.custom-course-pagination').hide();
                $('.loading').show();
            },
            success(data) {
                document.title = 'Courses Archive - North East Scotland College';
                $('.course-output').empty();
                $('.course-output').html(data);
                const count = $('.course-index-item').length;
                if (count < 10) {
                    $('#next-page-button').hide();
                } else {
                    $('#next-page-button').show();
                }
            },
            complete() {
                $('.loading').hide();
                $('.course-item-group').show();
                $('.custom-course-pagination').show();
            }
        });
    }

    checkPreviousURL() {
        const previousURL = document.referrer;
        if (previousURL.indexOf('courses-by-interest') >= 0) {
            const searchParams = new URLSearchParams(window.location.search);
            const param = searchParams.get('interests');
            const title = param.replace(/-/g, ' ');
            $('.course-archive-custom .title-section h1').text(`${title} Courses`);
        }
    }
}



export default CourseFilter;
