<?php


namespace Blue2\Scruffy\Controllers;


use DateTime;
use Exception;
use WP_Post;


class ImportController
{
    private $status = [
        'un-Published' => 'draft',
        'Published' => 'publish',
    ];

    private $replacements = [
        'Business and Management' => 'Business & Management',
    ];

    public function import($url)
    {
        try {
            $rows = $this->parseCsv($url);
            foreach ($rows as $row) {
                $this->saveCourseData($row);
            }
        } catch (Exception $e) {
            dd($e->getMessage());
        }
    }

    /**
     * @param $url
     * @return array
     * @throws Exception
     */
    private function parseCsv($url)
    {
        $handle = fopen($url, "r");

        $row = 1;
        $courses = [];
        if ($handle !== false) {

            $headings = fgetcsv($handle, 8192, ",");
            $totalColumns = count($headings);

            while (($data = fgetcsv($handle, 8192, ",")) !== false) {
                $row++;
                if (count($data) !== $totalColumns) {
                    throw new Exception('Row ' . $row . ' in CSV does not appear to contain all required data.');
                } else {
                    if ($data[0] !== '') {
                        $courses[] = array_combine($headings, $data);
                    }
                }
            }
            fclose($handle);
        }

        return $courses;
    }

    /**
     * @param $courseData
     * @throws Exception
     */
    private function saveCourseData($courseData)
    {
        $courseTitle = trim($courseData['Computed Title']);
        $course = $this->getCourseByComputedTitle($courseTitle);
        if ($course === null) {
            $course = $this->createCourse($courseData);
        }
        $this->updateCourse($course, $courseData);
    }

    private function getCourseByComputedTitle($courseTitle): ?WP_Post
    {
        $matchingCourses = get_posts([
            'numberposts' => -1,
            'post_type' => 'course',
            'meta_key' => 'computed_title',
            'meta_value' => $courseTitle,
            'post_status' => ['publish', 'draft'],
        ]);

        if (count($matchingCourses) > 1) {
            throw new Exception('Computed Course Title (' . $courseTitle . ') is not unique, cannot determine correct course to update.');
        }

        if (count($matchingCourses) === 0) {
            return null;
        }

        return $matchingCourses[0];
    }

    /**
     * @param $course
     * @return array|\WP_Post|null
     * @throws Exception
     */
    private function createCourse($course)
    {
        $newCourse = [
            'post_type' => 'course',
            'post_title' => $this->clean($course['Title']),
            'post_content' => $this->clean($course['Summary']),
            'post_status' => $this->getPostStatus($course['Status']),
        ];

        $postId = wp_insert_post($newCourse);
        if (is_wp_error($postId)) {
            throw new Exception('An error occurred creating new course (' . $$this->clean($course['Computed Title']) . ').');
        }

        return get_post($postId);
    }

    private function clean($data)
    {
        return wp_strip_all_tags(trim($data));
    }

    private function getPostStatus($courseStatus)
    {
        $courseStatus = trim($courseStatus);
        if (array_key_exists($courseStatus, $this->status)) {
            return $this->status[$courseStatus];
        } else {
            return 'draft';
        }
    }

    private function updateCourse($course, $courseData)
    {
        $this->updateCourseFields($course, $courseData);
        $this->updateCourseTaxonomies($course, $courseData);
        $this->updateCourseInstances($course, $courseData);
        $this->updateCourseRelatedCourses($course, $courseData);
    }

    private function updateCourseFields($course, $courseData)
    {
        $mappedFields = [
            'computed_title' => $courseData['Computed Title'],
            'title' => $courseData['Title'],
            'qualification_type' => $courseData['Qualification'],
            'why_take_this_course' => $courseData['Why take this course?'],
            'what_will_i_experience' => $courseData['What will I experience?'],
            'show_disclaimer' => $this->textToBoolean($courseData['Disclaimer']),
            'education_description' => $this->textToBoolean($courseData['College Information']),
            'university_description' => $this->textToBoolean($courseData['University Information']),
            'employment_opportunities' => $this->textToBoolean($courseData['Careers']),
            'employment_description' => $this->textToBoolean($courseData['Employment prospects']),
        ];

        foreach ($mappedFields as $field => $value) {
            update_field($field, $value, $course->ID);
        }
    }

    private function updateCourseTaxonomies($course, $courseData)
    {
        $mappedMultipleTaxonomies = [
            'mode-of-study' => $courseData['Mode of study'],
            'level' => $this->removeBrackets($courseData['Computed SCQF Level']),
        ];
        $mappedSingularTaxonomies = [
            'category' => $courseData['Subject Area'],
        ];

        foreach ($mappedMultipleTaxonomies as $taxonomy => $value) {
            $terms = $this->getTaxonomyTerms($value, $taxonomy);
            $this->addTermsToCourseTaxonomy($terms, $taxonomy, $course->ID);
        }

        foreach ($mappedSingularTaxonomies as $taxonomy => $value) {
            $terms = $this->getTaxonomyTermsSingular($value, $taxonomy);
            $this->addTermsToCourseTaxonomy($terms, $taxonomy, $course->ID);
        }
    }

    private function removeBrackets($string)
    {
        return str_replace(['(', ')'], '', $string);
    }

    private function getTaxonomyTerms($data, $taxonomy): array
    {
        $terms = [];

        $items = explode(',', $data);
        foreach ($items as $item) {
            $item = trim($item);
            $term = get_term_by('name', $item, $taxonomy, OBJECT);
            if ($term !== false) {
                $terms[] = $term;
            }
        }

        return $terms;
    }

    private function getTaxonomyTermsSingular($item, $taxonomy)
    {
        $terms = [];

        $item = $this->replace(trim($item));
        $term = get_term_by('name', $item, $taxonomy, OBJECT);
        if ($term !== false) {
            $terms[] = $term;

        }

        return $terms;
    }

    private function addTermsToCourseTaxonomy($terms, $taxonomy, $id)
    {
        $termSlugs = [];
        foreach ($terms as $term) {
            $termSlugs[] = $term->slug;
        }
        wp_set_object_terms($id, $termSlugs, $taxonomy);
    }

    private function replace($string)
    {
        foreach ($this->replacements as $search => $replace) {
            $string = str_replace($search, $replace, $string);
        }

        return $string;
    }

    private function updateCourseInstances($course, $courseData)
    {
        $this->removeInstanceFromCourse($courseData['Course Codes'], $course);

        add_row(
            'course_instances',
            [
                'duration' => $courseData['Duration'],
                'code' => $courseData['Course Codes'],
                'location' => $this->getLocation($courseData['Campuses & centres']),
                'fees' => $courseData['Part-time fees (&pound;)'],
                'start_date' => $this->getDate($courseData['Start date']),
                'end_date' => $this->getDate($courseData['End date']),
                'show_on_site' => $this->textToBoolean($courseData['Hide']),
                'online' => $this->textToBoolean($courseData['Online']),
                'places_available' => $this->textToBoolean($courseData['Places still available']),
            ],
            $course->ID
        );
    }

    private function removeInstanceFromCourse($instanceCode, $course)
    {
        $instances = get_field('course_instances', $course->ID);
        if ($instances !== null) {
            foreach ($instances as $index => $instance) {
                if ($instanceCode === $instance['code']) {
                    delete_row('course_instances', $index + 1, $course->ID);
                    break;
                }
            }
        }
    }

    private function getLocation(string $location)
    {
        $location = trim($location);
        $matchingLocations = $this->getPostsByExactTitle($location, 'location');

        if (count($matchingLocations) > 1) {
            throw new Exception('Course Location (' . $location . ') is not unique, cannot determine correct location to use for Course Instance.');
        }

        if (count($matchingLocations) === 1) {
            return $matchingLocations[0];
        }

        return null;
    }

    private function getPostsByExactTitle($pageTitle, $postType = false)
    {
        global $wpdb;

        $postTypeWhere = $postType ? 'AND post_type = %s' : '%s';
        $results = $wpdb->get_results($wpdb->prepare("SELECT * FROM $wpdb->posts WHERE post_title = %s $postTypeWhere AND post_status in ('publish', 'draft')", $pageTitle, $postType ? $postType : ''));

        $output = [];
        if ($results) {
            foreach ($results as $post) {
                $output[] = $post;
            }
        }
        return $output;
    }

    private function getDate(string $date)
    {
        $date = DateTime::createFromFormat('d M Y', $date);

        if ($date === false) {
            return null;
        }

        return $date->format('Y-m-d');
    }

    private function textToBoolean(string $value)
    {
        return (strtolower(trim($value)) === 'yes');
    }

    private function updateCourseRelatedCourses($course, $courseData)
    {
        $relatedCourseTitles = $courseData['Courses'];
        $relatedCourseTitles = array_filter(explode(',', $relatedCourseTitles));
        $currentRelatedCourses = get_field('education_course_opportunities', $course->ID);
        if (!is_array($currentRelatedCourses)) {
            $currentRelatedCourses = [];
        }


        foreach ($relatedCourseTitles as $courseTitle) {
            $relatedCourse = $this->getCourseByComputedTitle(trim($courseTitle));
            if ($relatedCourse !== null) {
                $currentRelatedCourses[] = $relatedCourse;
            }
        }

        $uniqueCourses = [];
        foreach ($currentRelatedCourses as $currentRelatedCourse) {
            $uniqueCourses[$currentRelatedCourse->ID] = $currentRelatedCourse;
        }
        $uniqueCourses = array_values($uniqueCourses);

        update_field('education_course_opportunities', $uniqueCourses, $course->ID);
    }
}
