<?php

namespace Blue2\Scruffy\Controllers;

use Blue2\Scruffy\Site;
use Knp\Snappy\Pdf;
use Blue2\Scruffy\Controllers\UniteController;

class CourseController
{
    public function __construct()
    {
        add_action('rest_api_init', [$this, 'registerEndPoints']);
        $this->registerActions();
        $this->registerFilters();
    }

    private function registerFilters()
    {
        add_filter('posts_where', [$this, 'courseInstanceMetaFieldCheck']);
        add_filter('wpseo_title', [$this, 'customCourseArchiveTitles']);
        add_filter( 'posts_where', [$this, 'wpse_298888_posts_where'], 10, 2 );
    }

    public function registerActions()
    {
        add_action('wp_ajax_filter_courses', [$this, 'ajaxFilter']);
        add_action('wp_ajax_nopriv_filter_courses', [$this, 'ajaxFilter']);
        add_action('wp_ajax_compare_courses', [$this, 'ajaxCompare']);
        add_action('wp_ajax_nopriv_compare_courses', [$this, 'ajaxCompare']);
        add_action('init', [$this, 'loadBrochure']);
        add_action('init', [$this, 'addAuthorToCourses']);
        add_action('wp_dashboard_setup', [$this, 'createCourseExportWidget']);
        add_action('admin_init', [$this, 'exportCourseTriggered']);
        add_action('save_post', [$this,'deleteInstances'], 10,3 );
        add_action('wp_dashboard_setup', [$this, 'registerManualTriggers']);
        add_action('updateInstances', [$this, 'updatePostInstances'], 10);
    }

    function wpse_298888_posts_where( $where, $query ) {
        global $wpdb;

        $starts_with = esc_sql( $query->get( 'starts_with' ) );

        if($starts_with == '1-10') {
            $letters = [1,2,3,4,5,6,7,8,9,10];
            $i = 0;
            foreach($letters as $l) {
                $join = $i == 0 ? 'AND (' : 'OR';
                $where .= " $join $wpdb->posts.post_title LIKE '$l%'";
                $i++;
            }

            $where .= ")";
        }
        else {
            $where .= " AND $wpdb->posts.post_title LIKE '$starts_with%'";
        }

        return $where;
    }


    public function registerEndPoints()
    {
        register_rest_route('nescol/v1', '/courses', [
            'methods' => ['GET', 'POST'],
            'callback' => [$this, 'getAllCourses'],
        ]);
    }

    public function registerManualTriggers()
    {
        wp_add_dashboard_widget('removeInstances', 'Remove course instances with start dates that have passed', [$this, "removeInstancesButton"]);
        wp_add_dashboard_widget('syncPartTime', 'Sync Part Time courses with Unite Data', [$this, "syncPartTimeButton"]);
        wp_add_dashboard_widget('syncFullTime', 'Sync Full Time courses with Unite Data', [$this, "syncFullTImeButton"]);
    }

    public function removeInstancesButton()
    {
        echo '<form method="post" name="remove-instances" id="remove-instances">
                <input type="submit" name="removeInstances" value="Remove Old Instances"></form>';
            
        if (isset($_POST['removeInstances'])) {
            $this->startUpdateInstances();
        }
    }

    public function syncPartTimeButton()
    {
        echo '<form method="post" name="sync-part-time" id="sync-part-time">
                <input type="submit" name="syncPartTime" value="Sync Part Time Courses"></form>';
            
        if (isset($_POST['syncPartTime'])) {
            UniteController::startUnitePartTimeImport();
        }
    }

    public function syncFullTimeButton()
    {
        echo '<form method="post" name="sync-full-time" id="sync-full-time">
                <input type="submit" name="syncFullTime" value="Sync Full Time Courses"></form>';
            
        if (isset($_POST['syncFullTime'])) {
            UniteController::startUniteFullTimeImport();
        }
    }

    public function addAuthorToCourses() {
        add_post_type_support( 'course', 'author' );
    }


    public function ajaxFilter()
    {
        $filters = $_POST['filters'];
        $paged = $_POST['pages'];
        $isTag = $_POST['isTag'];
        $tag = $_POST['tag'];
        $courses = self::getAllCourses($filters, $paged, $isTag, $tag);
        self::outputCourses($courses);
        wp_die();
    }

    public function ajaxCompare()
    {
        $compareCourses = $_POST['courses'];
        session_start();
        $_SESSION['courses'] = $compareCourses;
        self::outputCompareCourses($compareCourses);
        wp_die();
    }

    public function courseInstanceMetaFieldCheck($where)
    {
        $where = str_replace("meta_key = 'course_instances_$", "meta_key LIKE 'course_instances_%", $where);
        return $where;
    }

    public static function getCourseSubjects($id)
    {
        $subjectData = [];
        $subjects = get_the_category($id);
        foreach ($subjects as $subject) {
            array_push($subjectData, $subject->name);
        }
        return $subjectData;
    }

    public static function getCourseLevels($id)
    {
        $levelsData = [];
        $levels = get_the_terms($id, 'level');
        if ($levels) {
            foreach ($levels as $value) {
                array_push($levelsData, $value->name);
            }
        }
        return $levelsData;
    }

    public static function getCourseModesOfStudy($id)
    {
        $modeOfStudyData = [];
        $modesOfStudy = get_the_terms($id, 'mode-of-study');
        if ($modesOfStudy) {
            foreach ($modesOfStudy as $value) {
                array_push($modeOfStudyData, $value->name);
            }
        }
        return $modeOfStudyData;
    }

    public static function getCourseCampuses($id)
    {
        $campusesData = [];
        if (have_rows('course_instances', $id)) {
            while (have_rows('course_instances', $id)) {
                the_row();
                $location = get_sub_field('location');
                $showOnSite = get_sub_field('show_on_site');
                if ($location && $showOnSite) {
                    $title = get_the_title($location);
                    array_push($campusesData, $title);
                }
            }
        }
        sort($campusesData);
        return array_unique($campusesData);
    }

    public static function getCourseInstances($id)
    {
        $instanceStartDates = [];
        $instanceDurations = [];
        $instanceFees = [];

        if (have_rows('course_instances', $id)) {
            while (have_rows('course_instances', $id)) {
                the_row();

                if (get_sub_field('show_on_site') !== false) {
                    if (get_sub_field('start_date') !== '') {
                        $instanceStartDates[] = get_sub_field('start_date');
                    }
                    if (get_sub_field('duration') !== '') {
                        $instanceDurations[] = get_sub_field('duration');
                    }
                    if (get_sub_field('fees') !== '') {
                        $instanceFees[] = get_sub_field('fees');
                    }
                }
            }
        }

        return ['start_dates' => $instanceStartDates, 'duration' => $instanceDurations, 'fees' => $instanceFees];
    }

    public static function getCourseInstanceStartDates($dates)
    {
        $currentDate = date('d M Y');
        $currentDateString = strtotime($currentDate);
        $uniqueInstances = array_filter(array_unique($dates));
        $filteredDates = [];
        foreach($uniqueInstances as $dateInstance) {
            $date = strtotime($dateInstance);
            if($date > $currentDateString) {
                $filteredDates[] = date('d M Y', $date);
            }
        }
        $countInstances = count($filteredDates);
        if($countInstances == 1){
            $startDateOutput = $filteredDates[0];
        }
        else if ($countInstances == 2) {
            sort($filteredDates);
            $startDateOutput = implode('<br/>', $filteredDates);
        } elseif ($countInstances >= 3) {
            $startDateOutput = "Options Available";
        } else {
            $startDateOutput = 'No dates available';
        }
        return $startDateOutput;
    }

    public static function checkCompareSession($id)
    {
        if (isset($_SESSION['courses'])) {
            $compareCourses = $_SESSION['courses'];
            $courseArray = json_decode($compareCourses);
            $fontAwesomeClass = in_array($id, $courseArray) ? 'fas' : 'far';
            $compareState = in_array($id, $courseArray) ? 'compare-is-active' : '';
            $buttonText = in_array($id, $courseArray) ? 'Remove from compare' : 'Compare Course';
        } else {
            $buttonText = 'Compare Course';
            $compareState = "";
            $fontAwesomeClass = "far";
        }

        return [
            'button_text' => $buttonText,
            'button_state' => $compareState,
            'font_awesome_class' => $fontAwesomeClass,
        ];
    }

    public static function outputCompareCourses($courses)
    {
        if ($_SESSION['courses'] !== '[]') {
            $args = [
                'post_type' => 'course',
                'post__in' => json_decode($courses),
            ];
            $compareQuery = new \WP_Query($args);
            if ($compareQuery->have_posts()) {
                $courseList = []; ?>
                <div class="course-bar">
                    <div class="container">
                        <div class="inner-bar">
                            <?php while ($compareQuery->have_posts()) { ?>
                                <?php $compareQuery->the_post();
                                array_push($courseList, get_the_title());
                            } ?>
                            <h2>
                                <span class="course-compare-title">Compare Courses</span>: <?= implode(', ', $courseList); ?>
                            </h2>
                            <button class="trigger-compare" aria-label="Open Compare Panel"><span class="text-section">Compare</span>
                                <span class="d-lg-none">(<?= $compareQuery->found_posts; ?>)</span></button>
                        </div>
                    </div>
                </div>
                <div class="course-block">
                    <div class="container">
                        <div class="custom-compare-next compare-control">
                            <span class="fa-stack fa-2x">
                              <i class="fas fa-circle fa-stack-2x"></i>
                              <i class="fas fa-chevron-right fa-stack-1x fa-inverse"></i>
                            </span>
                        </div>
                        <div class="custom-compare-prev compare-control">
                            <span class="fa-stack fa-2x">
                              <i class="fas fa-circle fa-stack-2x"></i>
                              <i class="fas fa-chevron-left fa-stack-1x fa-inverse"></i>
                            </span>
                        </div>
                        <div class="swiper-container compare-swiper">
                            <div class="swiper-wrapper" role="feed">
                                <?php while ($compareQuery->have_posts()) {
                                    $compareQuery->the_post();
                                    $courseID = get_the_id();
                                    $categoryColour = Site::getParentCategoryColour(get_the_id());
                                    $courseInstances = self::getCourseInstances(get_the_id());
                                    if ($courseInstances) {
                                        $instanceDurations = $courseInstances['duration'];
                                        $instanceStartDates = $courseInstances['start_dates'];
                                    }

                                    $title = get_field('qualification_type') ? get_the_title() . ' (' . get_field('qualification_type') . ')' : get_the_title(); ?>
                                    <div class="swiper-slide">
                                        <button class="remove-from-compare" data-course-id="<?php the_id() ?>"
                                                aria-label="<?= 'Remove ' . get_the_title() . ' from course compare.' ?>">
                                            <i class="fal fa-times"></i> Remove
                                        </button>
                                        <div class="block-info" style="border-top:8px solid <?= $categoryColour; ?>">
                                            <h2><?= $title; ?></h2>
                                            <div class="block-excerpt">
                                                <?php the_excerpt(); ?>
                                            </div>
                                            <div class="course-compare-holder">
                                                <section class="course-section">
                                                    <h3><i class="fas fa-graduation-cap course-svg"
                                                           style="color:<?= $categoryColour; ?>"></i> Qualification Type
                                                    </h3>
                                                    <p>
                                                        <?php
                                                        $courseLevels = self::getCourseLevels(get_the_id());
                                                        if ($courseLevels || get_field('qualification_type')) {
                                                            self::qualificationOutput($courseLevels, get_field('qualification_type'));
                                                        } else {
                                                            echo 'n/a';
                                                        } ?>
                                                    </p>
                                                </section>
                                                <section class="course-section">
                                                    <h3><i class="fas fa-clock course-svg"
                                                           style="color:<?= $categoryColour; ?>"></i> Duration</h3>
                                                    <?php if (!empty($instanceDurations)) {
                                                        $uniqueInstances = array_unique($instanceDurations);
                                                        if (count($uniqueInstances) > 1) {
                                                            $durationOutput = "Varies";
                                                        } else {
                                                            $durationOutput = $uniqueInstances[0];
                                                        }
                                                    } else {
                                                        $durationOutput = 'n/a';
                                                    } ?>
                                                    <p>
                                                        <?= $durationOutput; ?>
                                                    </p>
                                                </section>
                                                <section class="course-section">
                                                    <h3><i class="fas fa-calendar-alt course-svg"
                                                           style="color:<?= $categoryColour; ?>"></i> Starts</h3>
                                                    <?php if (!empty($instanceStartDates)) {
                                                        $modesOfStudy = self::getCourseModesOfStudy(get_the_id());
                                                        $startDateOutput = self::getCourseInstanceStartDates($instanceStartDates);
                                                        if(($startDateOutput == 'No dates available' && !in_array('SMA', $modesOfStudy))){
                                                            $startDateOutput = 'n/a';
                                                        } else {
                                                            $startDateOutput = self::getCourseInstanceStartDates($instanceStartDates);
                                                        }
                                                    } else {
                                                        $startDateOutput = 'n/a';
                                                    } ?>
                                                    <p>
                                                        <?= $startDateOutput; ?>
                                                    </p>
                                                </section>
                                                <section class="course-section">
                                                    <h3><i class="fas fa-books course-svg"
                                                           style="color:<?= $categoryColour; ?>"></i> Mode of Study</h3>
                                                    <ul role="list">
                                                        <?php
                                                        $modesOfStudy = self::getCourseModesOfStudy(get_the_id());
                                                        if ($modesOfStudy) {
                                                            foreach ($modesOfStudy as $mode) { ?>
                                                                <li role="listitem"><?= $mode ?></li>
                                                            <?php }
                                                        } else {
                                                            echo 'n/a';
                                                        } ?>
                                                    </ul>
                                                </section>
                                                <section class="course-section">
                                                    <h3><i class="fas fa-shapes course-svg"
                                                           style="color:<?= $categoryColour; ?>"></i> Interests</h3>
                                                    <ul role="list">
                                                        <?php
                                                        $interests = self::getCourseSubjects(get_the_id());
                                                        if ($interests) {
                                                            foreach ($interests as $interest) { ?>
                                                                <li role="listitem"><?= $interest ?></li>
                                                            <?php }
                                                        } else {
                                                            echo 'n/a';
                                                        } ?>
                                                    </ul>
                                                </section>
                                                <section class="course-section">
                                                    <h3><i class="fas fa-school course-svg"
                                                           style="color:<?= $categoryColour; ?>"></i> Campus</h3>
                                                    <ul role="list">
                                                        <?php
                                                        $campuses = self::getCourseCampuses(get_the_id());
                                                        if ($campuses && count($campuses) <= 2) {
                                                            foreach ($campuses as $campus) { ?>
                                                                <li role="list-item"><?= $campus ?></li>
                                                            <?php }
                                                        } elseif(count($campuses) > 2) { ?>
                                                            <li role="list-item">Options Available</li>
                                                        <?php } else {
                                                            echo 'n/a';
                                                        } ?>
                                                    </ul>
                                                </section>
                                            </div>
                                            <div class="view-course">
                                                <a href="<?= get_the_permalink($courseID); ?>"
                                                   style="color:<?= $categoryColour; ?>; border-bottom:2px solid <?= $categoryColour; ?>">View
                                                    Course</a>
                                            </div>
                                        </div>
                                    </div>
                                    <?php
                                } ?>
                            </div>
                        </div>
                    </div>
                </div>
                <?php
            }
        }
    }

    public static function getAllCourses($args = null, $paged = null, $isTag = null, $tag = null)
    {
        $keywordPostIDs = [];
        $courseData = [];
        $taxQuery = ['relation' => 'AND'];
        $metaQuery = [];
        $keywords = "";

        if (isset($_GET['mode-of-study']) || $args['mode-of-study']) {
            $modeOfStudy = $args['mode-of-study'] ?: $_GET['mode-of-study'];
            $taxQuery[] = [
                'taxonomy' => 'mode-of-study',
                'terms' => $modeOfStudy,
                'field' => 'slug',
                'operator' => 'IN'
            ];
        }
        if (isset($_GET['interests']) || $args['interests']) {
            $interest = $args['interests'] ?: $_GET['interests'];
            $taxQuery[] = [
                'taxonomy' => 'category',
                'terms' => [$interest],
                'field' => 'slug',
                'operator' => 'IN'
            ];
        }
        if (isset($_GET['level']) || $args['level']) {
            $level = $args['level'] ?: $_GET['level'];
            $additionalTerm = "";
            preg_match_all('/(?:\d-)?\d$/', $level, $matches);
            if ($matches) {
                $levelNumber = $matches[0][0];
                $additionalTerm = 'broadly-comparable-to-scqf-level-' . $levelNumber;
                $levels = [$level, $additionalTerm];
            } else {
                $levels = [$level];
            }

            $taxQuery[] = [
                'taxonomy' => 'level',
                'terms' => $levels,
                'field' => 'slug',
                'operator' => 'IN'
            ];
        }
        if (isset($_GET['campus']) || $args['campus']) {
            $campus = $args['campus'] ?: $_GET['campus'];
            $metaQuery[] = [
                'key' => 'course_instances_$_location',
                'value' => $campus,
                'compare' => 'LIKE'
            ];
        }

        if ((isset($_GET['keyword']) && $_GET['keyword'] !== '') || $args['keyword']) {
            $keywords = $args['keyword'] ?: $_GET['keyword'];

            $taggedPostIDs = [];
            $searchedPostIDs = [];
            $levelPostIDs = [];
            $andIDs = [];

            $tagTermIDs = get_terms([
                'taxonomy' => 'post_tag',
                'name__like' => $keywords,
                'fields' => 'ids',
                'override_where' => true,
                'hide_empty' => false
            ]);

            $levelTermIDs = get_terms([
                'taxonomy' => 'level',
                'name__like' => $keywords,
                'fields' => 'ids',
                'override_where' => true,
                'hide_empty' => false
            ]);

            if (count($levelTermIDs) >= 1) {
                $levelPosts = get_posts([
                    'post_type' => 'course',
                    'numberposts' => -1,
                    'tax_query' => [
                        [
                            'taxonomy' => 'level',
                            'terms' => $levelTermIDs,
                            'field' => 'term_id'
                        ]
                    ],
                ]);
                foreach ($levelPosts as $level) {
                    array_push($levelPostIDs, $level->ID);
                }
            }

            if (count($tagTermIDs) >= 1) {
                $tagPosts = get_posts([
                    'post_type' => 'course',
                    'numberposts' => -1,
                    'meta_query' => $metaQuery,
                    'tax_query' => $taxQuery,
                    'tag__in' => $tagTermIDs
                ]);
                foreach ($tagPosts as $tag) {
                    array_push($taggedPostIDs, $tag->ID);
                }
            }

            if(preg_match('(&|and)', $keywords) === 1) {
                $additionalKeywordCheck = strpos($keywords, 'and') !== false ? str_replace('and', '&', $keywords) : str_replace('&', 'and', $keywords);
                $args = [
                    'post_type' => 'course',
                    'numberposts' => -1,
                    'meta_query' => $metaQuery,
                    'tax_query' => $taxQuery,
                    's' => $additionalKeywordCheck
                ];
                $searchAndPosts = get_posts($args);
                foreach ($searchAndPosts as $and) {
                    array_push($andIDs, $and->ID);
                }
            }

            $args = [
                'post_type' => 'course',
                'numberposts' => -1,
                'meta_query' => $metaQuery,
                'tax_query' => $taxQuery,
                's' => $keywords
            ];

            $searchPosts = get_posts($args);

            foreach ($searchPosts as $search) {
                array_push($searchedPostIDs, $search->ID);
            }

            $keywordPostIDs = array_unique(array_merge($searchedPostIDs, $taggedPostIDs, $levelPostIDs, $andIDs));
        }

        if (isset($_GET['letter']) || $args['letter'] !== '') {
            $letter = $args['letter'] ?: $_GET['letter'];
        }

        if ($keywordPostIDs) {
            $args = [
                'post_type' => 'course',
                'post_status' => 'publish',
                'posts_per_page' => 10,
                "orderby" => 'title',
                "order" => 'ASC',
                "post__in" => $keywordPostIDs,
                'paged' => $paged,
            ];
        } else {
            $args = [
                'post_type' => 'course',
                'post_status' => 'publish',
                'posts_per_page' => 10,
                "orderby" => 'title',
                "order" => 'ASC',
                'meta_query' => $metaQuery,
                'tax_query' => $taxQuery,
                'paged' => $paged,
                's' => $keywords
            ];
        }

        if($letter) {
            $args['starts_with'] = $letter;
        }

        if($isTag == "true" || is_tag()) {
            $queredObject = get_queried_object();
            $searchTag = get_queried_object() ? $queredObject->slug : $tag;
            $args['tag'] = $searchTag;
        }

        $courses = new \WP_Query($args);

        while ($courses->have_posts()) {
            $courses->the_post();
            $firstLetter = get_the_title()[0];
            $courseData[$firstLetter][] = [
                'id' => get_the_id(),
                'title' => get_the_title(),
                'excerpt' => get_the_excerpt(),
                'slug' => get_post_field('post_name', get_post()),
                'qualification_type' => get_field('qualification_type'),
                'duration' => get_field('course_duration'),
                'start_date' => get_field('start_date'),
                'level' => self::getCourseLevels(get_the_id()),
                'modes_of_study' => self::getCourseModesOfStudy(get_the_id()),
                'subject_areas' => self::getCourseSubjects(get_the_id()),
                'campuses' => self::getCourseCampuses(get_the_id()),
            ];
        }

        $filter = $_POST ? $_POST['filters'] : '';

        if($isTag == "true" || is_tag()) {
            $queredObject = get_queried_object();
            $searchTag = get_queried_object() ? $queredObject->slug : $tag;
            $links = paginate_links([
                'base' => esc_url(home_url('/tag/' . $searchTag . '/page/%#%/')),
                'format' => '/tag/' . $searchTag . '/page/%#%/',
                'total' => $courses->max_num_pages,
                'current' => $paged,
                'add_args' => $filter
            ]);
        } else {
            $links = paginate_links([
                'base' => esc_url(home_url('/courses/page/%#%/')),
                'format' => '/courses/page/%#%/',
                'total' => $courses->max_num_pages,
                'current' => $paged,
                'add_args' => $filter
            ]);
        }

        return ['data' => $courseData, 'pagination' => $links];
    }

    public static function outputCourses($courses)
    {

        $pagination = $courses['pagination'];
        $courses = $courses['data'];

        if ($courses) {
            if (session_status() == PHP_SESSION_NONE) {
                session_start();
            }
            foreach ($courses as $courseLetter => $row) { ?>
                <section class="course-item-group" role="feed">
                    <div class="row">
                        <div class="col-lg-1 course-letter-field">
                            <div><?= $courseLetter; ?></div>
                        </div>
                        <div class="col-lg-11">
                            <?php foreach ($row as $courseItem) {
                                $categoryColour = Site::getParentCategoryColour($courseItem['id']);
                                $compareSession = self::checkCompareSession($courseItem['id']);
                                $courseInstances = self::getCourseInstances($courseItem['id']);
                                if ($courseInstances) {
                                    $instanceDurations = $courseInstances['duration'];
                                    $instanceStartDates = $courseInstances['start_dates'];
                                } ?>
                                <article class="course-index-item">
                                    <div class="course-index-inner" style="border-color:<?= $categoryColour; ?>">
                                        <?php if ($courseItem['level'] || $courseItem['qualification_type']) { ?>
                                            <span class="qualification-scqf"
                                                  style="border:1px solid <?= $categoryColour; ?>; background:<?= $categoryColour; ?>">
                                                <?php self::qualificationOutput($courseItem['level'], $courseItem['qualification_type'], $courseItem['id']); ?>
                                            </span>
                                        <?php } ?>
                                        <?php
                                        if (has_tag('coming-soon', $courseItem['id'])) { ?>
                                            <span class="coming-soon"
                                                  style="border:1px solid <?= $categoryColour; ?>; color:<?= $categoryColour; ?>">Coming soon</span>
                                        <?php }
                                        if (get_field('activate_course_compare', 'option')) { ?>
                                            <span class="course-compare">
                                                <button aria-label="<?= "Compare toggle for the " . $courseItem['title'] . ' course'; ?>"
                                                        data-course-id="<?= $courseItem['id']; ?>"
                                                        id="<?= 'compare-item-' . $courseItem['id']; ?>"
                                                        class="add-to-compare <?= $compareSession['button_state']; ?>"
                                                        style="color:<?= $categoryColour; ?>"><i
                                                            class="<?= $compareSession['font_awesome_class'] ?> fa-square custom-compare-svg"></i>Compare Course</button>
                                            </span>
                                        <?php } ?>
                                        <h2>
                                            <a href="<?php the_permalink($courseItem['id']); ?>"><?= $courseItem['title']; ?></a><?php include(locate_template('templates/pins/pin-course.php', false, false)); ?>
                                        </h2>
                                        <p><?= $courseItem['excerpt']; ?></p>
                                        <hr/>
                                        <div class="row">
                                            <?php if (!empty($courseItem['qualification_type']) || !empty($courseItem['level'])) { ?>
                                                <div class="col-lg-2">
                                                    <div class="course-detail">
                                                        <h3><i class="fas fa-graduation-cap course-svg" style="color:<?= $categoryColour; ?>"></i> Qualification Type</h3>
                                                        <p>
                                                            <?php self::qualificationOutput($courseItem['level'], $courseItem['qualification_type'], $courseItem['id']); ?>
                                                        </p>
                                                    </div>
                                                </div>
                                            <?php }
                                            if (!empty($instanceDurations)) {
                                                $uniqueInstances = array_unique($instanceDurations);
                                                if (count($uniqueInstances) > 1) {
                                                    $durationOutput = 'Varies';
                                                } else {
                                                    $durationOutput = $uniqueInstances[0];
                                                } ?>
                                                <div class="col-lg-2">
                                                    <div class="course-detail">
                                                        <h3><i class="fas fa-clock course-svg"
                                                               style="color:<?= $categoryColour; ?>"></i> Duration</h3>
                                                        <p><?= $durationOutput; ?></p>
                                                    </div>
                                                </div>
                                                <?php
                                            }

                                            if(!empty($instanceStartDates)) {
                                                $startCheck = self::getCourseInstanceStartDates($instanceStartDates);
                                                if ($startCheck !== 'No dates available' || ($startCheck == 'No dates available' && in_array('SMA', $courseItem['modes_of_study']))) { ?>
                                                    <div class="col-lg-2">
                                                        <div class="course-detail">
                                                            <h3><i class="fas fa-calendar-alt course-svg" style="color:<?= $categoryColour; ?>"></i> Starts</h3>
                                                            <p><?= $startCheck; ?></p>
                                                        </div>
                                                    </div>
                                                <?php }
                                            }

                                            if (!empty($courseItem['modes_of_study']) && !in_array('SMA', $courseItem['modes_of_study'])) { ?>
                                                <div class="col-lg-2">
                                                    <div class="course-detail">
                                                        <h3><i class="fas fa-books course-svg"
                                                               style="color:<?= $categoryColour; ?>"></i> Mode of Study
                                                        </h3>
                                                        <ul role="list">
                                                            <?php foreach ($courseItem['modes_of_study'] as $mode) { ?>
                                                                <li role="listitem"><?= $mode ?></li>
                                                            <?php } ?>
                                                        </ul>
                                                    </div>
                                                </div>
                                            <?php }
                                            if (!empty($courseItem['subject_areas'])) { ?>
                                                <div class="col-lg-2">
                                                    <div class="course-detail">
                                                        <h3><i class="fas fa-shapes course-svg"
                                                               style="color:<?= $categoryColour; ?>"></i> Interests</h3>
                                                        <ul role="list">
                                                            <?php foreach ($courseItem['subject_areas'] as $sub) { ?>
                                                                <li role="listitem"
                                                                    data-pin-subjectcat="<?= get_cat_ID($sub); ?>"><?= $sub ?></li>
                                                            <?php } ?>
                                                        </ul>
                                                    </div>
                                                </div>
                                            <?php }
                                            if (!empty($courseItem['campuses'])) { ?>
                                                <div class="col-lg-2">
                                                    <div class="course-detail">
                                                        <h3><i class="fas fa-school course-svg"
                                                               style="color:<?= $categoryColour; ?>"></i> Campus</h3>
                                                        <ul role="list">
                                                            <?php
                                                            if ($courseItem['campuses'] && count($courseItem['campuses']) <= 2) {
                                                                foreach ($courseItem['campuses'] as $campus) { ?>
                                                                    <li role="list-item"><?= $campus ?></li>
                                                                <?php }
                                                            } elseif(count($courseItem['campuses']) > 2) { ?>
                                                                <li role="list-item">Options Available</li>
                                                            <?php } else {
                                                                echo 'n/a';
                                                            } ?>
                                                        </ul>
                                                    </div>
                                                </div>
                                            <?php } ?>
                                        </div>
                                        <div class="d-lg-none mobile-buttons">
                                            <ul role="list">
                                                <li role="listitem"><a href="<?php the_permalink($courseItem['id']); ?>"
                                                                       class="view"
                                                                       style="background:<?= $categoryColour; ?>"
                                                                       aria-label="<?= 'View' . $courseItem['title'] . ' course' ?>">View
                                                        Course</a></li>
                                                <?php if (get_field('activate_course_compare', 'option')) { ?>
                                                    <li role="listitem">
                                                        <button aria-label="<?= "Compare toggle for the " . $courseItem['title'] . ' course'; ?>"
                                                                data-course-id="<?= $courseItem['id']; ?>"
                                                                class="add-to-compare <?= $compareSession['button_state']; ?>"
                                                                id="<?= 'mobile-compare-item-' . $courseItem['id']; ?>"
                                                                style="color:<?= $categoryColour ?>; border-color:<?= $categoryColour; ?>"><?= $compareSession['button_text']; ?></button>
                                                    </li>
                                                <?php } ?>
                                            </ul>
                                        </div>
                                    </div>
                                </article>
                                <?php
                            } ?>
                        </div>
                    </div>
                </section>
            <?php } ?>
            <div class="custom-course-pagination">
                <div class="container">
                    <?= $pagination; ?>
                </div>
            </div>
        <?php } else { ?>
            <div class="no-course-results">
                <?php the_field('course_archive_no_results_message', 'option'); ?>
            </div>
        <?php }

    }

    public static function qualificationOutput($level, $qualification, $courseID = null)
    {
        global $post;
        $levels = $level;
        $qualificationType = $qualification;
        $id = $courseID ?: $post->ID;

        if(count($levels) > 1 && class_exists('WPSEO_Options')) {
            foreach($levels as $term) {
                $levelCheck = get_term_by('name', $term, 'level');
                if( get_post_meta($id, '_yoast_wpseo_primary_level',true) == $levelCheck->term_id ) {
                    $level = $levelCheck->name;
                }
            }
        } else {
            $level = implode(', ', $levels);
        }

        if ($qualificationType || $level) {
            if ($qualificationType && $level) {
                echo $qualificationType . " (" . $level . ")";
            } elseif ($qualificationType && !$level) {
                echo $qualificationType;
            } elseif (!$qualificationType && $level) {
                echo $level;
            }
        }
    }

    public static function generateTaxSelect($items, $slug, $name)
    {
        $defaultCheck = !isset($_GET[$slug]) ? 'checked' : ''; ?>
        <div class="select-box">
            <div class="select-box__current" tabindex="1">
                <div class="select-box__value">
                    <input class="select-box__input filter-select" type="radio" data-label=""
                           id="<?= 'disabled-' . $slug; ?>" name="<?= $slug; ?>" <?= $defaultCheck; ?> disabled>
                    <p class="select-box__input-text"><?= $name; ?></p>
                </div>
                <?php
                $itemCount = 0;
                foreach ($items as $item) {
                    $checked = isset($_GET[$slug]) && $_GET[$slug] == $item->slug ? 'checked' : ''; ?>
                    <div class="select-box__value">
                        <input class="select-box__input filter-select" type="radio" data-label="<?= $item->name ?>"
                               id="<?= $item->slug ?>" value="<?= $item->slug; ?>"
                               name="<?= $slug; ?>" <?= $checked; ?>>
                        <p class="select-box__input-text"><?= $item->name; ?></p>
                    </div>
                    <?php $itemCount++;
                } ?>
                <i class="far fa-chevron-down select-box__icon"></i>
            </div>
            <ul class="select-box__list" role="list">
                <li class="filters__back">
                    <button class="d-lg-none"><span class="fa fa-chevron-left"></span> <?= $name; ?></button>
                </li>
                <?php $itemListCount = 0;
                foreach ($items as $item) { ?>
                    <li role="listitem">
                        <label class="select-box__option" for="<?= $item->slug ?>"
                               aria-hidden="aria-hidden"><?= $item->name; ?></label>
                    </li>
                    <?php $itemListCount++;
                } ?>
            </ul>
        </div>
        <?php
    }

    public static function generateLocationSelect($items)
    {
        $locations = $items; ?>
        <div class="select-box">
            <div class="select-box__current" tabindex="1">
                <div class="select-box__value">
                    <input class="select-box__input filter-select" type="radio" data-label=""
                           id="<?= 'disabled-campus'; ?>" value="" name="campus" checked="checked" disabled>
                    <p class="select-box__input-text">Location</p>
                </div>
                <?php $locationCount = 0;
                foreach ($locations as $location) {
                    $getLocation = get_page_by_title($location, null, 'location');
                    $locationSlug = strtolower(str_replace(' ', '-', $location));
                    $checked = isset($_GET['campus']) && $_GET['campus'] == $getLocation->ID ? 'checked' : ''; ?>
                    <div class="select-box__value">
                        <input class="select-box__input filter-select" type="radio" data-label="<?= $location ?>"
                               id="<?= $locationCount ?>" value="<?= $getLocation->ID; ?>"
                               name="campus" <?= $checked; ?>>
                        <p class="select-box__input-text"><?= $location; ?></p>
                    </div>
                    <?php $locationCount++;
                } ?>
                <i class="far fa-chevron-down select-box__icon"></i>
            </div>
            <ul class="select-box__list" role="list">
                <li class="filters__back">
                    <button class="d-lg-none"><span class="fa fa-chevron-left"></span> Locations</button>
                </li>
                <?php
                $locationCountList = 0;
                $getLocation = get_page_by_title($location, null, 'location');
                $locationSlug = strtolower(str_replace(' ', '-', $location));
                foreach ($locations as $location) { ?>
                    <li role="listitem">
                        <label class="select-box__option" for="<?= $locationCountList ?>"
                               aria-hidden="aria-hidden"><?= $location ?></label>
                    </li>
                    <?php $locationCountList++;
                } ?>
            </ul>
        </div>
        <?php
    }

    public static function generateApplicationPanel($id)
    {
        $applicationPanel = get_field('application_panel');
        if ($applicationPanel == 'statistics' && have_rows('application_statistics')) {
            while (have_rows('application_statistics')) {
                the_row();
                $source = get_sub_field('source');
                $sourceText = $source ? $source['title'] : 'Source'; ?>
                <div class="application-statistics">
                    <div class="stat">
                        <h3><?php the_sub_field('value'); ?></h3>
                        <p><?php the_sub_field('point'); ?></p>
                        <?php if($source) { ?><a href="<?= $source['url']; ?>" class="statistic-source" target="<?= $source['target']; ?>"><?= $sourceText; ?></a><?php } ?>
                    </div>
                </div>
                <?php
            }
        } elseif ($applicationPanel == 'media' && (get_field('application_video') || get_field('application_image'))) { ?>
            <div class="application-holder d-flex align-items-center justify-content-center">
                <?php if (get_field('application_video')) {
                    $video = get_field('application_video'); ?>
                    <video autoplay muted loop controls class="opportunities-video">
                        <source src="<?= $video['url']; ?>" type="video/mp4">
                    </video>
                    <?php
                } else {
                    $appImage = get_field('application_image'); ?>
                    <img src="<?= $appImage['url']; ?>" alt="Opportunities">
                    <?php
                } ?>
            </div>
        <?php }
    }

    public static function generateDisclaimer($mode)
    {
        if (get_field('show_disclaimer')) {
            $mode = get_the_terms(get_the_id(), 'mode-of-study');
            if ($mode) {
                $modeSlug = $mode[0]->slug; ?>
                <div class="disclaimer">
                    <?php the_field($modeSlug . '_disclaimer', 'option'); ?>
                </div>
                <?php
            }
        }
    }

    public static function getApplicationData($mode)
    {
        if (have_rows('course_instances')) {
            while (have_rows('course_instances')) {
                the_row();
                $code = get_sub_field('code');

                if (has_term('school-links', 'mode-of-study')) {
                    $link = 'https://onlineservices.nescol.ac.uk/WCFWebsite/Nescol_Schools_Portal/external.aspx?modref=' . $code;
                } else {
                    $bookingString = $mode == 'part-time' || $mode == 'distance-learning' ? 'NescolBook' : "NescolApply";
                    $link = get_sub_field('link') ?: 'https://onlineservices.nescol.ac.uk/WCFWebsite/' . $bookingString . '/results.aspx?search=' . $code;
                }

                $linkType = get_sub_field('link') ? 'link' : 'pdf';

                //Check if an SMA course date instance has passed an it excludes it if so
                if (has_term('sma', 'mode-of-study') || has_term('part-time', 'mode-of-study')) {
                    $startDate = get_sub_field('start_date');
                    $currentDate = date('Y-m-d');
                    $startDateTime = strtotime($startDate);
                    $currentDateTime = strtotime($currentDate);
                    if ($startDateTime < $currentDateTime) {
                        continue;
                    }
                }

                $courseAppDate = [
                    'mode_of_study' => $mode,
                    'location' => get_sub_field('location'),
                    'duration' => get_sub_field('duration'),
                    'start_date' => get_sub_field('start_date'),
                    'end_date' => get_sub_field('end_date'),
                    'time' => get_sub_field('time'),
                    'day' => get_sub_field('day'),
                    'fees' => get_sub_field('fees'),
                    'code' => $code,
                    'qualification_portion' => get_sub_field('qualification_portion'),
                    'link' => $link,
                    'link_type' => $linkType,
                    'show_on_site' => get_sub_field('show_on_site'),
                    'places_available' => get_sub_field('places_available'),
                    'online' => get_sub_field('online'),
                    'disable_course_instance_cta' => get_sub_field('disable_course_instance_cta'),
                    'disabled_course_instance_message' => get_sub_field('disabled_course_instance_message'),
                    'disable_paper_applications' => get_field('disable_paper_applications'),
                    'disabled_paper_applications_message' => get_field('disabled_paper_applications_message'),
                    'hide_course_start_date' => get_sub_field('hide_course_start_date')
                ];
                $applicationData[] = $courseAppDate;
            }

            usort($applicationData, function ($a, $b) {
                return $a['start_date'] <=> $b['start_date'];
            });

            return $applicationData;
        }
    }

    public static function sortApplicationData($data)
    {
        $applyOnlineArray = [];
        $bookOnlineArray = [];
        $paperArray = [];

        if ($data) {
            foreach ($data as $applicationRow) {
                $showOnSite = $applicationRow['show_on_site'];
                $mode = $applicationRow['mode_of_study'];
                $online = $applicationRow['online'];

                if ($showOnSite) {
                    switch ($mode) {
                        case 'part-time':
                            if ($online) {
                                $bookOnlineArray[] = $applicationRow;
                            } else {
                                $paperArray[] = $applicationRow;
                            }
                            break;
                        case 'distance-learning':
                            if (!$online) {
                                $paperArray[] = $applicationRow;
                            } else {
                                $bookOnlineArray[] = $applicationRow;
                            }
                            break;
                        case 'full-time':
                            if (!$online) {
                                $paperArray[] = $applicationRow;
                            } else {
                                $applyOnlineArray[] = $applicationRow;
                            }
                            break;
                        case 'school-links':
                        case 'apprenticeships':
                            if ($online) {
                                $applyOnlineArray[] = $applicationRow;
                            }
                            break;
                        case 'training-for-business':
                        case 'sma':
                            $applyOnlineArray[] = $applicationRow;
                            break;
                    }
                }
            }
        }

        $applicationData = [];
        if (!empty($applyOnlineArray)) {
            $applicationData['Online Applications'] = $applyOnlineArray;
        }
        if (!empty($bookOnlineArray)) {
            $applicationData['Book Online'] = $bookOnlineArray;
        }
        if (!empty($paperArray)) {
            $applicationData['Paper Applications'] = $paperArray;
        }

        return $applicationData;
    }

    public static function generatePaperApplications()
    {
        $mode = get_the_terms(get_the_id(), 'mode-of-study');
        $modeSlug = $mode[0]->slug;
        if (get_field('paper_applications')) {
            $paperApplications = 'paper_applications';
            $append = "";
            $repeaterField = get_field($paperApplications);
        } else {
            $paperApplications = 'paper_applications_' . $modeSlug;
            $append = "option";
            $repeaterField = get_field($paperApplications, $append);
        }

        if (have_rows($paperApplications, $append)) {
            while (have_rows($paperApplications, $append)) {
                the_row();
                $pdfName = get_sub_field('name');
                $file = get_sub_field('file');
                if ($file) {
                    $downloadURL = $file['url'];
                    $downloadText = 'Download';
                } else {
                    $downloadURL = get_sub_field('url');
                    $downloadText = $modeSlug == 'distance-learning' ? 'Download' : 'Read More';
                } ?>

                <div class="row d-flex align-items-center paper-app-row">
                    <div class="col-xl-8 paper-app-header">
                        <h3><?= $pdfName; ?></h3>
                    </div>
                    <div class="col-xl-4">
                        <div class="application-cta">
                            <a href="<?= $downloadURL; ?>" class="cta" target="_blank"
                               aria-label="Open '<?= $pdfName ?>' PDF"><?= $downloadText; ?></a>
                        </div>
                    </div>
                </div>
                <?php
            }
        }
    }

    public static function outputApplicationInstances($sortedData) {

        $smaExcludedList = self::getExcludedCourses();
        $courseID = get_the_id();
        $systemMaintenance = get_field('enable_system_maintenance_mode', 'option');

        ?>
        <div class="applications-split">
            <div class="row">
                <div class="col-lg-6">
                    <div class="application-area">
                        <?php if (!in_array($courseID, $smaExcludedList) && have_rows('course_instances') && (!empty($sortedData['Book Online']) || !empty($sortedData['Online Applications']) || !empty($sortedData['Paper Applications']))) { ?>
                            <div class="applications-accordion">
                                <div class="accordion" id="applicationAccordion">
                                    <?php foreach ($sortedData as $key => $data) {
                                        $keySlug = strtolower(str_replace(' ', '-', $key));
                                        $terms = get_the_terms(get_the_id(), 'mode-of-study');
                                        $modesOfStudy = [];
                                        foreach ($terms as $term) {
                                            $modesOfStudy[] = $term->slug;
                                        }
                                        $endDateCheck = count(array_intersect(['full-time', 'part-time', 'training-for-business', 'school-link', 'distance-learning', 'sma'], $modesOfStudy)) ? false : true; ?>
                                        <div class="card">
                                            <div class="card-header" id="headingOnline<?= $keySlug; ?>">
                                                <h2 class="mb-0">
                                                    <button class="apply-focus btn btn-link accordion-toggle" type="button" data-toggle="collapse" data-target="#collapseOnlineapplications<?= $keySlug; ?>" aria-expanded="true" aria-controls="<?= 'Toggle ' . $key . ' accordion' ?>">
                                                        <?= $key ?> <i class="far fa-plus"></i>
                                                    </button>
                                                </h2>
                                            </div>
                                            <div id="collapseOnlineapplications<?= $keySlug; ?>" class="collapse" aria-labelledby="<?= $key . ' Information' ?>" data-parent="#applicationAccordion">
                                                <div class="card-body">
                                                    <?php
                                                    switch ($keySlug) {
                                                        case 'book-online':
                                                            $output = get_field('book_now_applications_default', 'option');
                                                            break;
                                                        case 'paper-applications':
                                                            $output = get_field('disable_paper_applications') ? "" : get_field('paper_applications_default', 'option');
                                                            break;
                                                        default:
                                                            $output = get_field('online_applications_default', 'option');
                                                    }

                                                    echo $output;

                                                    if ($data) {
                                                        foreach ($data as $instance) {

                                                            if($keySlug == 'paper-applications' && $instance['disable_paper_applications']) { ?>
                                                                <div class="disabled-paper-application-message">
                                                                    <?= $instance['disabled_paper_applications_message']; ?>
                                                                </div>
                                                                <?php break;
                                                            }

                                                            if ($instance['online']) {
                                                                $keyText = ($key == 'Online Applications') || has_term('school-links', 'mode-of-study') ? 'Apply Online' : 'Book Online';

                                                                if(($key == 'Online Applications') || has_term('school-links', 'mode-of-study')) {
                                                                    $keyText = 'Apply Online';
                                                                } else if(has_term('part-time', 'mode-of-study') && has_term('training-for-business', 'mode-of-study')) {
                                                                    $keyText = 'Book yourself';
                                                                } else {
                                                                    $keyText = 'Book Online';
                                                                }

                                                                $status = 'button-enabled';
                                                            } else {
                                                                $keyText = "";
                                                                $status = 'button-disabled';
                                                            } ?>
                                                            <div class="application-item">
                                                                <?php if ($instance['location']) { ?><h3 class="instance-location"><?= $instance['location']->post_title; ?></h3><?php } ?>
                                                                <ul>
                                                                    <?php if ($instance['start_date'] && !$instance['hide_course_start_date']) {
                                                                        $date = strtotime($instance['start_date']);
                                                                        $date = date('d M Y', $date); ?>
                                                                        <li class="app-block">
                                                                            <h4>
                                                                                <span class="highlight">Starts</span>
                                                                                <span class="value"><?= $date ?></span>
                                                                            </h4>
                                                                        </li>
                                                                        <?php
                                                                    }
                                                                    if ($instance['end_date'] && $endDateCheck && !has_term('school-links', 'mode-of-study')) {
                                                                        $date = strtotime($instance['end_date']);
                                                                        $date = date('d M Y', $date); ?>
                                                                        <li class="app-block">
                                                                            <h4>
                                                                                <span class="highlight">Ends</span>
                                                                                <span class="value"><?= $date ?></span>
                                                                            </h4>
                                                                        </li>
                                                                        <?php
                                                                    }
                                                                    if ($instance['duration']) { ?>
                                                                        <li class="app-block">
                                                                            <h4>
                                                                                <span class="highlight">Duration</span>
                                                                                <span class="value"><?= $instance['duration']; ?></span>
                                                                            </h4>
                                                                        </li>
                                                                    <?php }
                                                                    if ($instance['day']) { ?>
                                                                        <li class="app-block">
                                                                            <h4>
                                                                                <span class="highlight">Day</span>
                                                                                <span class="value"><?= $instance['day']; ?></span>
                                                                            </h4>
                                                                        </li>
                                                                    <?php }
                                                                    if ($instance['time']) { ?>
                                                                        <li class="app-block">
                                                                            <h4>
                                                                                <span class="highlight">Time</span>
                                                                                <span class="value"><?= $instance['time']; ?></span>
                                                                            </h4>
                                                                        </li>
                                                                    <?php }
                                                                    if ($instance['code']) { ?>
                                                                        <li class="app-block">
                                                                            <h4>
                                                                                <span class="highlight">Course Code</span>
                                                                                <span class="value"><?= $instance['code']; ?></span>
                                                                            </h4>
                                                                        </li>
                                                                    <?php }
                                                                    if ($instance['fees']) { ?>
                                                                        <li class="app-block">
                                                                            <h4>
                                                                                <span class="highlight">Fees</span>
                                                                                <span class="value"><?= '&pound;' . $instance['fees']; ?></span>
                                                                            </h4>
                                                                        </li>
                                                                    <?php }
                                                                    if ($instance['qualification_portion']) {
                                                                        $qualificationText = in_array('part-time', $modesOfStudy) ? 'Qualification' : 'Qualification Portion' ?>
                                                                        <li class="app-block">
                                                                            <h4>
                                                                                <span class="highlight"><?= $qualificationText; ?></span>
                                                                                <span class="value"><?= $instance['qualification_portion']; ?></span>
                                                                            </h4>
                                                                        </li>
                                                                        <?php
                                                                    } ?>
                                                                </ul>
                                                                <?php if(!$instance['disable_course_instance_cta']) { ?>
                                                                    <div class="application-cta <?= $status; ?>">
                                                                        <a href="<?= $instance['link']; ?>" aria-label="<?= "Book the " . get_the_title() . " course starting on " . get_sub_field('start_date'); ?>" class="cta <?php if($systemMaintenance) { echo 'system-maintenance'; } ?>" target="_blank"><?= $keyText; ?></a>

                                                                        <?php if(has_term('part-time', 'mode-of-study') && has_term('training-for-business', 'mode-of-study')) { ?>
                                                                            <a href="<?= 'https://onlineservices.nescol.ac.uk/WCFWebsite/Nescol_Employer_Portal/external.aspx?modref=' . $instance['code']; ?>" aria-label="<?= "Book the " . get_the_title() . " course starting on " . get_sub_field('start_date'); ?>" class="cta" target="_blank" <?php if($systemMaintenance) { echo 'disabled'; } ?>><?= 'Book for an employee'; ?></a>
                                                                        <?php } ?>

                                                                    </div>
                                                                <?php } else { ?>
                                                                    <div class="disabled-course-instance-message">
                                                                        <div class="disabled-paper-application-message">
                                                                            <?= $instance['disabled_course_instance_message']; ?>
                                                                        </div>
                                                                    </div>
                                                                <?php } ?>
                                                            </div>
                                                            <?php
                                                            //Add Paper Instance and default to paper accordion
                                                            if ($keySlug == 'paper-applications') {
                                                                self::generatePaperApplications();
                                                            }
                                                        }
                                                    } ?>
                                                </div>
                                            </div>
                                        </div>
                                        <?php
                                    } ?>

                                    <?php
                                    $mode = get_the_terms(get_the_id(), 'mode-of-study');
                                    $modeSlug = $mode[0]->slug;
                                    $defaultPaperApps = get_field('paper_applications');
                                    $modeOfStudyPaperApps = get_field('paper_applications_' . $modeSlug, 'option');

                                    if ((!array_key_exists('Paper Applications', $sortedData)) && ($defaultPaperApps || $modeOfStudyPaperApps)) { ?>
                                        <div class="card">
                                            <div class="card-header" id="headingOnlinepaper-applications">
                                                <h2 class="mb-0">
                                                    <button class="apply-focus btn btn-link accordion-toggle" type="button" data-toggle="collapse" data-target="#collapseOnlineapplicationspaper-applications" aria-expanded="true" aria-controls="<?= 'Toggle ' . $key . ' accordion' ?>">
                                                        Paper Applications <i class="far fa-plus"></i>
                                                    </button>
                                                </h2>
                                            </div>
                                            <div id="collapseOnlineapplicationspaper-applications" class="collapse" aria-labelledby="Paper Applications Information" data-parent="#applicationAccordion">
                                                <div class="card-body">
                                                    <?php if(!get_field('disable_paper_applications')) {
                                                        the_field('paper_applications_default', 'option');
                                                        self::generatePaperApplications();
                                                    } else { ?>
                                                        <div class="disabled-paper-application-message">
                                                            <?php the_field('disabled_paper_applications_message') ?>
                                                        </div>
                                                    <?php } ?>
                                                </div>
                                            </div>
                                        </div>
                                    <?php } ?>
                                </div>
                            </div>
                        <?php }
                        else {
                            get_template_part('/templates/sections/course/register-an-interest');
                        }  ?>
                        <div class="contact-message">
                            <?php self::generateUnderAccordionText(); ?>
                        </div>
                    </div>
                </div>
                <div class="col-lg-6">
                    <?php
                    self::generateApplicationPanel(get_the_id());
                    self::generateDisclaimer(get_the_id());
                    ?>
                </div>
            </div>
        </div>
    <?php }

    public static function generateUnderAccordionText()
    {
        $modeOfStudy = get_the_terms(get_the_id(), 'mode-of-study');
        if ($modeOfStudy) {
            $modeOfStudy = $modeOfStudy[0]->slug;
            $underContentOverride = get_field($modeOfStudy . '_under_accordion_content', 'option');
            if ($underContentOverride) {
                $content = $underContentOverride;
            } else {
                if ($modeOfStudy == 'sma') {
                    $content = get_field('sma_contact_message', 'option');
                } else {
                    $content = get_field('contact_message', 'option');
                }
            }
            echo $content;
        }
    }

    public static function getPostPrimaryMode($id = null)
    {
        $modeList = wp_get_post_terms($id, 'mode-of-study', ['fields' => 'all']);
        if (count($modeList) > 1) {
            foreach ($modeList as $mode) {
                if (get_post_meta($id, '_yoast_wpseo_primary_mode-of-study', true) == $mode->term_id) {
                    $primaryMode = $mode;
                }
            }
        } else {
            $primaryMode = $modeList[0];
        }
        return $primaryMode;
    }

    public static function getPDFInfoSections($id = null)
    {
        $infoArray = [];
        $primaryMode = self::getPostPrimaryMode($id);
        $modeString = $primaryMode->slug;
        $modePdfDefaults =  $modeString . '_pdf_defaults';
        $excludeGenericDefaults = get_field($modeString . '_exclude_generic_pdf_default_information', 'option');
        if (!$excludeGenericDefaults) {
            if (have_rows('generic_pdf_defaults', 'option')) {
                while (have_rows('generic_pdf_defaults', 'option')) {
                    the_row();
                    $infoArray[] = ['title' => get_sub_field('title'), 'content' => get_sub_field('content')];
                }
            };
        }

        if (have_rows($modePdfDefaults, 'option')) {
            while (have_rows($modePdfDefaults, 'option')) {
                the_row();
                $infoArray[] = ['title' => get_sub_field('title'), 'content' => get_sub_field('content')];
            }
        };
        return $infoArray;
    }

    public static function createPDFLayout($id)
    {
        ob_start();
        setup_postdata($id);
        $categoryColour = Site::getParentCategoryColour($id);
        $levels = self::getCourseLevels($id);
        $qualificationType = get_field('qualification_type', $id);
        $modesOfStudy = self::getCourseModesOfStudy($id);
        $subjects = self::getCourseSubjects($id);
        $locations = self::getCourseCampuses($id);
        $academicYear = has_term('full-time', 'mode-of-study', $id) ? '2021/2022' : '2020/2021';

        get_template_part('templates/style'); ?>
        <div class="hidden-pdf">
            <style>
                a{
                    color:<?= $categoryColour; ?>
                }
            </style>
            <div id="course-pdf-generation">
                <div class="above-top">
                    <p class="nescol-name">North East Scotland College</p>
                    <p class="academic-calendar">Academic Year <?= $academicYear; ?></p>
                </div>
                <div class="top-block" style="background:<?= $categoryColour; ?>; color:white;">
                    <div class="stats"><?php if ($levels || $qualificationType) {
                            self::qualificationOutput($levels, get_field('qualification_type', $id));
                        } ?></div>
                    <h1 id="course-title"><?= get_the_title($id); ?></h1>
                    <?php
                    $content_post = get_post($id);
                    $content = $content_post->post_content;
                    $content = apply_filters('the_content', $content);
                    $content = str_replace(']]>', ']]&gt;', $content);
                    echo $content; ?>
                </div>
                <div class="pdf-section">
                    <div class="key-info-block">
                        <h2>Key Information</h2>
                        <ul>
                            <?php if ($levels || $qualificationType) {?> <li><span style="color:<?= $categoryColour ?>">Qualification</span>: <?php self::qualificationOutput($levels, get_field('qualification_type', $id)); ?></li> <?php } ?>
                            <?php if ($modesOfStudy && !in_array('SMA', $modesOfStudy)) { ?><li><span style="color:<?= $categoryColour ?>">Mode of Study</span>: <?= implode(', ', $modesOfStudy); ?></li><?php } ?>
                            <?php if ($subjects) { ?><li><span style="color:<?= $categoryColour ?>">Interests</span>: <?= implode(', ', $subjects); ?></li><?php } ?>
                            <?php if ($locations) { ?><li><span style="color:<?= $categoryColour ?>">Locations</span>: <?= implode(', ', $locations); ?></li><?php } ?>
                        </ul>
                    </div>

                    <?php if (have_rows('info_boxes', $id)) { ?>
                        <br/>
                        <hr class="info-box-hr" />
                        <h2>Key Facts</h2>
                        <div class="info-blocks">
                            <?php while (have_rows('info_boxes', $id)) {
                                the_row();
                                if (get_row_layout() == 'text_block') { ?>
                                    <?php the_sub_field('text'); ?>
                                <?php } ?>
                                <?php
                            } ?>
                        </div>
                        <hr/>
                    <?php }
                    if (get_field('why_take_this_course', $id)) { ?>
                        <h2>Why take this course?</h2>
                        <?php the_field('why_take_this_course', $id);
                    }
                    if (get_field('what_will_i_experience', $id)) { ?>
                        <h2>What you will experience</h2>
                        <?php the_field('what_will_i_experience', $id);
                    } ?>
                </div>
                <div id="opportunities" class="pdf-section">
                    <?php
                    $opportunitiesTitle = get_field('opportunities_title_override', $id) ?: get_field('default_opportunities_title', 'option');
                    $opportunitiesDescription = get_field('opportunities_description_override', $id) ?: get_field('default_opportunities_description', 'option'); ?>
                    <h2><?= $opportunitiesTitle; ?></h2>
                    <?= $opportunitiesDescription; ?>
                </div>
                <?php $related = get_field('education_course_opportunities', $id);
                if (get_field('education_description', $id)  || $related) { ?>
                    <div class="opportunity-section education-block pdf-section">
                        <h2>Education at NESCol</h2>
                        <?php the_field('education_description', $id);
                        if ($related) { ?>
                            <ul role="list">
                                <?php
                                wp_reset_postdata();
                                foreach ($related as $post) {
                                    $qualification = get_field('qualification_type', $post->ID); ?>
                                    <li role="listitem"><a href="<?= get_the_permalink($post->ID); ?>"><?= $post->post_title ?> <?php if ($qualification) {
                                                echo '(' . $qualification . ')';
                                            } ?></a></li>
                                    <?php
                                } ?>
                            </ul>
                        <?php } wp_reset_postdata(); ?>
                    </div>
                <?php }
                if (get_field('university_description', $id) || get_field('university_course_opportunities', $id)) { ?>
                    <div class="opportunity-section university-block pdf-section">
                        <h2>University</h2>
                        <?php the_field('university_description', $id);
                        if (have_rows('university_course_opportunities'. $id)) { ?>
                            <ul role="list">
                                <?php while (have_rows('university_course_opportunities', $id)) {
                                    the_row(); ?>
                                    <li role="listitem">
                                        <?php if (get_sub_field('university_url', $id)) { ?>
                                        <a href="<?php the_sub_field('university_url', $id); ?>" target="_blank"><?php } ?>
                                            <?php
                                            if (get_sub_field('university_course', $id)) {
                                                $output = get_sub_field('university_name', $id) . ' - ' . get_sub_field('university_course', $id);
                                            } else {
                                                $output = get_sub_field('university_name', $id);
                                            }
                                            echo $output; ?>
                                            <?php if (get_sub_field('university_url', $id)) { ?></a><?php } ?>
                                    </li>
                                    <?php
                                } ?>
                            </ul>
                        <?php } ?>
                    </div>
                <?php }
                if (get_field('employment_description', $id) || get_field('employment_opportunities', $id)) { ?>
                    <div class="opportunity-section employment-block pdf-section">
                        <h2>Employment</h2>
                        <?php the_field('employment_description', $id);
                        if (have_rows('employment_opportunities', $id)) { ?>
                            <ul role="list">
                                <?php while (have_rows('employment_opportunities', $id)) {
                                    the_row(); ?>
                                    <li role="listitem" class="employment-list-item"><?php the_sub_field('opportunity', $id); ?></li>
                                    <?php
                                } ?>
                            </ul>
                        <?php } ?>
                    </div>
                <?php }

                $infoBoxes = self::getPDFInfoSections($id);
                foreach ($infoBoxes as $box) { ?>
                    <div class="default-sections">
                        <h2><?= $box['title']; ?></h2>
                        <?= $box['content']; ?>
                    </div>
                <?php } ?>

                <hr class="contact-split-bar" style="border-color:<?= $categoryColour; ?>" />
                <div class="contact-message">
                    <?php if (in_array('SMA', $modesOfStudy)) {
                        the_field('sma_contact_message', 'option');
                    } else {
                        the_field('contact_message', 'option');
                    } ?>
                </div>
            </div>
        </div>
        <?php $data = ob_get_contents();
        ob_end_clean();
        return($data);
    }

    public function loadBrochure()
    {
        if (isset($_POST['downloadCourseBrochure'])) {
            $id = $_POST['postid'];
            $title = get_the_title($id);
            $noSpaceTitle = str_replace(' ', '-', $title);
            $formattedTitle = preg_replace('/[^A-Za-z0-9\-]/', '', $noSpaceTitle);
            ob_start();
            $filename = "nescol-" . strtolower($formattedTitle) . '-course-brochure.pdf';
            echo self::createPDFLayout($id);
            $display = ob_get_contents();
            ob_end_clean();
            $this->downloadCoursePDF($filename, $display);
        }
    }

    public function downloadCoursePDF($filename, $display)
    {
        $pdf = $this->generatePDF($display);
        $this->showFile($pdf, $filename);
    }

    private function generatePDF($display)
    {
        $wkhtmltopdf = $this->getPDFPath();
        $tempFile = sys_get_temp_dir() . '/' . uniqid();
        $snappy = new Pdf($_SERVER['DOCUMENT_ROOT'] . $wkhtmltopdf);
        $snappy->generateFromHtml($display, $tempFile);
        $snappy->setOption('user-style-sheet', get_template_directory_uri() . '/dist/site.css');

        return $tempFile;
    }

    private function getPDFPath()
    {
        if ($_SERVER['DOCUMENT_ROOT'] == 'C:/workspace/hubert/web') {
            return $this->htmlToPdfPath = '/../vendor/wemersonjanuario/wkhtmltopdf-windows/bin/64bit/wkhtmltopdf.exe';
        }
        return $this->htmlToPdfPath = '/../vendor/h4cc/wkhtmltopdf-amd64/bin/wkhtmltopdf-amd64';
    }

    private function showFile($file, $filename)
    {
        header("Content-type: application/pdf");
        header("Content-Disposition: attachment; filename=$filename");

        readfile($file);
        exit();
    }

    public static function getCourseImage($id, $size = null)
    {
        $courseCategory = get_the_category($id);
        if (get_the_post_thumbnail_url($id)) {
            $image = get_the_post_thumbnail_url($id, $size);
            $set = 'cat-not-set';
        } elseif ($courseCategory) {
            if (get_field('category_image', 'category_' . $courseCategory[0]->term_id)) {
                $imageID = get_field('category_image', 'category_' . $courseCategory[0]->term_id);
                $imageSource = wp_get_attachment_image_src($imageID, $size);
                $image = $imageSource[0];
                $set = "cat-set";
            } else {
                $courseParentID = $courseCategory[0]->category_parent;
                if (get_field('category_image', 'category_' . $courseParentID)) {
                    $imageID = get_field('category_image', 'category_' . $courseParentID);
                    $imageSource = wp_get_attachment_image_src($imageID, $size);
                    $image = $imageSource[0];
                    $set = "cat-set";
                } else {
                    $image = get_field('default_image_logo', 'option');
                    $set = "cat-not-set";
                }
            }
        } else {
            $image = get_field('default_image_logo', 'option');
            $set = 'cat-not-set';
        }
        return ['image_url' => $image, 'image_format' => $set];
    }

    public static function customCourseSearch($search)
    {
        //Used for searching course instances for course code, post titles of courses and post_tag of courses
        global $wpdb;
        $courseIds = [];

        $wild = '%';
        $find = $search;
        $like = $wild . $wpdb->esc_like($find) . $wild;
        $sqlSearch = $wpdb->prepare("
        SELECT ID, post_title, meta_value, pex_nescoldev_terms.name, pex_nescoldev_term_taxonomy.taxonomy FROM pex_nescoldev_posts
        INNER JOIN pex_nescoldev_postmeta 
            ON (pex_nescoldev_posts.ID = pex_nescoldev_postmeta.post_id) 
        INNER JOIN pex_nescoldev_term_relationships
            ON (pex_nescoldev_posts.ID = pex_nescoldev_term_relationships.object_id)
        INNER JOIN pex_nescoldev_terms
            ON (pex_nescoldev_term_relationships.term_taxonomy_id = pex_nescoldev_terms.term_id)
        INNER JOIN pex_nescoldev_term_taxonomy
            ON (pex_nescoldev_terms.term_id = pex_nescoldev_term_taxonomy.term_id)
        WHERE 1=1 
        AND (pex_nescoldev_posts.post_password = '') 
        AND (pex_nescoldev_posts.post_status = 'publish')
        AND (pex_nescoldev_posts.post_type = 'course') 
        AND 
		  (pex_nescoldev_posts.post_title LIKE '%s'
        OR ((pex_nescoldev_postmeta.meta_key LIKE 'course_instances_%_code' AND CAST(pex_nescoldev_postmeta.meta_value AS CHAR) LIKE '%s'))
        OR (pex_nescoldev_terms.name LIKE '%s' AND pex_nescoldev_term_taxonomy.taxonomy = 'post_tag')
        OR (pex_nescoldev_terms.name LIKE '%s' AND pex_nescoldev_term_taxonomy.taxonomy = 'level'))
        GROUP BY pex_nescoldev_posts.ID 
        ORDER BY pex_nescoldev_posts.post_title ASC", $like, $like, $like, $like);

        $results = $wpdb->get_results($sqlSearch);
        foreach ($results as $result) {
            array_push($courseIds, (int)$result->ID);
        }
        return $courseIds;
    }

    public static function getExcludedCourses() {
        $courseData = [];
        $excludedSMACourses = [];
        global $wpdb;
        $customSQL = "SELECT DISTINCT post_id, MAX(meta_value) AS maxDate, group_concat(pex_nescoldev_terms.slug) as term_slugs FROM pex_nescoldev_postmeta 
 INNER JOIN pex_nescoldev_posts ON (pex_nescoldev_postmeta.post_id = pex_nescoldev_posts.ID)
   INNER JOIN pex_nescoldev_term_relationships
      ON (pex_nescoldev_posts.ID = pex_nescoldev_term_relationships.object_id)
  INNER JOIN pex_nescoldev_terms
      ON (pex_nescoldev_term_relationships.term_taxonomy_id = pex_nescoldev_terms.term_id)
  INNER JOIN pex_nescoldev_term_taxonomy
      ON (pex_nescoldev_terms.term_id = pex_nescoldev_term_taxonomy.term_id)
WHERE (pex_nescoldev_posts.post_password = '') 
AND (pex_nescoldev_posts.post_status = 'publish')
AND (pex_nescoldev_posts.post_type = 'course')
AND meta_key LIKE 'course_instances_%_start_date' GROUP BY post_id
HAVING term_slugs LIKE '%sma%' AND maxDate < NOW()";
        $results = $wpdb->get_results($customSQL);
        foreach($results as $result) {
            array_push($excludedSMACourses, $result->post_id);
        }
        return $excludedSMACourses;
    }

    public static function generateAlphabeticalFilter($activeLetters){
        $characters = range('A', 'Z');
        $activeCharacters = array_keys($activeLetters);
        $numericState = !empty(preg_grep('/[0-9]/',$activeCharacters)) ? 'active' : 'inactive';
        ?>
        <div class="character-filter">
            <div class="container">
                <ul>
                    <li><a href="" class="<?= $numericState; ?>">0-9</a></li>
                    <?php foreach($characters as $character) {
                        $characterState = in_array($character, $activeCharacters) ? 'active' : 'inactive'; ?>
                        <li><a href="" class="<?= $characterState; ?>"><?= $character; ?></a></li>
                    <?php } ?>
                </ul>
            </div>
        </div>
    <?php }

    public function customCourseArchiveTitles($title) {
        if(is_post_type_archive('course')) {
            $paged = get_query_var('paged');

            $interests = isset($_GET['interests']) ? $_GET['interests'] : "";
            $mode = isset($_GET['mode-of-study']) ? $_GET['mode-of-study'] : "";
            $level = isset($_GET['level']) ? $_GET['level'] : "";
            $campus = isset($_GET['campus']) ? $_GET['campus'] : "";
            $keyword = isset($_GET['keyword']) ? $_GET['keyword'] : "";

            if($interests || $mode || $level || $campus || $keyword) {

                if($mode){
                    $name = get_term_by('slug', $mode, 'mode-of-study')->name;
                } elseif($level) {
                    $name = get_term_by('slug', $level, 'level')->name;
                } elseif ($campus) {
                    $name = get_the_title($campus);
                } elseif ($interests) {
                    $name = get_term_by('slug', $interests, 'category')->name;
                } elseif ($keyword) {
                    $name = ucwords($keyword);
                }
                $title = $name . ' Archive';

            } else {
                $title = 'Courses Archive';
            }

            if($paged) {
                $title .= ' - Page ' . $paged . ' - ' . get_bloginfo();
            } else {
                $title .= ' - ' . get_bloginfo();
            }
        }
        return $title;
    }

    public static function getAllCoursesWithInstances() {
        $courseList = [];
        $args = [
            'post_type' => 'course',
            'posts_per_page' => -1,
            'orderby' => 'title',
            'order' => 'ASC'
        ];

        $courses = new \WP_Query($args);

        if($courses->have_posts()){
            while($courses->have_posts()){
                $courses->the_post();

                $courseInstances = get_field('course_instances');
                $modes = [];
                $levels = [];
                $categories = [];

                $modeOfStudies = get_the_terms(get_the_id(), 'mode-of-study');
                $levelsTaxonomy = get_the_terms(get_the_id(), 'level');
                $categoriesTaxonomy = get_the_category(get_the_id());

                $title = get_the_title();
                $postStatus = get_post_status();
                $qualificationType = get_sub_field('qualification_type');

                if($modeOfStudies) {
                    foreach ($modeOfStudies as $mode) {
                        $modes[] = $mode->name;
                    }
                }

                if($categoriesTaxonomy) {
                    foreach ($categoriesTaxonomy as $cat) {
                        $categories[] = $cat->name;
                    }
                }

                if($levelsTaxonomy) {
                    foreach ($levelsTaxonomy as $level) {
                        $levels[] = $level->name;
                    }
                }

                if($courseInstances) {
                    foreach ($courseInstances as $courseInstance) {
                        $courseList[] = [
                            'title' => $title,
                            'code' => $courseInstance['code'],
                            'category' =>  implode(', ', $categories),
                            'mode_of_study' => implode(', ', $modes),
                            'level' => implode(', ', $levels),
                            'qualification_type' => $qualificationType,
                            'post_status' => $postStatus,
                            'show_on_site' => $courseInstance['show_on_site'],
                            'online' => $courseInstance['online'],
                            'start_date' => $courseInstance['start_date'],
                            'end_date' => $courseInstance['end_date'],
                            'duration' => $courseInstance['duration'],
                            'fees' => $courseInstance['fees'],
                        ];
                    }
                }
            }
        }

        return array_unique($courseList, SORT_REGULAR);
    }

    public static function coursesCSVdownload($array) {
        header("Content-Type:application/csv");
        header("Content-Disposition:attachment;filename=courseExport.csv");
        $output = fopen("php://output",'w') or die("Can't open php://output");
        fputcsv($output, ['Title','Code', 'Interests', 'Modes of Study', 'Levels', 'Qualification Type', 'Post Status', 'Show on site', 'Online', 'Start Date', 'End Date', 'Duration', 'Fees']);
        foreach($array as $product) {
            fputcsv($output, $product);
        }
        fclose($output) or die("Can't close php://output");
    }

    public function createCourseExportWidget()
    {
        wp_add_dashboard_widget('courseExport', 'Export a CSV of Courses and their instances', [$this, "exportCoursesButton"]);
    }

    public function exportCoursesButton()
    {
        echo '<form method="post" name="export-courses" id="export-courses" enctype="multipart/form-data"><input type="submit" name="exportCourses" value="Export Courses"></form>';
    }

    public function exportCourseTriggered() {
        if (isset($_POST['exportCourses'])) {
            $courses = $this->getAllCoursesWithInstances();
            $this->coursesCSVdownload($courses);
            exit();
        }
    }

    public static function generateCategoryPDF() {
        $categories = get_the_category();
        if($categories[0]->parent !== 0) {
            $parentID = $categories[0]->parent;
            $category = get_field('progression_path_pdf', 'category_' . $categories[0]->term_id) ? $categories[0] : get_category($parentID); ;
        } else{
            $category = $categories[0];
        }
        $pdf = get_field('progression_path_pdf', 'category_' . $category->term_id);
        if($pdf) { ?>
            <a href="<?= $pdf['url']; ?>" target="_blank">Progression Pathway (PDF)</a>
        <?php }
    }

    public function deleteInstances($post_id, $post, $update) {
        // Only set for post_type = courses!
        if ('course' !== $post->post_type ) {
            return;
        }

        ini_set('memory_limit', '-1');
        ini_set('max_execution_time', '0');

        if (have_rows('course_instances', $post_id)) {
            while (have_rows('course_instances', $post_id)) {
                the_row();
                $startDate = get_sub_field('field_5d5eb43069aec');
                $startDateTime = strtotime($startDate);
                $currentDateTime = strtotime("now");

                if (($startDateTime < $currentDateTime) || !get_sub_field('field_5d5eb43069aec') || get_sub_field('field_5d5eb43069aec') == '') {
                    delete_row('course_instances', get_row_index());
                    continue;
                }
            }
        }
    }

    public static function startUpdateInstances()
    {
        if ( !wp_next_scheduled('updateInstances') ) {
            //run this in 30 seconds time
            wp_schedule_single_event(time() + 30, 'updateInstances');
        }
    }

    public function updatePostInstances() {
        ini_set('memory_limit', '-1');
        ini_set('max_execution_time', '0');

        $args = array(
            'post_type' => 'course',
            'posts_per_page'   => -1,
            'post_status'   =>   'any',   
        );
        $query = new \WP_Query($args);

        while ( $query->have_posts() ) : $query->the_post();
            $post_id = get_the_id();
            if (have_rows('course_instances', $post_id)) {
                while (have_rows('course_instances', $post_id)) {
                    the_row();
                    $startDate = get_sub_field('field_5d5eb43069aec');
                    $startDateTime = strtotime($startDate);
                    $currentDateTime = strtotime("now");
                    if (($startDateTime < $currentDateTime) || !get_sub_field('field_5d5eb43069aec') || get_sub_field('field_5d5eb43069aec') == '') {
                        delete_row('course_instances', get_row_index());
                        continue;
                    }
                }
            }
        endwhile;

        wp_reset_postdata();

        wp_mail( 'shane@blue2.co.uk', 'Instances updated', 'Instances have all been updated' );

        die();
    }
}
