<?php
namespace Blue2\Scruffy;

use Blue2\Scruffy\Controllers\CourseController;
use Blue2\Scruffy\Controllers\DashboardController;
use Blue2\Scruffy\Controllers\EventController;
use Blue2\Scruffy\Controllers\SMAController;

class Site extends \Blue2\Common\Site
{
    public function __construct()
    {
        parent::__construct();
        $this->registerShortCodes();
        $this->registerFilters();
        $this->registerActions();
        $this->setupControllers();
    }

    public function registerMenus()
    {
        register_nav_menus(array(
            'main_menu' => 'Main Menu',
            'external_resources' => 'External Resources',
            'footer_menu_1' => 'Footer Menu 1',
            'footer_menu_2' => 'Footer Menu 2',
            'footer_menu_3' => 'Footer Menu 3',
            'footer_menu_4' => 'Footer Menu 4',
            'footer_menu_5' => 'Footer Menu 5',
            'bottom_bar_menu' => 'Bottom Bar Menu',
            'mobile_menu' => 'Mobile Menu'
        ));
    }

    public function tidyExcerpt()
    {
        return ' ...';
    }

    private function registerShortCodes()
    {
        add_shortcode('dynamic_course_value', [$this, 'dynamicCourseValue']);
        add_shortcode('dynamic_course_mode_of_study', [$this, 'dynamicCourseModeOfStudy']);
        add_shortcode('dynamic_course_level', [$this, 'dynamicCourseLevel']);
    }

    private function setupControllers()
    {
        new CourseController();
        new EventController();
        new DashboardController();
        new SMAController();
    }

    private function registerFilters()
    {
        add_filter('excerpt_more', [$this, 'tidyExcerpt']);
        add_filter('wp_enqueue_scripts', [$this, 'enqueueScripts']);
        add_filter('acf/fields/google_map/api', [$this, 'myAcfGoogleMapApi']);
        add_filter('login_headerurl', [$this, 'customLoginLogoURL']);
        add_filter('posts_search', [$this,'searchByTitle'], 10, 2);
    }

    private function registerActions()
    {
        add_action('after_setup_theme', [$this, 'registerMenus']);
        add_action('after_setup_theme', [$this, 'registerMenus']);
        add_action('init', [$this, 'registerPostTypes']);
        add_action('init', [$this, 'addPostTypeOptionsPage']);
        add_action('init', [$this, 'registerPostTaxonomies']);
        add_action('init', [$this, 'customImageSizes']);
        add_action('pre_get_posts', [$this, 'setPostsPerPage']);
        add_action('wp_enqueue_scripts', [$this, 'localizeScripts'], 101);
        add_action('add_meta_boxes', [$this, 'addPageThemeMetaBox']);
        add_action('save_post', [$this,'pageThemeSavePostData']);
        add_action('login_enqueue_scripts', [$this, 'customLoginStylesheet']);
        add_action('acf/input/admin_head', [$this, 'my_acf_admin_head']);
        add_action('pre_get_posts', [$this, 'comingSoonTagQuery']);
        add_action('init', [$this, 'addTagsToPosts']);
        add_action('pre_get_posts', [$this, 'tagsSupportQuery']);
    }

    public function my_acf_admin_head()
    {
        wp_enqueue_style('acf-styles', get_template_directory_uri() . '/assets/styles/repeater.css');
    }

    public function registerPostTypes()
    {
        Post\Event::registerPostType();
        Post\Location::registerPostType();
        Post\Course::registerPostType();
        Post\Story::registerPostType();
    }

    public function customImageSizes()
    {
        add_image_size('story-image', 100, 100, true);
        add_image_size('news-image', 750, 500, true);
        add_image_size('news-featured-image', 1200, 600, true);
        add_image_size('hero-image', 1920, 750, true);
        add_image_size('home-split-image', 1200, 800, true);
    }

    public function myAcfGoogleMapApi($api)
    {
        $api['key'] = 'AIzaSyBGIcrBeK8X5gOOabNlKvc-C-UdoP59Xvw';
        return $api;
    }

    public function localizeScripts()
    {
        wp_localize_script('b2/js', 'ajaxObject', [ 'ajax_url' => admin_url('admin-ajax.php')]);
    }

    public function enqueueScripts()
    {
        if (is_page_template('template-contact.php')) {
            wp_enqueue_script('google-maps', '//maps.googleapis.com/maps/api/js?key=AIzaSyBGIcrBeK8X5gOOabNlKvc-C-UdoP59Xvw', ['jquery'], null, true);
        }
    }

    public function addSmaMetabox()
    {
        add_meta_box('sma-meta-box', 'Page Theme', 'smaMetaBox', 'page', 'side', 'high');
    }

    public function comingSoonTagQuery($query)
    {
        if ($query->is_main_query() && !is_admin() && $query->is_tag && $query->get('tag') == 'coming-soon') {
            $query->set('post_type', 'course');
        }
    }

    public function addTagsToPosts()
    {
        register_taxonomy_for_object_type('post_tag', 'page');
        register_taxonomy_for_object_type('post_tag', 'event');
        register_taxonomy_for_object_type('post_tag', 'story');
    }

    public function tagsSupportQuery($wp_query)
    {
        if ($wp_query->get('tag')) {
            $wp_query->set('post_type', 'any');
        }
    }

    public function addPostTypeOptionsPage()
    {
        if (function_exists('acf_add_options_page')) {
            acf_add_options_page([
                'title' => 'Course Settings',
                'parent' => 'edit.php?post_type=course',
                'capability' => 'edit_posts',
            ]);
        }
        if (function_exists('acf_add_options_page')) {
            acf_add_options_page([
                'title' => 'Stories Settings',
                'parent' => 'edit.php?post_type=story',
                'capability' => 'edit_posts',
            ]);
        }
    }

    public function searchByTitle($search, $wp_query)
    {
        if (!empty($search) && !empty($wp_query->query_vars['search_terms'])) {
            global $wpdb;
            $queryVars = $wp_query->query_vars;
            $n = !empty($queryVars['exact']) ? '' : '%';
            $search = [];
            foreach ((array) $queryVars['search_terms'] as $term) {
                $search[] = $wpdb->prepare("$wpdb->posts.post_title LIKE %s", $n . $wpdb->esc_like($term) . $n);
            }
            if (!is_user_logged_in()) {
                $search[] = "$wpdb->posts.post_password = ''";
            }
            $search = ' AND ' . implode(' AND ', $search);
        }
        return $search;
    }

    public static function getDefaultPostsPerPage()
    {
        $postsPerPage = get_option('posts_per_page');
        return $postsPerPage;
    }

    public static function getParentCategoryColour($id)
    {
        $courseCategory = get_the_category($id);
        if ($courseCategory) {
            $categoryParentID = $courseCategory[0]->category_parent;
        } else {
            $categoryParentID = "";
        }

        if ($categoryParentID != 0) {
            $categoryParent = get_term($categoryParentID, 'category');
            $categoryColour = get_field('category_colour', 'category_' . $categoryParentID) ? : "#623C7C";
        } else {
            $categoryParent = "";
            $categoryColour = "#623C7C";
        }

        return $categoryColour;
    }

    public static function getParentCategory($id)
    {
        $courseCategory = get_the_category($id);
        if ($courseCategory) {
            $courseParent = $courseCategory[0]->category_parent;
            $courseParent = get_cat_name($courseParent);
        } else {
            $courseParent = "";
        }

        return $courseParent;
    }

    public static function getCourseCount($modeOfStudy, $category)
    {
        $args = [
            'post_type' => 'course',
            'tax_query' => [
                'relation' => 'AND',
                [
                    'taxonomy' => 'mode-of-study',
                    'field' => 'slug',
                    'terms' => $modeOfStudy,
                ],
                [
                    'taxonomy' => 'category',
                    'field' => 'slug',
                    'terms' => $category,
                ]
            ]
        ];
        $countQuery = new \WP_Query($args);
        return $countQuery->found_posts;
    }

    public static function getAllLocations()
    {
        $locationData = [];
        global $wpdb;
        $customSQL = "SELECT DISTINCT meta_value FROM pex_nescol_postmeta WHERE meta_key like 'course_instances_%_location' AND meta_value != ''";
        $results = $wpdb->get_results($customSQL);
        foreach ($results as $result) {
            $title = get_the_title($result->meta_value);
            array_push($locationData, $title);
        }
        return $locationData;
    }

    public static function getAllNonParentCategories()
    {
        $categories = get_categories();
        $catData = [];
        foreach ($categories as $category) {
            if (0 != $category->parent) {
                array_push($catData, $category);
            }
        }
        return $catData;
    }

    public function registerPostTaxonomies()
    {
        $newsType = new \Blue2\Common\Taxonomy("news-type", "News Type", "News Types", "post", [], $args = [
            'show_ui' => true,
            'query_var' => true,
            'show_admin_column' => true,
        ]);
        $modeOfStudy = new \Blue2\Common\Taxonomy("mode-of-study", "Mode of Study", "Mode of Study", "course", [], $args = [
            'show_ui' => true,
            'query_var' => true,
            'show_admin_column' => true,
        ]);
        $locationType = new \Blue2\Common\Taxonomy("location-type", "Location Type", "Location Types", "location", [], $args = [
            'show_ui' => true,
            'query_var' => true,
            'show_admin_column' => true,
        ]);

        $level = new \Blue2\Common\Taxonomy("level", "Level", "Levels", "course", [], $args = [
            'show_ui' => true,
            'query_var' => false,
            'show_admin_column' => true,
        ]);
    }

    public function setPostsPerPage($query)
    {
        if (!is_admin() && $query->is_main_query() && is_post_type_archive(['event', 'story'])) {
            $query->set('posts_per_page', 11);
            return;
        }
        if (!is_admin() && $query->is_main_query() && is_home()) {
            $query->set('posts_per_page', 12);
            return;
        }
        if (!is_admin() && $query->is_main_query() && is_search()) {
            $query->set('posts_per_page', -1);
            return;
        }
    }

    public function dynamicCourseValue()
    {
        global $post;
        $title = get_the_title($post->ID);
        return $title;
    }

    public function dynamicCourseModeOfStudy() {
        global $post;
        $modeOfStudy = get_the_terms($post->ID, 'mode-of-study');
        return $modeOfStudy[0]->name;
    }

    public function dynamicCourseLevel() {
        global $post;
        $qualificationType = get_field('qualification_type', $post->ID);
        $level = get_the_terms($post->ID, 'level');

        if($level && $qualificationType) {
            $output = $qualificationType . ' (' . $level[0]->name . ')';
        } elseif($level) {
            $output = $level[0]->name;
        } else {
            $output = $qualificationType;
        }

        return $output;

    }

    public function addPageThemeMetaBox()
    {
        $screens = ['page'];
        foreach ($screens as $screen) {
            add_meta_box(
                'nescol-page-theme',
                'Page Theme',
                [$this, 'pageThemeHTML'],
                $screen,
                'side',
                'high'
            );
        }
    }

    public function pageThemeSavePostData($post_id)
    {
        if (array_key_exists('page_theme_field', $_POST)) {
            update_post_meta(
                $post_id,
                '_page_theme_meta_key',
                $_POST['page_theme_field']
            );
        }
    }

    public function pageThemeHTML($post)
    {
        $value = get_post_meta($post->ID, '_page_theme_meta_key', true); ?>
        <select name="page_theme_field" id="page_theme_field" class="postbox">
            <option value="nescol" <?php selected($value, 'nescol'); ?> selected>NESCol</option>
            <option value="sma" <?php selected($value, 'sma'); ?>>Scottish Maritime Academy</option>
        </select>
        <?php
    }

    public function customLoginStylesheet()
    {
        wp_enqueue_style('custom-login', get_stylesheet_directory_uri() . '/assets/styles/login.css');
    }

    public function customLoginLogoURL($url)
    {
        return home_url();
    }

    public static function generateRandomString($length = 3)
    {
        return substr(str_shuffle(str_repeat($x='abcdefghijklmnopqrstuvwxyz', ceil($length/strlen($x)))), 1, $length);
    }
}
