<?php

namespace Blue2\Scruffy\Controllers;

use DomDocument;

/**
 * OEmbedController
 *
 * Modifies video oEmbed html to lazy-load videos using iframe srcdoc behaviour
 *
 * @see https://css-tricks.com/lazy-load-embedded-youtube-videos/
 */
class OEmbedController
{
    public function __construct()
    {
        $this->registerActions();
    }

    public function registerActions()
    {
        add_filter('oembed_dataparse', [$this, 'oembedData'], 10, 3);
    }

    public function oembedData($html, $data, $uri)
    {
        // if type is not video, exit early
        if ($data->type !== 'video') return $html;

        // if html produced does not contain iframe, exit early
        if (strpos($html, 'iframe') === false) return $html;

        // embed urls
        $dom = new DomDocument();
        $dom->loadHTML($html);
        $iframe = $dom->getElementsByTagName('iframe')[0];
        $url = $iframe->getAttribute('src');
        $url = $this->formatUrl($data->provider_name, $url);

        return $this->formatIframe([
            'title'     => $data->title,
            'embed_url' => $this->formatSrc($data->provider_name, $url),
            'fallback'  => $url,
            'thumbnail' => $data->thumbnail_url,
            'width'     => $data->width,
            'height'    => $data->height
        ]);
    }

    public function formatIframe($args)
    {
        if ($args['width'] && $args['height']) {
            $aspect_ratio = sprintf(
                ' style="--aspect-ratio: %u/%u;"',
                $args['width'],
                $args['height']
            );
        }
        return sprintf('
            <div class="b-video"%s>
                <iframe
                    title="%s"
                    width="1920"
                    height="1080"
                    src="%s"
                    srcdoc="<style>*{padding:0;margin:0;overflow:hidden}html,body{height:100%%}img,span{position:absolute;width:100%%;top:0;bottom:0}img{height:100%%;object-fit:cover}span{margin:auto;height:1.5em;text-align:center;font:48px/1.5 sans-serif;color:white;text-shadow:0 0 0.5em black}</style><a href=\'%s\'><img src=\'%s\' alt=\'\'><span>&#x25BA;</span></a>"
                    frameborder="0"
                    allow="accelerometer; autoplay; encrypted-media; gyroscope; picture-in-picture"
                    allowfullscreen
                    loading="lazy"
                ></iframe>
            </div>',
            $aspect_ratio ?? '',
            $args['title'] ?: '',
            $args['fallback'] ?: $args['embed_url'],
            $args['embed_url'],
            $args['thumbnail']
        );
    }

    public function formatUrl($provider, $url)
    {
        if ($provider === 'YouTube') {
            $url = str_replace('youtube.com', 'youtube-nocookie.com', $url);
            $url = add_query_arg('modestbranding', '1', $url);
            $url = add_query_arg('rel', '0', $url);
        }

        if ($provider === 'Vimeo') {
            $url = add_query_arg('dnt', '1', $url);
        }

        return $url;
    }

    public function formatSrc($provider, $url)
    {
        if ($provider === 'YouTube') {
            $url = add_query_arg('autoplay', '1', $url);
        }

        if ($provider === 'Vimeo') {
            $url = add_query_arg('autoplay', '1', $url);
            $url = add_query_arg('autopause', '0', $url);
        }

        return $url;
    }
}
