<?php
namespace Blue2\Scruffy;

class Utils extends \Blue2\Common\Theme\Utils
{
    /**
     * Button_Row
     *
     * Outputs row of buttons for use in multiple sections
     *
     * @param  array $button
     * @param  array $args
     */
    public static function Button_Row($buttons, $args = [])
    {
        if (!$buttons) return;

        $default_args = [
            'wrap_class'    => '',
            'button_class'  => ''
        ];
        $args = array_merge($default_args, $args);

        echo '<div class="b-buttonRow '. $args['wrap_class'] . '">';

        foreach ($buttons as $button) {
            self::Button($button, $args);
        }

        echo '</div>';
    }

    /**
     * Button
     *
     * Outputs html of single button
     *
     * @param  array $button
     * @param  array $args
     */
    public static function Button($button, $args = [])
    {
        if (!$button) return;

        $default_args = [
            'button_class'  => ''
        ];
        $args = array_merge($default_args, $args);

        $icon_html = '';
        if ($button['button_icon'] ?? false) {
            $icon_html = sprintf(
                '<span class="%s" aria-hidden="true"></span>',
                $button['button_icon']
            );
        }

        printf(
            '<a class="b-button %s" data-style="%s" href="%s">
                %s
                %s
            </a>',
            $args['button_class'],
            $button['button_style'],
            $button['link']['url'],
            $icon_html,
            $button['link']['title']
        );
    }

    /**
     * acf_img
     *
     * generates consistent and full HTML for acf image array
     *
     * @param  array $image ACF Image array
     * @param  array $args  Arguments including 'class', 'alt', and 'loading'
     * @return string
     */
    public static function acf_img($image, $args = [])
    {
        // if image not specified, return nothing
        if (!$image) return '';

        $default_args = [
            'class'     => false,
            'alt'       => $image['alt'] ?: '',
            'loading'   => 'lazy'
        ];
        $args = array_merge($default_args, $args);

        $img_parts = [];
        $img_parts[] = '<img';

        // class
        if ($args['class']) {
            $img_parts[] = 'class="' . $args['class'] . '"';
        }

        // src
        $img_parts[] = 'src="' . $image['url'] . '"';

        // width
        if ($image['width']) {
            $img_parts[] = 'width="' . $image['width'] . '"';
        }

        // height
        if ($image['height']) {
            $img_parts[] = 'height="' . $image['height'] . '"';
        }

        // alt attribute
        $img_parts[] = 'alt="' . $args['alt'] . '"';

        // loading attribute
        $img_parts[] = 'loading="' . $args['loading'] . '"';

        $img = implode(' ', $img_parts) . '>';
        return $img;
    }

    /**
     * excerpt
     *
     * check if there is an existing excerpt for the post, if not have one created
     *
     * @param  int|null     $id         Page ID (optional)
     * @param  int|null     $cutoff     Length before cutting off content excerpt (optional)
     * @param  string|null  $tail       String to use at end of content excerpt (optional)
     * @return string
     */
    public static function excerpt($id = null, $cutoff = 172, $tail = '&hellip;')
    {
        // if there is a manually set excerpt, use that
        if (has_excerpt()) return get_the_excerpt($id);

        // get excerpt from content, process and trim it
        $source_text = self::excerpt_content($id);
        return self::excerpt_trim($source_text, $cutoff, $tail);
    }

    /**
     * excerpt_content
     *
     * process raw HTML coming from post content, removing tags and adjacent whitespace
     *
     * @param  int|null $id
     * @return string
     */
    public static function excerpt_content($id)
    {
        $excerpt = get_the_content($id);
        $excerpt = wp_strip_all_tags($excerpt);
        $excerpt = preg_replace('/\s+/', ' ', $excerpt);
        return $excerpt;
    }

    /**
     * excerpt_trim
     *
     * trim source text into except length and add ellipses
     *
     * @param  string       $source_text    Text to be processed
     * @param  int|null     $cutoff         Length before cutting off content excerpt
     * @param  string|null  $tail           String to use at end of content excerpt
     * @return string
     */
    public static function excerpt_trim($source_text, $cutoff, $tail)
    {
        $excerpt_length = (int) apply_filters('excerpt_length', $cutoff);
        $excerpt_more = apply_filters('excerpt_more', $tail);
        $text = wp_trim_words($source_text, $excerpt_length, $excerpt_more);

        return $text;
    }
}
