/* global google */

/**
 * googleMap
 *
 * Finds any google maps on page and initialises them using data attributes and children.
 *
 * Map Options:
 *  - data-map: Sets mode of map, changing behaviour
 *      - default:  Will center map on 'lat' and 'lng' options regardless of markers
 *      - bounds:   After markers added the map will be positioned and zoomed to fit markers.
 *                  'zoom' option with 'bounds' defines the maximum zoom level.
 *  - data-lat: Defines latitude of map center. In 'bounds' mode just defines initial center.
 *  - data-lng: Defines longitude of map center. In 'bounds' mode just defines initial center.
 *  - data-zoom: In 'default' mode sets zoom level. In 'bounds' mode sets maximum zoom level.
 *
 * eg: <div data-map="bounds" data-lat="56.46" data-lng="-2.97" data-zoom="13"></div>
 *
 * Markers are defined by child elements to the map with data-marker attribute.
 * Marker Options:
 *  - data-lat:     Latitude of map marker.
 *  - data-lng:     Longitude of map marker.
 *  - data-label:   Label displayed on map pin. A single character is recommended.
 *  - data-focus:   If this marker is clicked, focus and scroll to the element with this ID.
 */
const googleMap = () => {
    const maps = document.querySelectorAll('[data-map]');

    // Check if scrollIntoViewIfNeeded is supported so we can use that instead of scrollIntoView
    const supportsScrollIntoViewIfNeeded = Boolean(document.documentElement.scrollIntoViewIfNeeded);

    // Loop through all maps initialising
    maps.forEach(mapEl => {
        // Parse options from map element
        const lat = parseFloat(mapEl.getAttribute('data-lat'));
        const lng = parseFloat(mapEl.getAttribute('data-lng'));
        const zoom = parseFloat(mapEl.getAttribute('data-zoom'));
        const useBounds = mapEl.getAttribute('data-map') === 'bounds';

        // Loop markers within current mapEl, adding all data attributes to array.
        // Needs to be done before map init as that destroys the marker elements.
        const markers = [];
        mapEl.querySelectorAll('[data-marker]').forEach(markerEl => {
            markers.push(markerEl.dataset);
        });

        // Initialise map
        const map = new google.maps.Map(mapEl, {
            zoom,
            center: new google.maps.LatLng(lat, lng),
            mapTypeId: google.maps.MapTypeId.ROADMAP
        });

        // If 'bounds' mode set, create Bounds object to add markers to.
        let bounds;
        if (useBounds) {
            bounds = new google.maps.LatLngBounds();
        }

        // Loop through marker array, adding them as markers to the map
        markers.forEach(data => {
            // Parse options for lat/lng, set map to be added to, and define our custom icon
            const options = {
                position: new google.maps.LatLng(
                    parseFloat(data.lat),
                    parseFloat(data.lng)
                ),
                map,
                icon: {
                    // We are using font-awesome map-marker solid svg here, inline for convenience.
                    // This has to be encoded for use as a data:image.
                    url: 'data:image/svg+xml;utf-8,%3Csvg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 384 512" height="32" width="24"%3E%3Cpath fill="%237B2C87" d="M172 502C27 291 0 269 0 192a192 192 0 11384 0c0 77-27 99-172 310a24 24 0 01-40 0z"/%3E%3C/svg%3E',
                    // Anchor is center/bottom
                    anchor: new google.maps.Point(12, 32),
                    // Set center point for label, here chosen as slightly above center
                    labelOrigin: new google.maps.Point(12, 14)
                }
            };

            // If label is defined, add it to options object
            if (data.label) {
                options.label = {
                    text: data.label,
                    color: '#fff'
                };
            }

            // Define and add marker
            const marker = new google.maps.Marker(options);

            // If 'bounds' mode set, extend the bounds to include the position of this marker.
            if (useBounds) {
                bounds.extend(marker.getPosition());
            }

            // If 'data-focus' option set, add event listener for click. On click, if the value is
            // a valid ID, focus it and if scrollIntoViewIfNeeded supported scroll if outside viewport.
            // If scrollIntoViewIfNeeded not supported, use scrollIntoView.
            if (data.focus) {
                marker.addListener('click', () => {
                    const focusEl = document.getElementById(data.focus);
                    if (focusEl) {
                        focusEl.focus();
                        if (supportsScrollIntoViewIfNeeded) {
                            focusEl.scrollIntoViewIfNeeded();
                        } else {
                            focusEl.scrollIntoView();
                        }
                    }
                });
            }
        });

        // If 'bounds' mode set, reposition and zoom the map to include all markers.
        // As fitBounds is async, listen for change and then adjust the zoom to add some
        // padding and to ensure zoom is not above set maximum.
        if (useBounds) {
            google.maps.event.addListenerOnce(map, 'bounds_changed', function() {
                const currZoom = this.getZoom();
                const newZoom = Math.min(currZoom - 1, zoom);
                this.setZoom(newZoom);
            });
            map.fitBounds(bounds);
        }
    });
};

/**
 * awaitLoaded
 *
 * if google maps api already loaded then init immediately, else
 * add an event listener to wait for load
 */
const awaitLoaded = () => {
    if (typeof google.maps !== 'undefined') {
        googleMap();
    } else {
        document.documentElement.addEventListener('googleMapsLoad', googleMap, { once: true });
    }
};

export default awaitLoaded;
