<?php
namespace Blue2\Scruffy;

use WPSEO_Primary_Term;

class Utils extends \Blue2\Common\Theme\Utils
{
    /**
     * term_list
     *
     * take taxonomy slug and format all terms of current post in string seperated list
     *
     * @param  string $tax taxonomy slug
     * @return string
     */
    public static function term_list($tax)
    {
        return implode(', ', array_column(get_terms($tax), 'name'));
    }

    /**
     * symbol
     *
     * @param  mixed $id
     * @return void
     */
    public static function symbol($id, $class = '')
    {
        $viewboxes = [
            'shape-left'    => '0 0 150.5 267.4',
            'shape-right'   => '0 0 189.7 360'
        ];
        printf(
            '<svg viewBox="%s" class="%s" aria-hidden="true"><use xlink:href="#dvva-symbol-%s"/></svg>',
            $viewboxes[$id] ?? '0 0 100 100',
            $class,
            $id
        );
    }

    /**
     * acf_img
     *
     * generates consistent and full HTML for acf image array
     *
     * @param  array $image ACF Image array
     * @param  array $args  Arguments including 'class', 'alt', and 'loading'
     * @return string
     */
    public static function acf_img($image, $args = [])
    {
        // if image not specified, return nothing
        if (!$image) return '';

        $default_args = [
            'size'      => 'large',
            'class'     => false,
            'alt'       => $image['alt'] ?: '',
            'loading'   => 'lazy',
            'sizes'     => false
        ];
        $args = array_merge($default_args, $args);

        $img_parts = [];
        $img_parts[] = '<img';

        // class
        if ($args['class']) {
            $img_parts[] = 'class="' . $args['class'] . '"';
        }

        // src
        $img_parts[] = 'src="' . $image['sizes'][$args['size']] . '"';

        // if sizes is specified, add srcset and sizes attributes
        if ($args['sizes']) {
          $srcset = wp_get_attachment_image_srcset($image['ID'], $args['size']);
          if ($srcset) {
            $parts[] = 'srcset="' . $srcset . '"';
            $parts[] = 'sizes="' . implode(', ', $args['sizes']) . '"';
          }
        }

        // width
        if ($image['width']) {
            $img_parts[] = 'width="' . $image['width'] . '"';
        }

        // height
        if ($image['height']) {
            $img_parts[] = 'height="' . $image['height'] . '"';
        }

        // alt attribute
        $img_parts[] = 'alt="' . $args['alt'] . '"';

        // loading attribute
        $img_parts[] = 'loading="' . $args['loading'] . '"';

        $img = implode(' ', $img_parts) . '>';
        return $img;
    }

    /**
     * get_primary_taxonomy_term
     *
     * Returns the primary term for the chosen taxonomy set by Yoast SEO
     * or the first term selected.
     *
     * @see https://www.tannerrecord.com/how-to-get-yoasts-primary-category/
     *
     * @param int       $post   post id (optional)
     * @param string    $tax    taxonomy to query, defaults to category (optional)
     * @return WP_Term  either primary or first term
     */
    public static function get_primary_term($post = false, $tax = 'category')
    {
        if (!$post) {
            $post = get_the_ID();
        }

        $terms = get_the_terms($post, $tax);

        if ($terms) {
            // if Yoast activated
            if (class_exists('WPSEO_Primary_Term')) {
                $wpseo_primary_term = new WPSEO_Primary_Term($tax, $post);
                $wpseo_primary_term = $wpseo_primary_term->get_primary_term();
                $term = get_term($wpseo_primary_term);

                if (!is_wp_error($term)) {
                    return $term;
                }
            }

            return $terms[0];
        }

        return false;
    }

    /**
     * excerpt
     *
     * check if there is an existing excerpt for the post, if not have one created
     *
     * @param  int|null     $id         Page ID (optional)
     * @param  int|null     $cutoff     Length before cutting off content excerpt (optional)
     * @param  bool         $include_p  Should paragraph tags be included automatically
     * @param  string|null  $tail       String to use at end of content excerpt (optional)
     * @return string
     */
    public static function excerpt($id = null, $cutoff = 172, $include_p = true, $tail = '&hellip;')
    {
        // if there is a manually set excerpt, use that
        if (has_excerpt()) {
            $excerpt = self::excerpt_manual($id, $include_p);

        } else {
            // get excerpt from content, process and trim it
            $source_text = self::excerpt_content($id);
            $excerpt = self::excerpt_trim($source_text, $cutoff, $tail);
        }

        if ($include_p) {
            $excerpt = '<p>' . $excerpt . '</p>';
        }

        return $excerpt;
    }

    /**
     * excerpt_manual
     *
     * process manual excerpts where applicable
     *
     * @param  int|null     $id
     * @param  bool         $include_p
     * @return string
     */
    public static function excerpt_manual($id, $include_p)
    {
        $excerpt = get_the_excerpt($id);

        if ($include_p) {
            $excerpt = str_replace("\n", '</p><p>', $excerpt);
        }

        return $excerpt;
    }

    /**
     * excerpt_content
     *
     * process raw HTML coming from post content, removing tags and adjacent whitespace
     *
     * @param  int|null $id
     * @return string
     */
    public static function excerpt_content($id)
    {
        $excerpt = get_the_content($id);
        $excerpt = wp_strip_all_tags($excerpt);
        $excerpt = preg_replace('/\s+/', ' ', $excerpt);
        return $excerpt;
    }

    /**
     * excerpt_trim
     *
     * trim source text into except length and add ellipses
     *
     * @param  string       $source_text    Text to be processed
     * @param  int|null     $cutoff         Length before cutting off content excerpt
     * @param  string|null  $tail           String to use at end of content excerpt
     * @return string
     */
    public static function excerpt_trim($source_text, $cutoff, $tail)
    {
        $excerpt_length = (int) apply_filters('excerpt_length', $cutoff);
        $excerpt_more = apply_filters('excerpt_more', $tail);
        $text = wp_trim_words($source_text, $excerpt_length, $excerpt_more);

        return $text;
    }
}
