<?php
namespace Blue2\Scruffy;

use Blue2\Scruffy\Controllers\InvestorInformationController;

class Site extends \Blue2\Common\Site
{
    public function __construct()
    {
        parent::__construct();

        add_theme_support('post-thumbnails');
        add_action('after_setup_theme', [$this, 'registerMenus']);
        add_action('after_setup_theme', [$this, 'disableAdminBar']);
        add_filter('excerpt_more', [$this, 'tidyExcerpt']);

        $this->registerShortCodes();
        $this->registerFilters();
        $this->registerActions();
        $this->setupControllers();
    }

    public function registerMenus()
    {
        register_nav_menus([
            'main_menu' => 'Main Menu',
            'dc_thomson_footer_menu' => 'DC Thomson - Footer Menu',
            'quick_links_footer_menu' => 'Quick Links - Footer Menu',
            'legal_footer_menu' => 'Legal - Footer Menu'
        ]);
    }

    public function tidyExcerpt()
    {
        return '...';
    }

    private function registerShortCodes()
    {
        // Empty
    }

    private function registerFilters(){
        add_filter( 'excerpt_length', [$this, 'customExcerptLength'], 999 );
        add_filter('acf/fields/google_map/api', [$this, 'myAcfGoogleMapApi']);
        add_filter('wp_enqueue_scripts', [$this, 'enqueueScripts']);
        add_filter('posts_search', [$this, 'advancedSearch'], 500, 2 );

        $enableChangePasswordEmail = get_field('enable_change_password_email', 'option');
        if($enableChangePasswordEmail) {
            add_filter('send_password_change_email', '__return_false');
        }
    }

    private function registerActions() {
        add_action('init', [$this, 'registerPostTypes']);
        add_action('init', [$this, 'customImageSizes']);
        add_action('admin_menu', [$this, 'customAdminPages']);
        add_action('pre_get_posts', [$this, 'newsArchiveAdjustments']);

        add_action('future_to_pending', [$this, 'pendingEmail']);
        add_action('new_to_pending', [$this, 'pendingEmail']);
        add_action('draft_to_pending', [$this, 'pendingEmail']);
        add_action('auto-draft_to_pending', [$this, 'pendingEmail']);
        add_action('login_enqueue_scripts', [$this, 'customLoginStylesheet']);
        add_action('template_redirect', [$this, 'redirectLoggedOutUsers']);
        add_action('init', [$this, 'dashboardRedirect']);
        add_action('wp_dashboard_setup', [$this, 'registerWidgets']);

        add_action('wp_ajax_import_users', [$this, 'importUsers']);
        add_action('wp_ajax_nopriv_import_users', [$this, 'importUsers']);
    }

    public function registerWidgets()
    {
        wp_add_dashboard_widget('importUsers', 'Import Users', [$this, "importUsersButton"]);
    }

    private function setupControllers(){
        new InvestorInformationController();
    }

    public function registerPostTypes()
    {
        Post\Business::registerPostType();
        Post\Quote::registerPostType();
        Post\InvestorInformation::registerPostType();
    }

    public static function getDefaultImage() {
        $path = get_template_directory() . '/dist/images/DCT.svg';
        $file = file_get_contents($path);
        // remove xml header
        $file = str_replace(
            '<?xml version="1.0" encoding="UTF-8"?>',
            '',
            $file
        );
        // add class and attributes for accessibility
        return str_replace(
            '<svg',
            '<svg class="default-image" role="img" aria-label="DC Thomson"',
            $file
        );
    }

    public function customImageSizes()
    {
        add_image_size('hero-image', 632, 632, true);
        add_image_size('primary-business-image', 397, 575, true);
        add_image_size('secondary-business-image', 496, 258, true);
        add_image_size('featured-news', 630, 379, true);
        add_image_size('news-carousel', 323, 194, true);
        add_image_size('primary-quote-image', 603, 651, true);
        add_image_size('secondary-quote-image', 1110, 444, true);
        add_image_size('numbered-carousel', 990, 657, true);
        add_image_size('featured-quote-image', 766, 460, true);
        add_image_size('board-image', 495, 545, true);
        add_image_size('primary-image-section', 952, 571, true);
        add_image_size('secondary-image-section', 476, 476, true);
        add_image_size('people-carousel-image', 414, 455, true);
        add_image_size('single-news-image', 1041, 691, false);
        add_image_size('other-business-image', 295, 0, false);
        add_image_size('pdf-image', 312, 392, true);
        add_image_size('business-secondary-image-1', 476, 690, true);
        add_image_size('business-secondary-image-2', 476, 309, true);
        add_image_size('info-block-section', 793, 535, true);
        add_image_size('timeline-image', 243, 0, false);
        add_image_size('news-archive-featured-image', 766, 461, true);
    }

    public static function acfImage($id, $size, $postID = null) {
        if($id) {
            $imageAlt = get_post_meta($id, '_wp_attachment_image_alt', TRUE);
            $imageDetails = wp_get_attachment_image_src($id, $size);
            $image = '<img src="' . $imageDetails[0] . '" alt="' . $imageAlt . '" loading="lazy"/>';
            $imageClass = 'featured-image';
        } else {
            $image = self::getDefaultImage($postID);
            $imageClass = 'default-image';
        }
        return ['output' => $image, 'class' => $imageClass];
    }

    public function customAdminPages()
    {
        if (function_exists('acf_add_options_page')) {
            acf_add_options_page([
                'title' => 'News Settings',
                'parent' => 'edit.php',
                'capability' => 'edit_posts',
            ]);
            acf_add_options_page([
                'title' => 'Business Settings',
                'parent' => 'edit.php?post_type=business',
                'capability' => 'edit_posts',
            ]);
            acf_add_options_page([
                'title' => 'Investor Information Settings',
                'parent' => 'edit.php?post_type=investor-information',
                'capability' => 'edit_posts',
            ]);
        }
    }

    public function customExcerptLength( $length ) {
        return 20;
    }

    public function newsArchiveAdjustments( $query ) {
        if ( !is_admin() && $query->is_main_query() && is_home() ) {
            $excludesIDs = [];
            $featuredNewsArticle = get_field('featured_news_article', 'option');
            $featuredVideoArticle = get_field('featured_video_article', 'option');
            if($featuredNewsArticle) {
                foreach($featuredNewsArticle as $article) {
                    array_push($excludesIDs, $article->ID);
                };
            }
            if($featuredVideoArticle) {
                foreach($featuredVideoArticle as $article) {
                    array_push($excludesIDs, $article->ID);
                };
            }
            $query->set('post__not_in', $excludesIDs);
        }
    }

    public function myAcfGoogleMapApi($api)
    {
        $api['key'] = 'AIzaSyDw4RXj6mkfuUEQhhWEFZMFwJjDuCNt2xE';
        return $api;
    }

    public function enqueueScripts()
    {
        if (is_page_template('template-contact.php')) {
            wp_enqueue_script('google-maps', '//maps.googleapis.com/maps/api/js?key=AIzaSyDw4RXj6mkfuUEQhhWEFZMFwJjDuCNt2xE', ['jquery'], null, true);
        }
    }


    public function advancedSearch( $search, $wp_query ) {
        if ( empty( $search ))
            return $search;
        $terms = $wp_query->query_vars['s'];
        $exploded = explode( ' ', $terms );
        if( $exploded === FALSE || count( $exploded ) == 0 ) {
            $exploded = array(0 => $terms);
        }
        $search = '';

        foreach( $exploded as $tag ) {
            $search .= " AND (
            (pex_dcthomson_posts.post_title LIKE '%$tag%')
            OR (pex_dcthomson_posts.post_content LIKE '%$tag%')
            OR EXISTS
            (
                SELECT * FROM pex_dcthomson_terms
                INNER JOIN pex_dcthomson_term_taxonomy
                    ON pex_dcthomson_term_taxonomy.term_id = pex_dcthomson_terms.term_id
                INNER JOIN pex_dcthomson_term_relationships
                    ON pex_dcthomson_term_relationships.term_taxonomy_id = pex_dcthomson_term_taxonomy.term_taxonomy_id
                WHERE taxonomy = 'post_tag'
                    AND object_id = pex_dcthomson_posts.ID
                    AND pex_dcthomson_terms.name LIKE '%$tag%'
            )
        )";
        }

        return $search;
    }

    function pendingEmail($post)
    {
        $emails = get_field('pending_review_email_address', 'option');
        $title = wp_strip_all_tags(get_the_title($post->ID));
        $url = get_permalink($post->ID);
        $message = "Link to post: \n{$url}";
        wp_mail($emails, "Pending Post: {$title}", $message);
    }


    function disableAdminBar() {
        if (current_user_can('administrator') || current_user_can('editor') ) {
            // user can view admin bar
            show_admin_bar(true); // this line isn't essentially needed by default...
        } else {
            // hide admin bar
            show_admin_bar(false);
        }
    }

    public function customLoginStylesheet()
    {
        wp_enqueue_style('custom-login', get_stylesheet_directory_uri() . '/assets/styles/login.css');
    }

    public function redirectLoggedOutUsers() {
        if(!is_user_logged_in()) {
            wp_redirect(wp_login_url());
            exit;
        }
    }


    public function dashboardRedirect(){
        if( is_admin() && !defined('DOING_AJAX') && (current_user_can('subscriber')) ){
            wp_redirect(home_url());
            exit;
        }
    }

    public function importUsersButton()
    {
        echo '<form method="post" name="import-users" id="import-users" enctype="multipart/form-data">
                <input type="file" accept=".csv" id="csv-upload" name="importUser" value="Choose File">
                <input type="submit" name="importUsers" value="Import User"></form>';
        ob_start(); ?>
        <script>
            jQuery('body').on('submit', '#import-users', function(e) {
                alert('ok');
                const formData = new FormData();
                const fileData = jQuery('#csv-upload').prop('files')[0];
                formData.append('file', fileData);
                formData.append('action', 'import_users');
                formData.append('nonce', '<?= wp_create_nonce('ajax-nonce'); ?>');
                jQuery.ajax({
                    url: '<?= admin_url('admin-ajax.php'); ?>',
                    type: 'POST',
                    data: formData,
                    processData: false,
                    contentType: false,
                    success : function( data ){
                        console.log(data)
                    }
                });
                e.preventDefault();
            });
        </script>
        <?php $js = ob_get_clean();
        echo $js;
    }


    public function importUsers($file) {
        if (!function_exists('wp_handle_upload')) {
            require_once(ABSPATH . 'wp-admin/includes/file.php');
        }
        $csv = $_FILES['file'];
        if(!$csv) {
            wp_send_json_error('No CSV File');
        }
        $extension = pathinfo($_FILES['file']['name'], PATHINFO_EXTENSION);
        if($extension !== 'csv') {
            wp_send_json_error('Not a valid CSV file');
        }
        $success = [];
        $error = [];
        $data = $_POST;

        // Open file in read mode
        $csvFile = fopen($_FILES['file']['tmp_name'], 'r');
        fgetcsv($csvFile); // Skipping header row
        $length = 7;
        $usersUpdated = 0;
        $usersAdded = 0;

        while(($csvData = fgetcsv($csvFile)) !== FALSE) {
            $csvData = array_map("utf8_encode", $csvData);
            // Row column length
            $dataLen = count($csvData);
            if (!($dataLen == $length)) continue;
            // Assign value to variables
            $id = trim($csvData[0]);
            $userLogin = trim($csvData[1]);
            $userEmail = trim($csvData[2]);
            $userPass = trim($csvData[3]);
            $userRole = trim($csvData[4]);
            $plusMemberStatus = trim($csvData[5]) == '0' ? false : true;
            $plusMemberMetaKey = trim($csvData[6]);

            $userdata = [
                'ID' => $id,    //(int) User ID. If supplied, the user will be updated.
                'user_pass' => $userPass,   //(string) The plain-text user password.
                'user_login' => $userLogin,   //(string) The user's login username.
                'user_email' => $userEmail,   //(string) The user email address.
                'role' => $userRole,   //(string) User's role.
            ];

            $user_id = wp_insert_user( $userdata ) ;
            if (!is_wp_error($user_id)) {
                update_user_meta($user_id, 'plus_member', $plusMemberStatus);
                if($id !== '' && $id > 0) {
                    $usersUpdated++;
                } else{
                    $usersAdded++;
                }
            }
        }
        $success['updated'] = $usersUpdated;
        $success['added'] =  $usersAdded;
        wp_send_json_success($success);
    }
}
