<?php
namespace Blue2\Scruffy;

use Blue2\Scruffy\Controllers\InvestorInformationController;
use Blue2\Scruffy\Controllers\CorporateGovernanceController;

class Site extends \Blue2\Common\Site
{
    public function __construct()
    {
        parent::__construct();

        add_theme_support('post-thumbnails');
        add_action('after_setup_theme', [$this, 'registerMenus']);
        add_action('after_setup_theme', [$this, 'disableAdminBar']);
        add_filter('excerpt_more', [$this, 'tidyExcerpt']);

        $this->registerShortCodes();
        $this->registerFilters();
        $this->registerActions();
        $this->setupControllers();
    }

    public function registerMenus()
    {
        register_nav_menus([
            'main_menu' => 'Main Menu',
            'dc_thomson_footer_menu' => 'DC Thomson - Footer Menu',
            'quick_links_footer_menu' => 'Quick Links - Footer Menu',
            'legal_footer_menu' => 'Legal - Footer Menu'
        ]);
    }

    public function tidyExcerpt()
    {
        return '...';
    }

    private function registerShortCodes()
    {
        // Empty
    }

    private function registerFilters(){
        add_filter( 'excerpt_length', [$this, 'customExcerptLength'], 999 );
        add_filter('acf/fields/google_map/api', [$this, 'myAcfGoogleMapApi']);
        add_filter('wp_enqueue_scripts', [$this, 'enqueueScripts']);
        add_filter( 'posts_search', [$this, 'advancedSearch'], 500, 2 );
    }

    private function registerActions() {
        add_action('init', [$this, 'registerPostTypes']);
        add_action('init', [$this, 'customImageSizes']);
        add_action('admin_menu', [$this, 'customAdminPages']);
        add_action('pre_get_posts', [$this, 'newsArchiveAdjustments']);

        add_action('future_to_pending', [$this, 'pendingEmail']);
        add_action('new_to_pending', [$this, 'pendingEmail']);
        add_action('draft_to_pending', [$this, 'pendingEmail']);
        add_action('auto-draft_to_pending', [$this, 'pendingEmail']);
        add_action('admin_head', [$this, 'customAdminStyles']);
    }

    private function setupControllers(){
        new InvestorInformationController();
        new CorporateGovernanceController();
    }

    public function registerPostTypes()
    {
        Post\Business::registerPostType();
        Post\Quote::registerPostType();
        Post\InvestorInformation::registerPostType();
        Post\CorporateGovernance::registerPostType();
    }

    public static function getDefaultImage() {
        $path = get_template_directory() . '/dist/images/DCT.svg';
        $file = file_get_contents($path);
        // remove xml header
        $file = str_replace(
            '<?xml version="1.0" encoding="UTF-8"?>',
            '',
            $file
        );
        // add class and attributes for accessibility
        return str_replace(
            '<svg',
            '<svg class="default-image" role="img" aria-label="DC Thomson"',
            $file
        );
    }

    public function customImageSizes()
    {
        add_image_size('hero-image', 632, 632, true);
        add_image_size('primary-business-image', 397, 575, true);
        add_image_size('secondary-business-image', 496, 258, true);
        add_image_size('featured-news', 630, 379, true);
        add_image_size('news-carousel', 323, 194, true);
        add_image_size('primary-quote-image', 603, 651, true);
        add_image_size('secondary-quote-image', 1110, 444, true);
        add_image_size('numbered-carousel', 925, 614, true);
        add_image_size('featured-quote-image', 766, 460, true);
        add_image_size('board-image', 495, 545, true);
        add_image_size('primary-image-section', 952, 571, true);
        add_image_size('secondary-image-section', 476, 476, true);
        add_image_size('people-carousel-image', 414, 455, true);
        add_image_size('single-news-image', 1041, 691, false);
        add_image_size('other-business-image', 295, 0, false);
        add_image_size('pdf-image', 312, 392, true);
        add_image_size('business-secondary-image-1', 476, 690, true);
        add_image_size('business-secondary-image-2', 476, 309, true);
    }

    public static function acfImage($id, $size, $postID = null) {
        if($id) {
            $imageAlt = get_post_meta($id, '_wp_attachment_image_alt', TRUE);
            $imageDetails = wp_get_attachment_image_src($id, $size);
            $image = '<img src="' . $imageDetails[0] . '" alt="' . $imageAlt . '" loading="lazy"/>';
            $imageClass = 'featured-image';
        } else {
            $image = self::getDefaultImage($postID);
            $imageClass = 'default-image';
        }
        return ['output' => $image, 'class' => $imageClass];
    }

    public function customAdminPages()
    {
        if (function_exists('acf_add_options_page')) {
            acf_add_options_page([
                'title' => 'News Settings',
                'parent' => 'edit.php',
                'capability' => 'edit_posts',
            ]);
            acf_add_options_page([
                'title' => 'Business Settings',
                'parent' => 'edit.php?post_type=business',
                'capability' => 'edit_posts',
            ]);
            acf_add_options_page([
                'title' => 'Investor Information Settings',
                'parent' => 'edit.php?post_type=investor-information',
                'capability' => 'edit_posts',
            ]);
            acf_add_options_page([
                'title' => 'Corporate Governance Settings',
                'parent' => 'edit.php?post_type=corporate-governance',
                'capability' => 'edit_posts',
            ]);
        }
    }

    public function customExcerptLength( $length ) {
        return 20;
    }

    public function newsArchiveAdjustments( $query ) {
        if ( !is_admin() && $query->is_main_query() && is_home() ) {
            $excludesIDs = [];
            $featuredNewsArticles = get_field('featured_news_articles', 'option');
            if($featuredNewsArticles) {
                foreach($featuredNewsArticles as $article) {
                    array_push($excludesIDs, $article->ID);
                };
                $query->set('post__not_in', $excludesIDs);
            }
        }
    }

    public function myAcfGoogleMapApi($api)
    {
        $api['key'] = 'AIzaSyDw4RXj6mkfuUEQhhWEFZMFwJjDuCNt2xE';
        return $api;
    }

    public function enqueueScripts()
    {
        if (is_page_template('template-contact.php')) {
            wp_enqueue_script('google-maps', '//maps.googleapis.com/maps/api/js?key=AIzaSyDw4RXj6mkfuUEQhhWEFZMFwJjDuCNt2xE', ['jquery'], null, true);
        }
    }


    public function advancedSearch( $search, $wp_query ) {
        if ( empty( $search ))
            return $search;
        $terms = $wp_query->query_vars['s'];
        $exploded = explode( ' ', $terms );
        if( $exploded === FALSE || count( $exploded ) == 0 ) {
            $exploded = array(0 => $terms);
        }
        $search = '';

        foreach( $exploded as $tag ) {
            $search .= " AND (
            (pex_dcthomson_posts.post_title LIKE '%$tag%')
            OR (pex_dcthomson_posts.post_content LIKE '%$tag%')
            OR EXISTS
            (
                SELECT * FROM pex_dcthomson_terms
                INNER JOIN pex_dcthomson_term_taxonomy
                    ON pex_dcthomson_term_taxonomy.term_id = pex_dcthomson_terms.term_id
                INNER JOIN pex_dcthomson_term_relationships
                    ON pex_dcthomson_term_relationships.term_taxonomy_id = pex_dcthomson_term_taxonomy.term_taxonomy_id
                WHERE taxonomy = 'post_tag'
                    AND object_id = pex_dcthomson_posts.ID
                    AND pex_dcthomson_terms.name LIKE '%$tag%'
            )
        )";
        }

        return $search;
    }

    function pendingEmail($post)
    {
        $emails = get_field('pending_review_email_address', 'option');
        $title = wp_strip_all_tags(get_the_title($post->ID));
        $url = get_permalink($post->ID);
        $message = "Link to post: \n{$url}";
        wp_mail($emails, "Pending Post: {$title}", $message);
    }


    function disableAdminBar() {
        if (current_user_can('administrator') || current_user_can('editor') ) {
            // user can view admin bar
            show_admin_bar(true); // this line isn't essentially needed by default...
        } else {
            // hide admin bar
            show_admin_bar(false);
        }
    }


    function customAdminStyles() {
        if(!current_user_can('administrator') || current_user_can('editor')) {
            echo '<style>
                #footer-thankyou {display:none;}
                #adminmenu {display:none;}
                #adminmenuback {display:none;}
                .profile-php .update-nag { display:none; }
                .profile-php .user-url-wrap { display:none; }
                .profile-php .form-table:nth-of-type(1) { display:none; }
                .profile-php .form-table:nth-of-type(2) { display:none; }
                .profile-php .form-table:nth-of-type(4) { display:none; }
                .profile-php #your-profile h2:nth-of-type(1) {display:none;}
                .profile-php #your-profile h2:nth-of-type(2) {display:none;}
                .profile-php #your-profile h2:nth-of-type(4) {display:none;}
        </style>';
        }
    }


}
