import { fontawesome, throttle } from './utils';
import Swiper from 'swiper';

const nav = {
    breakpoint: 992,
    isOpen: false,
    lastWidth: window.innerWidth,
    el: {
        scrollEl:   document.documentElement,
        toggle:     document.querySelector('.l-nav__toggle'),
        nav:        document.querySelector('.l-nav'),
        header:     document.querySelector('.l-header'),
        submenus:   document.querySelectorAll('.l-nav__menu .sub-menu')
    },
    init: () => {
        nav.el.toggle.addEventListener('click', nav.toggle);
        window.addEventListener('resize', nav.resize);

        if (nav.el.submenus) {
            nav.el.submenus.forEach((el, i) => {
                const parent = el.parentElement;
                const link = parent.firstChild;

                parent.addEventListener('mouseover', () => nav.submenuOpen(parent, link, i));
                parent.addEventListener('mouseout', () => nav.submenuClose(parent, link, i));
            });
        }
    },

    toggle: () => {
        const isOpen = nav.el.toggle.getAttribute('data-state') === 'open';
        if (isOpen) {
            nav.close();
        } else {
            nav.open();
        }
    },

    open: () => {
        nav.el.scrollEl.style.overflow = 'hidden';

        nav.el.header.setAttribute('data-state', 'open');

        nav.el.nav.setAttribute('data-state', 'open');

        nav.el.toggle.setAttribute('data-state', 'open');
        nav.el.toggle.setAttribute('aria-expanded', 'true');
        nav.el.toggle.setAttribute('aria-label', 'Close primary navigation');

        nav.el.nav.focus();

        nav.isOpen = true;
    },

    close: () => {
        nav.el.scrollEl.style.overflow = null;

        nav.el.header.setAttribute('data-state', 'closed');

        nav.el.nav.setAttribute('data-state', 'closed');

        nav.el.toggle.setAttribute('data-state', 'closed');
        nav.el.toggle.setAttribute('aria-expanded', 'false');
        nav.el.toggle.setAttribute('aria-label', 'Open primary navigation');

        nav.el.toggle.focus();

        nav.isOpen = false;
    },

    resize: throttle(() => {
        const currentWidth = window.innerWidth;

        // going from below breakpoint to above
        if (nav.lastWidth <= nav.breakpoint && currentWidth > nav.breakpoint) {
            // if nav open, close it
            if (nav.isOpen) {
                nav.close();
            }

            // add submenu aria
            nav.submenuAria(true);
        }

        // going from above breakpoint to below
        if (nav.lastWidth > nav.breakpoint && currentWidth <= nav.breakpoint) {
            // remove submenu aria
            nav.submenuAria(false);
        }

        nav.lastWidth = currentWidth;
    }, 1000)
};

const common = () => {
    nav.init();
    fontawesome();

    if ($('.hero-swiper').length) {
        if ($('.hero-swiper .swiper-slide').length === 1) {
            $('.swiper-wrapper').addClass('disabled');
        }

        new Swiper('.hero-swiper', {
            watchOverflow: true,
            loop: true,
            spaceBetween: 42,
            slidesPerView: 1,
            pagination: {
                el: '.swiper-pagination',
                clickable: true
            },
            navigation: {
                nextEl: '.custom-next',
                prevEl: '.custom-prev',
            },
        });
    }

    if ($('.news-swiper').length) {
        new Swiper('.news-swiper', {
            breakpoints: {
                0: {
                    slidesPerView: 1,
                    spaceBetween: 25
                },
                768: {
                    slidesPerView: 2,
                    spaceBetween: 25
                },
                992: {
                    slidesPerView: 3,
                    spaceBetween: 25
                },
                1200: {
                    slidesPerView: 3,
                    spaceBetween: 55
                },
                1500: {
                    spaceBetween: 55,
                    slidesPerView: 4,
                }
            },
            navigation: {
                nextEl: '.custom-news-next',
                prevEl: '.custom-news-prev',
            },
        });
    }

    if ($('.numbered-slider-swiper').length) {
        new Swiper('.numbered-slider-swiper', {
            pagination: {
                el: '.numbered-slider-pagination',
                clickable: true,
                renderBullet(index, className) {
                    return `<span class='${className}'>${index + 1}</span>`;
                },
            },
            spaceBetween: 50,
            slidesPerView: 1,
        });
    }

    if ($('.business-swiper').length) {
        new Swiper('.business-swiper', {
            effect: 'coverflow',
            loop:true,
            grabCursor: true,
            centeredSlides: true,
            slidesPerView: 1.5,
            coverflowEffect: {
                rotate: 50,
                stretch: 0,
                depth: 300,
                modifier: 1.5,
                slideShadows: false,
            },
            scrollbar: {
                draggable: true,
                el: '.swiper-scrollbar',
                hide: false,
            },
            navigation: {
                nextEl: '.business-showcase-next',
                prevEl: '.business-showcase-prev',
            },
        });
    }

    let timelineSwiper;
    let yearSwiper;

    function initTimelineSlides() {
        timelineSwiper = new Swiper('.tab-pane.active .timeline-swiper', {
            observer: true,
            observeParents: true,
            breakpoints: {
                0: {
                    slidesPerView: 1,
                    spaceBetween: 55
                },
                720: {
                    slidesPerView: 2,
                    spaceBetween: 55
                },
                1200: {
                    slidesPerView: 3,
                    spaceBetween: 55
                },
            },
            navigation: {
                nextEl: '.timeline-next',
                prevEl: '.timeline-prev',
            },
        });
    }


    function initYearSlides() {
        yearSwiper = new Swiper('.tab-pane.active .year-jumps', {
            observer: true,
            observeParents: true,
            breakpoints: {
                0: {
                    slidesPerView: 1,
                    spaceBetween: 10
                },
                720: {
                    slidesPerView: 2,
                    spaceBetween: 10
                },
                768: {
                    slidesPerView: 3,
                    spaceBetween: 10
                },
                1200: {
                    slidesPerView: 4,
                    spaceBetween: 10
                },
                1500: {
                    slidesPerView: 5,
                    spaceBetween: 10
                },
            },
            navigation: {
                nextEl: '.year-next',
                prevEl: '.year-prev',
            },
        });
    }

    function yearSplicer() {
        initYearSlides();
        const yearBlocks = [];
        const timelineJumps = $('.tab-pane.active .timeline-jumps');
        const windowWidth = $(window).width();
        const blockOutput = [];
        const years = JSON.parse(timelineJumps.attr('data-years'));

        if (windowWidth <= 720) {
            console.log(years);
            $.each(years, function(index, value) {
                blockOutput.push(`<div class="swiper-slide"><div class="timeline-jump" data-start-year="${value}">${value}</div></div>`);
            });
        } else {
            let blockSize = 3;
            if (windowWidth <= 1200) {
                blockSize = 2;
            }
            while (years.length > 0) {
                yearBlocks.push(years.splice(0, blockSize));
            }
            //Get First and last items in loop
            $.each(yearBlocks, function(index, value) {
                if (value.length < 2) {
                    return value;
                }
                blockOutput.push(`<div class="swiper-slide"><div class="timeline-jump" data-start-year="${value[0]}">${value[0]} - ${value[value.length - 1]}</div></div>`);
            });
        }
        yearSwiper.removeAllSlides();
        yearSwiper.appendSlide(blockOutput);
    }

    if ($('.timeline-swiper').length) {
        initTimelineSlides();
        yearSplicer();
    }

    $('.timeline-navigation .nav-tabs a').on('shown.bs.tab', function() {
        timelineSwiper.destroy(true);
        timelineSwiper = null;

        yearSwiper.destroy(true);
        yearSwiper = null;

        initTimelineSlides();
        yearSplicer();
    });

    if ($('.timeline-swiper').length) {
        $(window).on('resize', function() {
            yearSplicer();
        });
    }

    $('body').on('click', '.timeline-jump', function() {
        const startYear = $(this).data('start-year');
        $('.tab-pane.active .timeline-swiper .timeline-slide').each(function(index, item) {
            const yearSelector = $(item).find('.year')
                .text();

            if (parseInt(yearSelector, 10) === startYear) {
                console.log('we have found the year:', index);
                timelineSwiper.slideTo(index);
            }
        });
    });
};

export default common;
