<?php

namespace B2;

use B2\Order\Order;
use B2\Order\OrderItem;

class ClickCollectPlugin
{
    private $pluginDirectoryPath;
    private $pluginDirectoryURL;

    public function __construct($pluginDirectoryPath, $pluginDirectoryURL)
    {
        $this->pluginDirectoryPath = $pluginDirectoryPath;
        $this->pluginDirectoryURL = $pluginDirectoryURL;

        $this->setupControllers();
        $this->registerShortCodes();
        $this->registerActions();
        $this->registerFilters();
    }

    private function setupControllers()
    {
        $this->LoginController = new LoginController($this->pluginDirectoryPath);
        $this->LocationController = new LocationController($this->pluginDirectoryPath);
        $this->DashboardController = new DashboardController($this->pluginDirectoryPath);
        $this->OrderController = new OrderController($this->pluginDirectoryPath, $this->pluginDirectoryURL);
    }

    private function registerShortCodes()
    {
        add_shortcode('click-collect-login', [$this->LoginController, 'loginPage']);
        add_shortcode('click-collect-location', [$this->LocationController, 'locationPage']);
        add_shortcode('click-collect-dashboard', [$this->DashboardController, 'DashboardPage']);
    }

    private function registerActions()
    {
        add_action('wp_enqueue_scripts', [$this, 'enqueueAssets']);
        add_action('wp_enqueue_scripts', [$this, 'addjquery']);
    }

    private function registerFilters()
    {
        add_filter('page_template', [$this, 'clickCollectTemplates']);
        add_filter('woocommerce_reports_order_statuses', [$this, 'includeExtraOrderStatuses'], 20, 1);
    }

    public function includeExtraOrderStatuses($statuses)
    {
        return ['processing', 'in-progress', 'completed', 'collected', 'being-collated', 'ready'];
    }


    // Called on plugin activation

    public function addJquery()
    {
        wp_deregister_script('jquery');
        wp_enqueue_script('jquery', 'https://ajax.googleapis.com/ajax/libs/jquery/3.1.1/jquery.min.js', [], null, true);
    }

    public function pluginActivation()
    {
        $this->createPages();
    }

    private function createPages()
    {
        $pageDefinitions = [
            'click-collect-login' => [
                'post_title' => 'Click & Collect Login',
            ],
            'click-collect-location' => [
                'post_title' => 'Click & Collect Location',
            ],
            'click-collect-dashboard' => [
                'post_title' => 'Click & Collect Dashboard',
            ],
        ];

        foreach ($pageDefinitions as $slug => $pageArgs) {
            // Check that the page doesn't exist already
            $query = new \WP_Query('pagename=' . $slug);
            if (!$query->have_posts()) {
                // Add the page using the data from the array above
                $defaultArgs = [
                    'post_name' => $slug,
                    'post_status' => 'publish',
                    'post_type' => 'page',
                    'ping_status' => 'closed',
                    'comment_status' => 'closed',
                ];
                $args = array_merge($pageArgs, $defaultArgs);
                wp_insert_post($args);
            }
        }
    }

    public function enqueueAssets()
    {
        if (is_page('click-collect-dashboard')) {
            wp_enqueue_script('webix-js', '//cdn.webix.com/edge/webix.js', [], '1', true);
            wp_enqueue_style('webix-css', '//cdn.webix.com/edge/webix.css', []);
            wp_enqueue_script('click-collect-js', $this->pluginDirectoryURL . '/assets/scripts/webix-scripts.js', [], '2', true);
        }
        if (is_page(['click-collect-dashboard', 'click-collect-location', 'click-collect-login'])) {
            wp_enqueue_style('click-collect-css', $this->pluginDirectoryURL . '/assets/styles/main.css', [], '1', false);
        }
    }

    public function clickCollectTemplates($page_template)
    {
        if (is_page(['click-collect-login', 'click-collect-location', 'click-collect-dashboard'])) {
            $page_template = $this->pluginDirectoryPath . '/templates/template-click-collect.php';
        }
        return $page_template;
    }

    public function checkUserType()
    {
        return !empty(array_intersect(['butcher', 'administrator'], $this->getUserRoles()));
    }

    public function getUserRoles()
    {
        $roles = '';
        if (is_user_logged_in()) {
            $user = wp_get_current_user();
            $roles = $user->roles;
        }
        return $roles;
    }
}
