<?php

namespace B2;

use B2\Order\Order;
use B2\Order\OrderItem;
use B2\Order\OrderItemMetaData;

class OrderController
{
    private $pluginDirectoryPath;
    private $pluginDirectoryURL;

    public function __construct($pluginDirectoryPath, $pluginDirectoryURL)
    {

        $this->pluginDirectoryPath = $pluginDirectoryPath;
        $this->pluginDirectoryURL = $pluginDirectoryURL;

        $this->registerActions();
        $this->registerFilters();

    }

    private function registerActions()
    {
        add_action('init', [$this, 'registerNewOrderStatus']);

        add_action('rest_api_init', function () {
            register_rest_route('click-collect/v1', '/get-all-orders/', [
                'methods' => 'GET',
                'callback' => [$this, 'getOrders'],
            ]);
        });

        add_action('rest_api_init', function () {
            register_rest_route('click-collect/v1', '/get-orders/(?P<id>\d+)', [
                'methods' => 'GET',
                'callback' => [$this, 'viewOrders'],
            ]);
        });

        add_action('rest_api_init', function () {
            register_rest_route('click-collect/v1', '/set-shop-order-number/(?P<shopNumber>\d+)/(?P<id>\d+)', [
                'methods' => 'GET',
                'callback' => [$this, 'setShopNumber'],
            ]);
        });

        add_action('rest_api_init', function () {
            register_rest_route('click-collect/v1', '/update-order-status/(?P<status>\d+)/(?P<id>\d+)', [
                'methods' => 'GET',
                'callback' => [$this, 'setOrderStatus'],
            ]);
        });

        add_action('rest_api_init', function () {
            register_rest_route('click-collect/v1', '/set-final-price/', [
                'methods' => 'POST',
                'callback' => [$this, 'setFinalPrice'],
            ]);
        });

        add_action('rest_api_init', function () {
            register_rest_route('click-collect/v1', '/get-notes/(?P<id>\d+)', [
                'methods' => 'GET',
                'callback' => [$this, 'getNotes'],
            ]);
        });

        add_action('rest_api_init', function () {
            register_rest_route('click-collect/v1', '/add-note/', [
                'methods' => 'POST',
                'callback' => [$this, 'addNote'],
            ]);
        });

        add_action('rest_api_init', function () {
            register_rest_route('click-collect/v1', '/get-all-orders-with-items/', [
                'methods' => 'POST',
                'callback' => [$this, 'getAllOrdersForPrint'],
            ]);
        });

    }

    private function registerFilters()
    {
        add_filter('wc_order_statuses', [$this, 'addToWoocommerceOrderStatus']);
    }

    public static function getOrders($orderTypes)
    {
        if (class_exists('WC_Order_Query')) {
            $query = new \WC_Order_Query([
                'limit' => -1,
                'orderby' => 'date',
                'order' => 'ASC',
            ]);
            $orders = $query->get_orders();

        }

        if ($orderTypes == 'currentOrders') {
            $orderData = self::getOrderDataAsJson($orders);
        } else {
            $orderData = self::getPastOrderDataAsJson($orders);

        }

        return $orderData;
    }

    public static function getOrderDataAsJson($orders)
    {
        $days = [];
        foreach ($orders as $singleOrder) {

            if ($_GET['location'] === strtolower(self::getShippingDetails($singleOrder)['Pickup Location'])) {
                $pickupDate = self::getShippingDetails($singleOrder)['Pickup Date'];
                $pickupDate = strtotime($pickupDate);
                $newDate = date('l jS \of F', $pickupDate);

                $order = new Order($singleOrder);

                if ($order->getOrderStatus() != "6" && $order->getOrderStatus() != "7" && $order->getOrderStatus() != "8") {

                    if (!isset($days[$pickupDate])) {
                        $days[$pickupDate] = [
                            'date' => $newDate,
                            'orders' => [],
                        ];
                    }

                    $days[$pickupDate]['orders'][] = $order->newOrderData();
                }

            }
        }

        return json_encode($days);
    }

    /**
     * @param $order
     * @return array
     */
    public static function getShippingDetails($order)
    {

        $shippingDataArray = [];
        foreach ($order->get_items('shipping') as $itemId => $shippingObject) {
            $shippingData = $shippingObject->get_data();
            foreach ($shippingData['meta_data'] as $singleShippingData) {
                $singleShippingData = $singleShippingData->get_data();
                switch ($singleShippingData['key']) {
                    case '_pickup_location_name':
                        $shippingDataArray['Pickup Location'] = $singleShippingData['value'];
                        break;
                    case '_pickup_date':
                        $shippingDataArray['Pickup Date'] = $singleShippingData['value'];
                        break;
                    case '_pickup_location_address':
                        $shippingDataArray['Pickup address'] = $singleShippingData['value'];
                        break;
                }
            }
        }
        return $shippingDataArray;
    }

    public static function getPastOrderDataAsJson($orders)
    {
        $days = [];
        foreach ($orders as $singleOrder) {


            if ($_GET['location'] === strtolower(self::getShippingDetails($singleOrder)['Pickup Location'])) {
                $pickupDate = self::getShippingDetails($singleOrder)['Pickup Date'];
                $pickupDate = strtotime($pickupDate);
                $newDate = date('l jS \of F', $pickupDate);

                $order = new Order($singleOrder);

                if (!isset($days[$pickupDate])) {
                    $days[$pickupDate] = [
                        'date' => $newDate,
                        'orders' => [],
                    ];
                }

                $days[$pickupDate]['orders'][] = $order->newOrderData();

            }
        }

        return json_encode($days);
    }

    public static function searchOrders($searchTerm)
    {
        $isPostcode = self::checkPostcode($searchTerm);

        if ($isPostcode === true) {
            $orders = wc_get_orders([
                'billing_postcode' => $searchTerm,
            ]);
        } else {
            $firstNameArgs = [
                'limit' => -1,
                'billing_first_name' => $searchTerm,
            ];
            $firstNameQuery = new \WC_Order_Query($firstNameArgs);
            $firstNameOrders = $firstNameQuery->get_orders();

            $surnameArgs = [
                'limit' => -1,
                'billing_last_name' => $searchTerm,
            ];
            $surnameQuery = new \WC_Order_Query($surnameArgs);
            $surnameOrders = $surnameQuery->get_orders();

            $orders = array_unique(array_merge($firstNameOrders, $surnameOrders), SORT_REGULAR);
        }

        $orderData = self::searchOrderDataAsJson($orders);

        return $orderData;
    }

    public static function checkPostcode($postcode)
    {
        $postcode = strtoupper(str_replace(' ', '', $postcode));
        if (preg_match("/(^[A-Z]{1,2}[0-9R][0-9A-Z]?[\s]?[0-9][ABD-HJLNP-UW-Z]{2}$)/i", $postcode) || preg_match("/(^[A-Z]{1,2}[0-9R][0-9A-Z]$)/i", $postcode)) {
            return true;
        } else {
            return false;
        }
    }

    public static function searchOrderDataAsJson($orders)
    {
        $days = [];
        foreach ($orders as $singleOrder) {

            if ($_GET['location'] === strtolower(self::getShippingDetails($singleOrder)['Pickup Location'])) {
                $pickupDate = self::getShippingDetails($singleOrder)['Pickup Date'];
                $pickupDate = strtotime($pickupDate);
                $newDate = date('l jS \of F', $pickupDate);

                $order = new Order($singleOrder);

                if (!isset($days[$pickupDate])) {
                    $days[$pickupDate] = [
                        'date' => $newDate,
                        'orders' => [],
                    ];
                }
                $days[$pickupDate]['orders'][] = $order->newOrderData();

            }
        }

        return json_encode($days);
    }

    public static function viewOrders($data)
    {

        $orderId = $data->get_params()['id'];
        $order = wc_get_order($orderId);

        $shipping = self::getShippingDetails($order);

        $coupons = self::getCoupons($orderId);

        $orderItems['pickup_date'] = $shipping['Pickup Date'];
        $orderItems['price'] = $order->get_total();
        $orderItems['coupons'] = $coupons;


        $orderItems['orders'] = [];
        foreach ($order->get_items() as $itemKey => $item) {
            $newOrderItem = new OrderItem($item, $itemKey, $orderId);

            $orderItems['orders'][] = $newOrderItem->newOrderItemData();
        };

        return $orderItems;
    }

    public static function getCoupons($orderId)
    {
        $order = wc_get_order($orderId);
        $couponArray = [];

        foreach ($order->get_used_coupons() as $couponName) {
            $couponObject = get_page_by_title($couponName, OBJECT, 'shop_coupon');
            $couponID = $couponObject->ID;
            $couponObject = new \WC_Coupon($couponID);

            $couponArray['title'] = $couponObject->get_code();
            $couponArray['description'] = $couponObject->get_description();
            if ($couponObject->get_discount_type() == 'percent') {
                $couponArray['amount'] = $couponObject->get_amount().'%';
            } else {
                $couponArray['amount'] = '£'.$couponObject->get_amount();
            }

        }
        return $couponArray;

    }


    public static function setShopNumber($data)
    {
        $shopNumber = $data->get_params()['shopNumber'];
        $orderNumber = $data->get_params()['id'];

        if (!add_post_meta($orderNumber, 'shop_id', $shopNumber, true)) {
            update_post_meta($orderNumber, 'shop_id', $shopNumber);
        }

    }

    public static function setFinalPrice($data)
    {

        $finalPrice = $data->get_params()['formValues'];
        $orderNumber = json_decode($data->get_params()['id'])->row;

        if (!add_post_meta($orderNumber, 'final_price', $finalPrice, true)) {
            update_post_meta($orderNumber, 'final_price', $finalPrice);
        }

    }

    public static function setOrderStatus($data)
    {
        $statusNumber = $data->get_params()['status'];
        $orderNumber = $data->get_params()['id'];

        $order = wc_get_order($orderNumber);

        $successMessage = '';

        if ($statusNumber == 1) {
            $order->update_status('wc-processing');
            $successMessage = 'Order Updated to New Order';
        }
        if ($statusNumber == 2) {
            $order->update_status('wc-awaiting');
            $successMessage = 'Order Updated to Awaiting Processing';
        }
        if ($statusNumber == 3) {
            $order->update_status('wc-order-started');
            $successMessage = 'Order Updated to Started';
        }
        if ($statusNumber == 4) {
            $order->update_status('wc-being-collated');
            $successMessage = 'Order Updated to Being Collated';
        }
        if ($statusNumber == 5) {
            $shopNumber = get_post_meta($orderNumber, 'shop_id');
            $finalPrice = get_post_meta($orderNumber, 'final_price');
            if (!$shopNumber || !$finalPrice) {
                $successMessage = 'Order not Updated, please check you have input the shop number and final price';
            } else {
                $order->update_status('wc-ready');
                self::emailCustomerOrder($shopNumber, $order);
                $successMessage = 'Order Updated to Ready for Collection';
            }
        }
        if ($statusNumber == 6) {
            $shopNumber = get_post_meta($orderNumber, 'shop_id');
            $finalPrice = get_post_meta($orderNumber, 'final_price');
            if (!$shopNumber || !$finalPrice) {
                $successMessage = 'Order not Updated, please check you have input the shop number and final price';
            } else {
                self::emailCustomerThankYou($order);
                $order->update_status('wc-collected');
                $successMessage = 'Order has been collected';
            }
        }
        if ($statusNumber == 7) {
            $order->update_status('wc-cancelled');
            $successMessage = 'This order has been marked as cancelled';
        }
        if ($statusNumber == 8) {
            $order->update_status('wc-notcollected');
            $successMessage = 'This order has been marked as not collected';
        }

        return $successMessage;

    }

    public static function emailCustomerOrder($shopNumber, $order)
    {
        $shipping = self::getShippingDetails($order);
        $order = new Order($order);

        $finalPrice = $order->getFinalPrice();
        $orderItems = [];

        foreach ($order->getOrderItems() as $itemKey => $item) {
            $newOrderItem = new OrderItem($item, $itemKey, $order->getOrderID());
            $orderItems[] = $newOrderItem->newOrderItemData();
        };

        $date = strtotime($shipping['Pickup Date']);

        $to = $order->getEmail();
        $subject = 'Your Simon Howie Order is ready to collect at our ' . $shipping['Pickup Location'] . ' branch';
        $message = '
            <div style="background-color:#f7f7f7;padding:10px 100px;text-align:center;">
            <img src="https://s3-eu-west-1.amazonaws.com/blue2digital/pex/simonhowie/2019/05/02152443/SimonHowie-logo-on-white.png" style="text-align:center;display:block;width:300px;padding:15px;height:auto;margin:15px auto;" alt="Simon Howie Logo">        
            <div style="background-color:#ea7600;width:100%;color:#fff;font-size:28px;text-align:center;padding:10px 0;">Your Simon Howie Order</div>
            <div style="background-color:#fff; padding:10px; text-align:left;">
            Dear ' . $order->getName() . '<br>
            This is to confirm your order will be ready to collect as requested on ' . date('l jS \of F Y', $date) . ' at ' . $shipping['Pickup Location'] . '<br>
            Your shop order number is <strong style="color:#ea7600;">' . $shopNumber[0] . '</strong> please quote this when collecting your order.<br>

            The final price for the order is: <strong style="color:#ea7600;">£' . $finalPrice . '</strong>';

        $message .= '<table style="border-collapse: collapse;padding:25px 0;">';
        $message .= '<thead style="font-weight:bold;">';
        $message .= '<tr style="border-bottom:1px solid #000; color: #ea7600;">';
        $message .= '<td style="padding:10px;font-weight:bold;border:1px solid #ddd;">Product Title</td><td style="padding:10px;border:1px solid #ddd;">Weight (g)</td><td style="padding:10px;border:1px solid #ddd;">Pack Size</td><td style="padding:10px;border:1px solid #ddd;">Qty</td>';
        $message .= '</tr>';
        $message .= '</thead>';
        $message .= '<tbody>';
        foreach ($orderItems as $orderItem) {
            $message .= '<tr>';
            $message .= '<td style="padding:10px;border:1px solid #ddd;">' . $orderItem['product_title'] . '</td><td style="padding:10px;border:1px solid #ddd;">' . $orderItem['weight'] . '</td><td style="padding:10px;border:1px solid #ddd;">' . $orderItem['size'] . '</td><td style="padding:10px;border:1px solid #ddd;">' . $orderItem['qty'] . '</td>';
            $message .= '</tr>';
        };
        $message .= '</tbody>';
        $message .= '</table>';
        $message .= '<br>';
        $message .= '<br>';
        $message .= 'For collection times please see our <a href="https://www.thescottishbutcher.com/local-butchers-perth-auchterarder/locations-opening-times/">website</a><br>';
        $message .= 'A problem with your order or its collection? Please contact us at <a href="mailto:collect@simonhowiefoods.co.uk">collect@simonhowiefoods.co.uk</a><br>';
        $message .= 'We look forward to welcoming you back to one of our stores.<br>';
        $message .= 'Many thanks, The Simon Howie Team.<br>';
        $message .= '</div>';
        $message .= '</div>';
        $headers = ['Content-Type: text/html; charset=UTF-8'];

        wp_mail($to, $subject, $message, $headers);
    }

    public static function emailCustomerThankYou($order)
    {

        $order = new Order($order);

        $to = $order->getEmail();
        $subject = 'Thank you for collecting your Simon Howie Order';
        $message = '
        <img src="https://s3-eu-west-1.amazonaws.com/blue2digital/pex/simonhowie/2019/05/02152443/SimonHowie-logo-on-white.png" style="display:block;width:300px;height:auto;margin:15px 0;" alt="Simon Howie Logo">
        Dear ' . $order->getName() . '<br>Thank you for collecting your recent order.<br>';
        $message .= 'We hope that you like our new Click & Collect service, if you have any feedback about our products please leave a review <a href="https://www.thescottishbutcher.com/review-our-products/">here</a>';
        $message .= 'If you have had a problem with your order or its collection, please contact us at <a href="mailto:collect@simonhowiefoods.co.uk">collect@simonhowiefoods.co.uk</a><br>';
        $message .= 'We look forward to welcoming you back to one of our stores.<br>';
        $message .= 'Many thanks, The Simon Howie Team.<br>';

        $headers = ['Content-Type: text/html; charset=UTF-8'];

        wp_mail($to, $subject, $message, $headers);

    }

    public static function registerNewOrderStatus()
    {
        register_post_status('wc-awaiting', [
            'label' => 'Awaiting Processing',
            'public' => true,
            'exclude_from_search' => false,
            'show_in_admin_all_list' => true,
            'show_in_admin_status_list' => true,
            'label_count' => _n_noop('Awaiting processing <span class="count">(%s)</span>', 'Awaiting processing <span class="count">(%s)</span>'),
        ]);
        register_post_status('wc-order-started', [
            'label' => 'Order Started',
            'public' => true,
            'exclude_from_search' => false,
            'show_in_admin_all_list' => true,
            'show_in_admin_status_list' => true,
            'label_count' => _n_noop('Order started <span class="count">(%s)</span>', 'Order started <span class="count">(%s)</span>'),
        ]);
        register_post_status('wc-being-collated', [
            'label' => 'Being Collated',
            'public' => true,
            'exclude_from_search' => false,
            'show_in_admin_all_list' => true,
            'show_in_admin_status_list' => true,
            'label_count' => _n_noop('Being collated <span class="count">(%s)</span>', 'Being collated <span class="count">(%s)</span>'),
        ]);
        register_post_status('wc-ready', [
            'label' => 'Ready for collection',
            'public' => true,
            'exclude_from_search' => false,
            'show_in_admin_all_list' => true,
            'show_in_admin_status_list' => true,
            'label_count' => _n_noop('Ready for collection <span class="count">(%s)</span>', 'Ready for collection <span class="count">(%s)</span>'),
        ]);
        register_post_status('wc-collected', [
            'label' => 'Collected',
            'public' => true,
            'exclude_from_search' => false,
            'show_in_admin_all_list' => true,
            'show_in_admin_status_list' => true,
            'label_count' => _n_noop('Collected <span class="count">(%s)</span>', 'Collected <span class="count">(%s)</span>'),
        ]);
        register_post_status('wc-notcollected', [
            'label' => 'Not Collected',
            'public' => true,
            'exclude_from_search' => false,
            'show_in_admin_all_list' => true,
            'show_in_admin_status_list' => true,
            'label_count' => _n_noop('Not Collected <span class="count">(%s)</span>', 'Not Collected <span class="count">(%s)</span>'),
        ]);
        register_post_status('wc-datechanged', [
            'label' => 'Collection Date Changed',
            'public' => true,
            'exclude_from_search' => false,
            'show_in_admin_all_list' => true,
            'show_in_admin_status_list' => true,
            'label_count' => _n_noop('Collection Date Changed <span class="count">(%s)</span>', 'Collection Date Changed <span class="count">(%s)</span>'),
        ]);
    }

    public static function getNotes($data)
    {
        global $wpdb;

        $orderNumber = $data->get_params()['id'];
        $orderNotes = [];

        $tablePrefixed = $wpdb->prefix . 'comments';
        $results = $wpdb->get_results("
            SELECT *
            FROM $tablePrefixed
            WHERE  `comment_post_ID` = $orderNumber
            AND  `comment_type` LIKE  'order_note'
        ");

        foreach ($results as $note) {
            $orderNotes[] = [
                'note_id' => $note->comment_ID,
                'note_date' => $note->comment_date,
                'note_author' => $note->comment_author,
                'note_content' => $note->comment_content,
            ];
        }

        return $orderNotes;

    }

    public static function addNote($data)
    {
        $noteData = json_decode($data->get_params()['formValues']);

        if (isset($noteData->order_id->row)) {
            $order = wc_get_order($noteData->order_id->row);
        } else {
            $order = wc_get_order($noteData->order_id);
        }

        if ($order) {
            $order->add_order_note($noteData->note);
            $order->save();
            $response = 'success';
        } else {
            $response = 'error';
        }

        return $response;
    }

    public static function getSingleOrderForPrint($orderId)
    {
        $order = wc_get_order($orderId);

        $orderData = self::singleOrderDataForPrint($order);

        return $orderData;
    }

    public static function singleOrderDataForPrint($order)
    {
        $orderData = [];

        $order = new Order($order);
        $orderId = $order->getOrderID();
        if (!isset($orderData[$orderId])) {
            $orderData[$orderId] = [
                'order' => $order->newOrderData(),
                'orderItems' => [],
            ];
        }
        $orderItems = [];
        foreach ($order->getOrderItems() as $itemKey => $item) {
            $newOrderItem = new OrderItem($item, $itemKey, $orderId);
            $orderItems[] = $newOrderItem->newOrderItemData();
        };
        $orderData[$orderId]['orderItems'] = $orderItems;
        return $orderData;
    }

    public static function getAllOrdersForPrint($location, $date, $status)
    {
        $ordersForDate = date('Y-m-d', $date);

        $orderData = [];
        if (class_exists('WC_Order_Query')) {
            $query = new \WC_Order_Query([
                'limit' => -1,
                'orderby' => 'date',
                'order' => 'ASC',
            ]);
            $orders = $query->get_orders();

            $orderData = self::orderDataForPrint($orders, $location, $ordersForDate, $status);
        }
        return $orderData;
    }


    public static function orderDataForPrint($orders, $location, $date, $status)
    {

        $orderData = [];
        foreach ($orders as $singleOrder) {
            $pickupDate = self::getShippingDetails($singleOrder)['Pickup Date'];

            if ($location === strtolower(self::getShippingDetails($singleOrder)['Pickup Location']) && $date === $pickupDate) {
                $order = new Order($singleOrder);
                $orderId = $order->getOrderID();

                if ($status === 'to-be-made') {
                    $statuses = ['5', '6', '7', '8'];
                } else {
                    $statuses = ['6', '7', '8'];
                }


                if (!in_array($order->getOrderStatus(), $statuses)) {

                    if (!isset($orderData[$orderId])) {
                        $orderData[$orderId] = [
                            'order' => $order->newOrderData(),
                            'orderItems' => [],
                        ];
                    }

                    $orderItems = [];
                    foreach ($order->getOrderItems() as $itemKey => $item) {
                        $newOrderItem = new OrderItem($item, $itemKey, $orderId);
                        $orderItems[] = $newOrderItem->newOrderItemData();
                    };

                    $orderData[$orderId]['orderItems'] = $orderItems;
                }
            }
        }
        return $orderData;
    }

    function dateSort($a, $b)
    {
        return strtotime($a) - strtotime($b);
    }

    function addToWoocommerceOrderStatus($orderStatuses)
    {
        $newOrderStatuses = [];
        foreach ($orderStatuses as $key => $status) {
            $newOrderStatuses[$key] = $status;
            if ('wc-processing' === $key) {
                $newOrderStatuses['wc-awaiting'] = 'Awaiting processing';
                $newOrderStatuses['wc-order-started'] = 'Order started';
                $newOrderStatuses['wc-being-collated'] = 'Being collated';
                $newOrderStatuses['wc-ready'] = 'Ready for collection';
                $newOrderStatuses['wc-collected'] = 'Collected';
                $newOrderStatuses['wc-notcollected'] = 'Not Collected';
                $newOrderStatuses['wc-datechanged'] = 'Collection Date Changed';
            }
        }
        return $newOrderStatuses;
    }

}