<?php
namespace Blue2\Scruffy\Controllers;

use Knp\Snappy\Pdf;

class TradeController
{
    public function __construct()
    {
        $this->registerActions();
        $this->registerFilters();

        $minMax = \WC_Min_Max_Quantities::get_instance();
        $minMax->minimum_order_quantity = $this->applyWholesaleMin(0, null);
        $minMax->maximum_order_quantity = $this->applyWholesaleMax(0, null);
    }

    private function registerActions()
    {
        add_action('init', [$this, 'registerTrade']);
        //add_action('pre_get_posts', [$this, 'removeTradeFromSearch']);
        add_action('woocommerce_register_trade_form_start', [$this, 'addTradeFieldsToForm'], 15);
        add_action('woocommerce_register_trade_form_start', [$this, 'addBillingFieldsToForm'], 15);
        add_action('woocommerce_created_customer', [$this, 'saveTradeCustomerFields']); // register/checkout
        add_action('woocommerce_created_customer', [$this, 'saveBillingCustomerFields']); // register/checkout
        add_action('personal_options_update', [$this, 'saveTradeCustomerFields']); // edit own account admin
        add_action('edit_user_profile_update', [$this, 'saveTradeCustomerFields']); // edit other account admin
        add_action('set_user_role', [$this, 'sendActivationEmail'], 10, 3); // edit other account admin
        add_action('woocommerce_save_account_details', [$this, 'saveTradeCustomerFields']); // edit WC account
        add_action('show_user_profile', [$this, 'addAdminFields'], 30); // admin: edit profile
        add_action('edit_user_profile', [$this, 'addAdminFields'], 30); // admin: edit other users
        add_action('woocommerce_add_to_cart', [$this, 'limitCartTotal'], 10, 6);
        //add_action('woocommerce_product_options_general_product_data', [$this, 'addWholesalePrice']);
        //add_action('woocommerce_product_after_variable_attributes', [$this, 'addVariationWholesalePrice'], 99, 3);
        //add_action('woocommerce_process_product_meta', [$this, 'saveWholesalePrice']);
        //add_action('woocommerce_save_product_variation', array($this, 'saveWholesalePrice'));
        add_action('woocommerce_created_customer', [$this, 'notifyAdmin'], 90, 3);
        add_action('init', [$this, 'addPosEndpoints']);
        add_action('parse_query', [$this, 'handlePosEndpoints']);
        //woocommerce_cart_product_price
    }

    public function removeTradeFromSearch($query)
    {
        if (!is_admin() && !TradeController::isTradeCustomer()) {
            $query->set('meta_query', [
                'relation' => 'OR',
                [
                    'key' => 'trade_only',
                    'value' => '1',
                    'compare' => '!='
                ],
                [
                    'key' => 'trade_only',
                    'compare' => 'NOT EXISTS',
                    'value' => ''
                ]
            ]);
        }
    }

    public function addPosEndpoints()
    {
        add_rewrite_endpoint('view-document', EP_PERMALINK | EP_PAGES);
        add_rewrite_endpoint('download-document', EP_PERMALINK | EP_PAGES);
    }

    public function handlePosEndpoints()
    {
        if (current_user_can('administrator') || self::isTradeCustomer()) {
            if (get_query_var('view-document') != '') {
                header('Content-type: application/pdf');
                $this->getSalesSheet(get_query_var('view-document'));
                die;
            }

            if (get_query_var('download-document') != '') {
                $productId = get_query_var('download-document');
                header('Content-Disposition: attachment; filename=' . $productId . '-salessheet.pdf');
                $this->getSalesSheet($productId);
                die;
            }
        }
    }

    public function getSalesSheet($productId)
    {
        $product = wc_get_product($productId);
        ob_start();
        include(locate_template('/templates/trade/sales-sheet.php'));
        $contents = ob_get_clean();
        $this->createPDF($contents);
    }

    private function createPDF($contents)
    {
        if ($_SERVER['HTTP_HOST'] == 'walkers.test') {
            $snappy = new Pdf(get_template_directory() . '/vendor\wemersonjanuario\wkhtmltopdf-windows\bin\64bit\wkhtmltopdf.exe');
        } else {
            $snappy = new Pdf(get_template_directory() . '/vendor/h4cc/wkhtmltopdf-amd64/bin/wkhtmltopdf-amd64');
        }
        $snappy->setOption('margin-top', '0.5cm');
        $snappy->setOption('margin-left', '0.5cm');
        $snappy->setOption('margin-right', '0.5cm');
        $snappy->setOption('margin-bottom', '0.5cm');
        $snappy->setOption('page-size', 'A4');
        echo $snappy->getOutputFromHtml($contents);
    }

    private function registerFilters()
    {
        add_filter('woocommerce_product_get_price', [$this, 'applyWholesalePrice'], 10, 2);
        add_filter('woocommerce_variation_prices_price', [$this, 'applyWholesalePrice'], 10, 2);
        add_filter('woocommerce_product_variation_get_price', [$this, 'applyWholesalePrice'], 10, 2);
        add_filter('woocommerce_variation_is_visible', [$this, 'hideSingleForTrade'], 10, 4);
        add_filter("option_woocommerce_minimum_order_quantity", [$this, 'applyWholesaleMin'], 10, 2);
        add_filter("option_woocommerce_maximum_order_quantity", [$this, 'applyWholesaleMax'], 10, 2);
        add_filter('woocommerce_package_rates', [$this, 'calculateWholesaleShipping'], 999, 2);
        add_filter('woocommerce_shipping_free_shipping_is_available', [$this, 'disableShippingForWholesale']);
        add_filter('woocommerce_update_cart_validation', [$this, 'limitCartTotalOnUpdate'], 10, 4);
        //add_filter('woocommerce_get_price_html', [$this, 'getWholesalePriceHTML'], 10, 2);
        //add_filter('acf/fields/google_map/api', [$this, 'setGoogleAPI']);
    }

    public static function isTradeCustomer($userId = null)
    {
        if ($userId === 0) {
            return false;
        }

        $user = $userId ? get_user_by('id', $userId) : wp_get_current_user();

        return $user && (in_array('trade_customer', $user->roles));// || $user->has_cap('access_trade_section'));
    }

    public static function canAccessPage()
    {
        if (current_user_can('administrator')) {
            return true;
        }

        $tradeOnlyPage = get_field('trade_only_page');

        if ($tradeOnlyPage) {
            return self::isTradeCustomer();
        }

        return true;
    }

    public static function isWholesaleCustomer($userId = null)
    {
        if (self::isTradeCustomer($userId)) {
            $user = $userId ? get_user_by('id', $userId) : wp_get_current_user();

            if (get_field('wholesale_price_lock', 'option')) {
                $countryLock = get_field('wholesale_price_country', 'option');
                $userCountry = get_user_meta($user->ID, 'billing_country', true);
                return $userCountry == $countryLock;
            }
        }
        return false;
    }


    public function applyWholesalePrice($price, $product)
    {
        if (self::isWholesaleCustomer()) {
            if ($product->meta_exists('trade_price')) {
                $wsPrice = $product->get_meta('trade_price');
                return $wsPrice != '' ? $wsPrice : $price;
            }
        }
        return $price;
    }

    public function getWholesalePriceHTML($price, $product)
    {
        if (self::isWholesaleCustomer()) {
            if ($product->meta_exists('trade_price')) {
                $wsPrice = $product->get_meta('trade_price');
                if ($wsPrice != '') {
                    return wc_price($wsPrice) . $product->get_price_suffix();
                }
            }
        }
        return $price;
    }


    public function registerTrade()
    {
        add_role('trade_customer', 'Trade Customer', ['read' => true]);
    }

    public function getTradeCustomerFieldsUS()
    {
        return apply_filters('walkers_trade_fields', [
            'trade_tax_id_number' => [
                'required' => true,
                'type' => 'text',
                'label' => __('Tax ID number', 'walkers'),
                'custom_attributes' => ['required' => 'required'],
            ],
            'trade_phone_number' => [
                'required' => true,
                'type' => 'text',
                'label' => __('Phone Number', 'walkers'),
                'custom_attributes' => ['required' => 'required'],
            ],
            'trade_website' => [
                'type' => 'text',
                'label' => __('Website Address', 'walkers'),
            ],
            'website_checkbox' => [
                'type' => 'checkbox',
                'label' => 'I don\'t have a website', 'walkers',
            ],
            'trade_customer_type' => [
                'type' => 'select',
                'required' => true,
                'custom_attributes' => ['required' => 'required'],
                'label' => __('Customer Type', 'walkers'),
                'options' => ['' => 'Please select',
                    'Grocery' => 'Grocery',
                    'Gifting' => 'Gifting',
                    'Wholesale' => 'Wholesale',
                    'Catering' => 'Catering',
                    'Distributor' => 'Distributor',
                    'Retailer' => 'Retailer',
                    'Other' => 'Other (please specify)']
            ],
            'trade_customer_type_other' => [
                'required' => true,
                'custom_attributes' => ['required' => 'required'],
                'type' => 'text',
                'label' => __('if other please specify', 'walkers'),
            ],
            'trade_agreement' => [
                'type' => 'checkbox',
                'custom_attributes' => ['required' => 'required'],
                'label' => __('Do you agree to our <a href="/trade-terms-conditions/" target="_blank">terms and conditions</a>?', 'walkers'),
                'required' => true,
            ],
        ]);
    }

    public function getTradeCustomerFields()
    {
        return apply_filters('walkers_trade_fields', [
            'trade_market' => [
                'required' => true,
                'custom_attributes' => ['required' => 'required'],
                'type' => 'select',
                'label' => __('Market', 'walkers'),
                'options' => ['' => 'Please select',
                    'uk' => 'UK',
                    'international' => 'International',]
            ],
            'trade_company_number' => [
                'required' => true,
                'custom_attributes' => ['required' => 'required'],
                'type' => 'text',
                'label' => __('Company Registration Number', 'walkers'),
            ],
            'trade_vat_code' => [
                'required' => true,
                'custom_attributes' => ['required' => 'required'],
                'type' => 'text',
                'label' => __('VAT code', 'walkers'),
            ],
            'trade_mobile_number' => [
                'required' => true,
                'custom_attributes' => ['required' => 'required'],
                'type' => 'text',
                'label' => __('Phone Number', 'walkers'),
            ],
            'trade_website' => [
                'required' => true,
                'custom_attributes' => ['required' => 'required'],
                'type' => 'text',
                'label' => __('Website Address', 'walkers'),
            ],
            'trade_customer_type' => [
                'required' => true,
                'custom_attributes' => ['required' => 'required'],
                'type' => 'select',
                'label' => __('Customer Type', 'walkers'),
                'options' => ['' => 'Please select',
                    'Grocery' => 'Grocery',
                    'Gifting' => 'Gifting',
                    'Wholesale' => 'Wholesale',
                    'Catering' => 'Catering',
                    'Distributor' => 'Distributor',
                    'Retailer' => 'Retailer',
                    'Other' => 'Other (please specify)']
            ],
            'trade_customer_type_other' => [
                'type' => 'text',
                'label' => __('if other please specify', 'walkers'),
            ],
            'trade_business_type' => [
                'required' => true,
                'custom_attributes' => ['required' => 'required'],
                'type' => 'select',
                'label' => __('Business type', 'walkers'),
                'options' => ['' => 'Please select',
                    'Sole trader' => 'Sole trader',
                    'Partnership' => 'Partnership',
                    'Limited Partnership' => 'Limited Partnership',
                    'Limited Company' => 'Limited Company',
                    'Corporation' => 'Corporation']
            ],
            'trade_length' => [
                'required' => true,
                'custom_attributes' => ['required' => 'required'],
                'type' => 'text',
                'label' => __('How long have you been trading?', 'walkers'),
            ],
            'trade_agreement' => [
                'type' => 'checkbox',
                'label' => __('Do you agree to our <a href="/trade-terms-conditions/" target="_blank">terms and conditions</a>?', 'walkers'),
                'required' => true,
                'custom_attributes' => ['required' => 'required'],
            ],
        ]);
    }

    public function addTradeFieldsToForm()
    {
        do_action('woocommerce_register_form_start');
        $siteLocation = get_field('current_site', 'option');
        $fields = $siteLocation == 'uk' ? $this->getTradeCustomerFields() : $this->getTradeCustomerFieldsUS();
        foreach ($fields as $key => $field_args) {
            $value = null;
            woocommerce_form_field($key, $field_args, $value);
        }
    }

    public function saveTradeCustomerFields($customer_id)
    {
        $siteLocation = get_field('current_site', 'option');
        $fields = $siteLocation == 'uk' ? $this->getTradeCustomerFields() : $this->getTradeCustomerFieldsUS();

        foreach ($fields as $key => $field_args) {
            $sanitize = isset($field_args['sanitize']) ? $field_args['sanitize'] : 'wc_clean';
            $value = isset($_POST[$key]) ? call_user_func($sanitize, $_POST[$key]) : '';
            update_user_meta($customer_id, $key, $value);
        }
    }

    public function addBillingFieldsToForm()
    {
        global $woocommerce;
        $siteLocation = get_field('current_site', 'option');
        add_filter('woocommerce_billing_fields', [$this, 'updateTradeFormFields']);
        $checkout = $woocommerce->checkout();

        foreach ($checkout->get_checkout_fields('billing') as $key => $field) {
            $noKeyMatch = $siteLocation == 'us' ? $key != 'billing_email' && $key != 'billing_phone' && $key != 'billing_title' : $key != 'billing_email' && $key != 'billing_phone';
            if ($noKeyMatch) {
                woocommerce_form_field($key, $field, $checkout->get_value($key));
            }
        }
    }

    public function saveBillingCustomerFields($customer_id)
    {
        global $woocommerce;
        $siteLocation = get_field('current_site', 'option');
        $checkout = $woocommerce->checkout();
        foreach ($checkout->get_checkout_fields('billing') as $key => $field) {
            $noKeyMatch = $siteLocation == 'us' ? $key != 'billing_email' && $key != 'billing_phone' && $key != 'billing_title' : $key != 'billing_email';
            if ($noKeyMatch) {
                // Condition to add firstname and last name to user meta table
                if ($key == 'billing_first_name' || $key == 'billing_last_name') {
                    $new_key = explode('billing_', $key);
                    update_user_meta($customer_id, $new_key[1], $_POST[$key]);
                }
                update_user_meta($customer_id, $key, $_POST[$key]);
            }
        }
    }

    public function getTotalOrderCases($cart_item_key = null, $quantity = null)
    {
        global $woocommerce;
        $items = $woocommerce->cart->get_cart();

        $totalCases = 0;

        foreach ($items as $item) {
            $type = $item['variation'] ? $item['variation']['attribute_pa_product-size'] : null;
            
            $newQuantity =  $item['key'] == $cart_item_key ? $quantity : $item['quantity'];
            if ($type == 'case') {
                $totalCases += $newQuantity;
            } elseif ($type == 'single') {
                //get variation _packs_per_case
                $case = null;
                $product = wc_get_product($item['data']->get_parent_id());
                foreach ($product->get_children() as $variationId) {
                    $variation = wc_get_product($variationId);
                    $attributes = $variation->get_variation_attributes();
                    if ($attributes['attribute_pa_product-size'] == 'case') {
                        $case = $variationId;
                        break;
                    }
                }
                $packsPerCase = get_post_meta($case, '_packs_per_case', true);

                if (is_numeric($packsPerCase)) {
                    $totalCases += $newQuantity/$packsPerCase;
                }
            }
        }

        return $totalCases;
    }

    public function limitCartTotal($cart_item_key, $product_id, $quantity, $variation_id, $variation, $cart_item_data)
    {
        if (!self::isWholesaleCustomer()) {
            $totalCases = $this->getTotalOrderCases();

            $maxCases = get_option('wc_settings_retail_max_cases', 10);
            
            if ($totalCases > $maxCases) {
                global $woocommerce;
                $woocommerce->cart->remove_cart_item($cart_item_key);
                throw new \Exception(sprintf('<a href="%s" class="button wc-forward">%s</a> %s', wc_get_cart_url(), __('View cart', 'woocommerce'), sprintf(__('You cannot add more than %s cases or equivalent to your cart.', 'woocommerce'), $maxCases)));
            }
        }
    }
    public function limitCartTotalOnUpdate($passed_validation, $cart_item_key, $values, $quantity)
    {
        if (!self::isWholesaleCustomer()) {
            $totalCases = $this->getTotalOrderCases($cart_item_key, $quantity);

            $maxCases = get_option('wc_settings_retail_max_cases', 10);
            if ($totalCases > $maxCases) {
                wc_add_notice(sprintf(__('You cannot add more than %s cases or equivalent to your cart.', 'woocommerce'), $maxCases), 'error');
                return false;
            }
        }
        return $passed_validation;
    }

    public function applyWholesaleMax($value, $option)
    {
        if (self::isWholesaleCustomer()) {
            $maxItems = get_field('wholesale_max_quantity', 'option');
            return $maxItems;
        }
        return $value;
    }

    public function applyWholesaleMin($value, $option)
    {
        if (self::isWholesaleCustomer()) {
            $minItems = get_field('wholesale_min_quantity', 'option');
            return $minItems;
        }
        return $value;
    }

    public function calculateWholesaleShipping($rates, $packages = array())
    {
        $newRates = [];
        $tradeRates = [];

        /*foreach ($rates as $rate) {
            if ($rate->get_label() =='Trade Shipping') {
                $tradeRates[] = $rate;
            } else {
                $newRates[] = $rate;
            }
        }*/

        if (self::isWholesaleCustomer()) {
            foreach ($rates as &$rate) {
                $rate->set_cost(25);
            }
        }

        return $rates;
    }

    public function disableShippingForWholesale($isAvailable)
    {
        if (self::isWholesaleCustomer()) {
            return false;
        }

        return $isAvailable;
    }

    public function addTradeRegForm()
    {
        ob_start();
        get_template_part('templates/parts/trade-registration');
        $content = ob_get_contents();
        ob_end_clean();
        return $content;
    }

    public function addWholesalePrice($variation = false, $variationId = null)
    {
        $id = $variation ? 'trade_price[' . $variationId . ']' : 'trade_price';
        $postId = $variation ? $variationId : get_the_ID();

        $args = array(
            'id' => $id,
            'name' => $id,
            'label' => __('Trade Price (' . get_woocommerce_currency_symbol() . ')', 'woocommerce'),
            'value' => get_post_meta($postId, 'trade_price', true),
            'class' => 'short wc_input_price',
            'desc_tip' => true,
            'description' => __('The Trade price of the product.', 'woocommerce'),
            'wrapper_class' => 'form-row form-row-first',
            'data_type' => 'price',
        );

        woocommerce_wp_text_input($args);
    }

    public function addVariationWholesalePrice($loop, $variation_data, $variation)
    {
        $this->addWholesalePrice(true, $variation->ID);
    }

    public function saveWholesalePrice($post_id)
    {
        $product = wc_get_product($post_id);
        $wholesale = isset($_POST['trade_price']) ? $_POST['trade_price'] : '';
        $wholesale = is_array($wholesale) ? sanitize_text_field(array_pop($wholesale)) : sanitize_text_field($wholesale);
        $product->update_meta_data('trade_price', $wholesale);
        $product->save();
    }

    public function notifyAdmin($customerId, $new_customer_data, $password_generated)
    {
        if (get_user_meta($customerId, 'trade_agreement', true) == 1) {
            ob_start();
            include(locate_template('/woocommerce/emails/new-trade-customer.php'));
            $contents = ob_get_clean();
            $tradeRegistrationRecipient = get_field('registration_email', 'option');
            wp_mail($tradeRegistrationRecipient, 'A customer has registered for a trade account', $contents, "Content-Type: text/html\r\n");
        }
    }

    public function sendActivationEmail($userId, $role, $old_roles)
    {
        $activationSent = get_user_meta($userId, 'activation_sent', true);
        if ($role == 'trade_customer' && $activationSent != '1') {
            $user_info = get_userdata($userId);

            ob_start();
            include(locate_template('/woocommerce/emails/trade-customer-active.php'));
            $contents = ob_get_clean();
            wp_mail($user_info->user_email, 'Your trade account has been activated', $contents, "Content-Type: text/html\r\n");
            update_user_meta($userId, 'activation_sent', true);
        }
    }

    public function addAdminFields()
    {
        $siteLocation = get_field('current_site', 'option');
        $fields = $siteLocation == 'uk' ? $this->getTradeCustomerFields() : $this->getTradeCustomerFieldsUS();
        $user_id = isset($_GET['user_id']) ? (int)$_GET['user_id'] : get_current_user_id(); ?>
        <h2><?php _e('Additional Information', 'wemyss'); ?></h2>
        <table class="form-table" id="iconic-additional-information">
            <tbody>
            <?php foreach ($fields as $key => $field_args) {
            $value = get_user_meta($user_id, $key, true); ?>
                <tr>
                    <th>
                        <label for="<?php echo $key; ?>"><?php echo $field_args['label']; ?></label>
                    </th>
                    <td>
                        <?php $field_args['label'] = false; ?>
                        <?php woocommerce_form_field($key, $field_args, $value); ?>
                    </td>
                </tr>
                <?php
        } ?>
            </tbody>
        </table>
        <?php
    }

    public static function getTradeProducts($type = null)
    {
        $args = [
            'post_type' => 'product',
            'posts_per_page' => '-1',
            'meta_query' => [
                'relation' => 'OR',
                [
                    'key' => 'show_hamper_layout',
                    'value' => 0,
                    'type' => 'BOOLEAN'
                ],
                [
                    'key' => 'show_hamper_layout',
                    'compare' => 'NOT EXISTS'
                ]
            ],
        ];

        if (isset($_GET['search'])) {
            $args['s'] = $_GET['search'];
        }

        if ($type) {
            $args['meta_query'] = [[
                'key' => $type,
                'value' => '',
                'compare' => '='
            ]];
        }

        $products = new \WP_Query($args);

        return $products;
    }

    public static function getTradeAttachments($tradeProducts = null)
    {
        $attachments = ['data' => [], 'count' => 0];
        while ($tradeProducts->have_posts()) {
            $tradeProducts->the_post();
            $galleryImages = get_post_meta(get_the_id(), '_product_image_gallery', true);

            if ($galleryImages) {
                $galleryImageIds = explode(',', $galleryImages);
                $attachments['count'] += count($galleryImageIds);
                $attachment = ['title' => get_the_title(), 'imageIds' => $galleryImageIds];
                $attachments['data'][] = $attachment;
            }
        }
        wp_reset_postdata();
        return $attachments;
    }

    public static function getGallerySection($label, $type)
    {
        $tradeProducts = self::getTradeProducts(null);
        $attachments = self::getTradeAttachments($tradeProducts); ?>
        <div class="row trade-panels__row">
        <p class="trade-row-cat" id="<?= $type ?>"><?= $label ?> <span
                    class="item-total"><?= $attachments['count'] ?></span></p>

        <?php
        $loop = 0;
        foreach ($attachments['data'] as $attachment) {
            foreach ($attachment['imageIds'] as $imageId) {
                $smallInfo = wp_get_attachment_image_src($imageId, 'trade-small');
                $mediumInfo = wp_get_attachment_image_src($imageId, 'trade-medium');
                $largeInfo = wp_get_attachment_image_src($imageId, 'trade-large');

                if ($loop == 8) {
                    ?>
                    </div>
                    <div class="collapse" id="collapse<?= $type ?>">
                    <div class="row trade-panels__row">
                    <?php
                } ?>
                <div class="trade-panel col-12 col-sm-6 col-lg-3">
                    <div class="trade-panel-inner">
                        <div class="trade-panel__img">
                            <img src="<?= $smallInfo[0] ?>"/>
                        </div>
                        <div class="trade-panel__caption">
                            <p class="trade-panel__title"><?= $attachment['title'] ?></p>
                            <a href="<?= $smallInfo[0] ?>" class="block-link" target="_blank">Download Small
                                (960x640)</a>
                            <a href="<?= $mediumInfo[0] ?>" class="block-link" target="_blank">Download Medium
                                (2880x1920)</a>
                            <a href="<?= $largeInfo[0] ?>" class="block-link" target="_blank">Download Large
                                (3600x2400)</a>
                        </div>
                    </div>
                </div>
                <?php
                $loop++;
                if ($loop >= 8 && $loop == $attachments['count']) {
                    echo '</div>';
                }
            }
        } ?>
        </div>
        <div class="row trade-panels__row">
            <a class="show-all" data-toggle="collapse" href="#collapse<?= $type ?>" role="button" aria-expanded="false"
               aria-controls="collapse<?= $type ?>">Show all<br><i class="far fa-chevron-down"></i></a>
        </div>
        <?php
    }

    public static function getPhotographySection($label, $type)
    {
        $tradeProducts = self::getTradeProducts($type);

        $count = 0;
        $attachments = [];
        while ($tradeProducts->have_posts()) {
            $tradeProducts->the_post();
            $images = get_field($type);

            $attachments[] = ['title' => get_the_title(), 'images' => $images];
            $count += count($images);
        }
        wp_reset_postdata(); ?>
    <div class="row trade-panels__row">
        <p class="trade-row-cat" id="<?= $type ?>"><?= $label ?> <span class="item-total"><?= $count ?></span></p>

        <?php
        $loop = 0;
        foreach ($attachments

        as $attachment) {
            foreach ($attachment['images'] as $imageId) {
                $smallInfo = wp_get_attachment_image_src($imageId, 'trade-small');
                $mediumInfo = wp_get_attachment_image_src($imageId, 'trade-medium');
                $largeInfo = wp_get_attachment_image_src($imageId, 'trade-large');

                if ($loop == 8) {
                    ?>
    </div>
    <div class="collapse" id="collapse<?= $type ?>">
    <div class="row trade-panels__row">
        <?php
                } ?>
        <div class="trade-panel col-12 col-sm-6 col-lg-3">
            <div class="trade-panel-inner">
                <div class="trade-panel__img">
                    <img src="<?= $smallInfo[0] ?>"/>
                </div>
                <div class="trade-panel__caption">
                    <p class="trade-panel__title"><?= $attachment['title'] ?></p>
                    <a href="<?= $smallInfo[0] ?>" class="block-link" target="_blank">Download Small (960x640)</a>
                    <a href="<?= $mediumInfo[0] ?>" class="block-link" target="_blank">Download Medium (2880x1920)</a>
                    <a href="<?= $largeInfo[0] ?>" class="block-link" target="_blank">Download Large (3600x3240)</a>
                </div>
            </div>
        </div>
        <?php
        $loop++;
                if ($loop >= 8 && $loop == $count) {
                    echo '</div>';
                }
            }
        } ?>
    </div>
    <div class="row trade-panels__row">
        <a class="show-all" data-toggle="collapse" href="#collapse<?= $type ?>" role="button" aria-expanded="false"
           aria-controls="collapse<?= $type ?>">Show all<br><i class="far fa-chevron-down"></i></a>
    </div>
    <?php
    }

    public static function getDocumentSection($label, $type)
    {
        $tradeProducts = self::getTradeProducts(); ?>

        <div class="row trade-panels__row">
        <p class="trade-row-cat" id="<?= $type ?>"><?= $label ?> <span
                    class="item-total"><?= $tradeProducts->found_posts ?></span></p>

        <?php
        $loop = 0;
        while ($tradeProducts->have_posts()) {
            $tradeProducts->the_post();

            $thumbnailId = get_post_thumbnail_id(get_the_id());
            $imageObject = wp_get_attachment_image_src($thumbnailId, 'large-card-image');
            $image = $imageObject[0];

            if ($loop == 8) {
                ?>
                </div>
                <div class="collapse" id="collapse<?= $type ?>">
                <div class="row trade-panels__row">
                <?php
            } ?>
            <div class="trade-panel col-12 col-sm-6 col-lg-3">
                <div class="trade-panel-inner">
                    <div class="trade-panel__img">
                        <img src="<?= $image ?>"/>
                    </div>
                    <div class="trade-panel__caption">
                        <p class="trade-panel__title"><?= get_the_title() ?></p>
                        <a href="download-document/<?= get_the_ID() ?>" target="_blank">Download</a>
                        <a href="view-document/<?= get_the_ID() ?>" target="_blank">View PDF</a>
                    </div>
                </div>
            </div>
            <?php
            $loop++;
            if ($loop == $tradeProducts->found_posts) {
                echo '</div>';
            }
        }
        wp_reset_postdata(); ?>
        </div>

        <div class="row trade-panels__row">
            <a class="show-all" data-toggle="collapse" href="#collapse<?= $type ?>" role="button" aria-expanded="false"
               aria-controls="collapse<?= $type ?>">Show all<br><i class="far fa-chevron-down"></i></a>
        </div>

        <?php
    }

    public function hideSingleForTrade($visible, $variationId, $parentId, $variation)
    {
        if (get_field('trade_hide_single', 'option') && self::isWholesaleCustomer()) {
            $attributes = $variation->get_variation_attributes();
            if ($attributes['attribute_pa_product-size'] !== 'case') {
                return false;
            }
        }

        return $visible;
    }

    public function updateTradeCountries($fields)
    {
        $countries = new \WC_Countries();
        $countries = $countries->__get('countries');
        $fields['billing']['billing_country'] = array(
            'type'      => 'select',
            'required' => true,
            'label'     => __('Country / Region', 'woocommerce'),
            'options'   => $countries,
            'custom_attributes' => ['required' => 'required']
        );
        return $fields;
    }


    public function updateTradeFormFields($fields)
    {
        add_filter('woocommerce_checkout_fields', [$this, 'updateTradeCountries']);
        $siteLocation = get_field('current_site', 'option');
        $fields['billing_company']['required'] = true;

        $fields['billing_first_name']['custom_attributes'] = ['required' => 'required'];
        $fields['billing_last_name']['custom_attributes'] = ['required' => 'required'];
        $fields['billing_country']['custom_attributes'] = ['required' => 'required'];
        $fields['billing_postcode']['custom_attributes'] = ['required' => 'required'];
        $fields['billing_company']['custom_attributes'] = ['required' => 'required'];
        $fields['billing_address_1']['custom_attributes'] = ['required' => 'required'];
        $fields['billing_city']['custom_attributes'] = ['required' => 'required'];

        if ($siteLocation == 'us') {
            $fields['billing_city']['label'] = "City";
            $fields['billing_postcode']['label'] = "Zip Code";
        }
        return $fields;
    }
}
