<?php

class Blue2_Borealis_Api_Service
{
    private $company;
    private $instance;
    private $api_key;

    public function __construct()
    {
        $this->company = BLUE2_BOREALIS_API_COMPANY_NAME;
        $this->instance = BLUE2_BOREALIS_API_INSTANCE;
        $this->api_key = BLUE2_BOREALIS_API_KEY;
    }

    public function checkIndividualExists(string $email)
    {
        $endpoint = $this->getIndividualsEndpoint();

        $queryStringArgs = [
            'filter' => "email:{$email}",
        ];
        $queryString = http_build_query($queryStringArgs);

        $endpoint .= "?{$queryString}";

        $args = $this->getCurlArgs($endpoint, 'GET');
        $responseData = $this->runCurlRequest($args);

        if (isset($responseData['total']) && $responseData['total'] > 0) {
            // Should return ID of individual
            return $responseData['data'][0]['sysId'];
        }

        return false;
    }

    public function createIndividual(array $data)
    {
        $args = $this->getCurlArgs($this->getIndividualsEndpoint(), 'POST', $data);
        $responseData = $this->runCurlRequest($args);

        // Should return ID of individual
        return $responseData['sysId'];
    }

    public function updateIndividual(int $individualId, array $data)
    {
        $args = $this->getCurlArgs($this->getIndividualsEndpoint() . "/{$individualId}", 'POST', $data);
        $responseData = $this->runCurlRequest($args);

        return $responseData;
    }

    public function checkOrganisationExists(string $name)
    {
        $endpoint = $this->getOrganisationsEndpoint();

        $queryStringArgs = [
            'filter' => "organisationName:{$name}",
        ];
        $queryString = http_build_query($queryStringArgs);

        $endpoint .= "?{$queryString}";

        $args = $this->getCurlArgs($endpoint, 'GET');
        $responseData = $this->runCurlRequest($args);

        if (isset($responseData['total']) && $responseData['total'] > 0) {
            // Should return ID of organisation
            return $responseData['data'][0]['sysId'];
        }

        return false;
    }

    public function createOrganisation(array $data)
    {
        $args = $this->getCurlArgs($this->getOrganisationsEndpoint(), 'POST', $data);
        $responseData = $this->runCurlRequest($args);

        // Should return ID of organisation
        return $responseData['sysId'];
    }

    public function createCommunication(array $data)
    {
        $args = $this->getCurlArgs($this->getCommunicationsEndpoint(), 'POST', $data);
        $responseData = $this->runCurlRequest($args);

        return $responseData;
    }

    public function addIndividualToDistributionList(int $individualId, int $distributionListId)
    {
        $args = $this->getCurlArgs($this->getDistributionListEndpoint($distributionListId), 'POST', [
            'baseStakeholderIdAdd' => $individualId,
        ]);
        $responseData = $this->runCurlRequest($args);

        // Should return ID of distribution list
        return $responseData;
    }

    public function addIndividualToEngagementPlan(int $individualId, int $planId)
    {
        $args = $this->getCurlArgs($this->getEngagementPlansEndpoint($planId), 'POST', [
            'stkCatEpStkTypeId' => null,
            'addBaseStakeholderIdMulti' => [$individualId]
        ]);
        $responseData = $this->runCurlRequest($args);

        // Should return ID of engagement plan
        return $responseData;
    }

    public function addIndividualToOrganisation(int $organisationId, array $data)
    {
        $url = $this->getOrganisationsEndpoint() . "/{$organisationId}/members";
        $args = $this->getCurlArgs($url, 'POST', $data);
        $responseData = $this->runCurlRequest($args);

        return $responseData;
    }

    public function addContactDataToOrganisation(int $organisationId, array $data)
    {
        $args = $this->getCurlArgs($this->getOrganisationsEndpoint() . "/{$organisationId}/contacts", 'POST', $data);
        $responseData = $this->runCurlRequest($args);

        return $responseData;
    }

    private function getIndividualsEndpoint(): string
    {
        return "stakeholders_engagement~register~individuals";
    }

    private function getOrganisationsEndpoint(): string
    {
        return "stakeholders_engagement~register~organisations";
    }

    private function getCommunicationsEndpoint(): string
    {
        return "stakeholders_engagement~consultations~communications";
    }

    private function getDistributionListEndpoint(int $distributionListId): string
    {
        return "stakeholders_engagement~stk_administration_tools~distribution_lists/{$distributionListId}/stakeholders";
    }

    private function getEngagementPlansEndpoint(int $planId): string
    {
        return "stakeholders_engagement~planning~engagement_plans/{$planId}/stakeholders";
    }

    private function getUri()
    {
        return "https://{$this->company}.boreal-is.com/api/v2/{$this->instance}/";
    }

    private function getCurlArgs(string $endpoint, string $method, array $data = [])
    {
        $url = $this->getUri() . $endpoint;

        $args = [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_CUSTOMREQUEST => $method,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'api_key: ' . $this->api_key
            ],
        ];

        if ($method === 'POST') {
            $args[CURLOPT_POST] = true;
            $args[CURLOPT_POSTFIELDS] = json_encode($data);
        }

        return $args;
    }

    private function runCurlRequest(array $args)
    {
        $curl = curl_init();

        curl_setopt_array($curl, $args);

        $response = curl_exec($curl);

        // Check response is valid
        if ($response === false) {
            throw new \Exception(curl_error($curl), curl_errno($curl));
        }

        // Check response status code
        $statusCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);

        // If status code is not 200 or 201, return false
        if ($statusCode !== 200 && $statusCode !== 201) {
            throw new \Exception("Request failed with status code {$statusCode}: {$response}");
        }

        return json_decode($response, true);
    }
}
