<?php

/**
 * The public-facing functionality of the plugin.
 *
 * @link       https://blue2.co.uk
 * @since      1.0.4
 *
 * @package    Blue2_Borealis_Api
 * @subpackage Blue2_Borealis_Api/public
 */

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the public-facing stylesheet and JavaScript.
 *
 * @package    Blue2_Borealis_Api
 * @subpackage Blue2_Borealis_Api/public
 * @author     Blue2 <ross@blue2.co.uk>
 */
class Blue2_Borealis_Api_Public
{

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * The API service class for the plugin.
	 * 
	 * @since    1.0.0
	 * @access   private
	 * @var      Blue2_Borealis_Api_Service    $service    The API service class for the plugin.
	 */
	private $service;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of the plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct($plugin_name, $version)
	{
		$this->plugin_name = $plugin_name;
		$this->version = $version;
		$this->service = new Blue2_Borealis_Api_Service();
	}

	public function send_project_updates_form_data_to_borealis($form_data)
	{
		$fieldsRequired = [
			'first_name' => "firstName",
			'last_name' => "lastName",
			'email' => "onAddEmail",
			'phone_number' => "onAddPhonenumber",
			'organisation' => false,
			'job_title' => false,
			'postcode' => "onAddPostcode",
		];
		$formFields = $form_data['fields'];

		$parsedData = [];
		$individualData = [];
		foreach ($formFields as $formField) {
			if (isset($fieldsRequired[$formField['key']])) {
				$parsedData[$formField['key']] = $formField['value'];

				if ($fieldsRequired[$formField['key']] !== false) {
					$borealisKey = $fieldsRequired[$formField['key']];

					if ($borealisKey) {
						$individualData[$borealisKey] = $formField['value'];
					}
				}
			}
		}

		// Check if individual exists
		$individualId = $this->service->checkIndividualExists($individualData['onAddEmail']);

		// Create individual if not exists
		if (!$individualId) {
			$individualId = $this->service->createIndividual($individualData);
		}

		// Check if the Site URL is equal to bellrockwind.co.uk
		// If not, then it'll be broadshorewind.co.uk
		$siteUrl = get_site_url();
		if (str_contains($siteUrl, 'bellrockwind.co.uk')) {
			$primaryEngagementPlanId = 12;
		} else {
			$primaryEngagementPlanId = 11;
		}

		// Add Stakeholder to the Primary Engagement Plan
		try {
			$this->service->addIndividualToEngagementPlan($individualId, $primaryEngagementPlanId);
		} catch (Exception $e) {
			error_log($e->getMessage());
		}

		// Check is organisation exists
		if (isset($parsedData['organisation']) && $parsedData['organisation']) {
			$organisationId = $this->service->checkOrganisationExists($parsedData['organisation']);

			if (!$organisationId) {
				$organisationId = $this->service->createOrganisation([
					'organisationName' => $parsedData['organisation']
				]);
			}

			$postData = [
				'baseIndividualId' => $individualId
			];

			if (isset($parsedData['job_title']) && $parsedData['job_title']) {
				$postData['positionName'] = $parsedData['job_title'];
			}

			$this->service->addIndividualToOrganisation($organisationId, $postData);
		}
	}

	public function send_supplier_registration_form_data_to_borealis($form_data)
	{
		$fieldsRequired = [
			'first_name' => "firstName",
			'last_name' => "lastName",
			'email' => "onAddEmail",
			'telephone_number' => "onAddPhonenumber",
			'company' => false,
			'company_website' => false,
			'company_description' => false,
			'project_phases_where_interest_is_expressed' => false,
			'services_where_interest_is_expressed' => false,
			'services_where_interest_is_expressed_other' => false,
			'primary_location_where_supplies_and_personnel_will_originate' => false,
			'scottish_supply_chain' => false,
			'registered_office_address' => "onAddAddress",
			'registered_office_postcode' => "onAddPostcode",
		];
		$formFields = $form_data['fields'];

		$parsedData = [];
		$individualData = [];
		foreach ($formFields as $formField) {
			if (isset($fieldsRequired[$formField['key']])) {
				$parsedData[$formField['key']] = $formField['value'];

				if ($fieldsRequired[$formField['key']] !== false) {
					$borealisKey = $fieldsRequired[$formField['key']];

					if ($borealisKey) {
						$individualData[$borealisKey] = $formField['value'];
					}
				}
			}
		}

		$individualId = $this->service->checkIndividualExists($individualData['onAddEmail']);

		if (!$individualId) {
			$individualId = $this->service->createIndividual($individualData);
		}

		// Add Stakeholder to the Supply Chain Registration Portal Engagement Plan
		try {
			$this->service->addIndividualToEngagementPlan($individualId, 15);
		} catch (Exception $e) {
			error_log($e->getMessage());
		}

		$communicationText = "";
		if (isset($parsedData['company_description']) && $parsedData['company_description']) {
			$communicationText .= "{$parsedData['company_description']}";
		}
		if (isset($parsedData['services_where_interest_is_expressed']) && $parsedData['services_where_interest_is_expressed']) {
			$services = is_array($parsedData['services_where_interest_is_expressed']) ? $parsedData['services_where_interest_is_expressed'] : explode(",", $parsedData['services_where_interest_is_expressed']);

			// Make sure all services are integers
			$services = array_map(function ($service) {
				return intval(trim($service));
			}, $services);

			try {
				$this->service->updateIndividual($individualId, [
					'stkCatIssueCategoryId' => $services
				]);
			} catch (Exception $e) {
				error_log($e->getMessage());
			}
		}

		$organisationId = $this->service->checkOrganisationExists($parsedData['company']);

		if (!$organisationId) {
			$postData = [
				'organisationName' => $parsedData['company'],
				'onAddAddress' => $parsedData['registered_office_address'],
				'onAddPostcode' => $parsedData['registered_office_postcode'],
				'email' => $parsedData['email'],
			];

			if ($parsedData['company_description']) {
				$postData['description'] = $communicationText;
			}

			$organisationId = $this->service->createOrganisation($postData);
		}

		if ($organisationId) {
			try {
				$this->service->addIndividualToOrganisation($organisationId, [
					'baseIndividualId' => $individualId
				]);
			} catch (Exception $e) {
				error_log($e->getMessage());
			}
		}

		if ($organisationId && isset($parsedData['company_website']) && $parsedData['company_website']) {
			$this->service->addContactDataToOrganisation($organisationId, [
				'contactDetails' => $parsedData['company_website'],
				'baseCatContactTypeId' => 4
			]);
		}

		if ($organisationId && isset($parsedData['telephone_number']) && $parsedData['telephone_number']) {
			$this->service->addContactDataToOrganisation($organisationId, [
				'contactDetailsPhone' => $parsedData['telephone_number'],
				'baseCatContactTypeId' => 16
			]);
		}
	}
}
