<?php

/**
 * The public-facing functionality of the plugin.
 *
 * @link       https://blue2.co.uk
 * @since      1.0.0
 *
 * @package    Blue2_Borealis_Api
 * @subpackage Blue2_Borealis_Api/public
 */

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the public-facing stylesheet and JavaScript.
 *
 * @package    Blue2_Borealis_Api
 * @subpackage Blue2_Borealis_Api/public
 * @author     Blue2 <ross@blue2.co.uk>
 */
class Blue2_Borealis_Api_Public
{

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * The API service class for the plugin.
	 * 
	 * @since    1.0.0
	 * @access   private
	 * @var      Blue2_Borealis_Api_Service    $service    The API service class for the plugin.
	 */
	private $service;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of the plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct($plugin_name, $version)
	{
		$this->plugin_name = $plugin_name;
		$this->version = $version;
		$this->service = new Blue2_Borealis_Api_Service();
	}

	public function send_project_updates_form_data_to_borealis($form_data)
	{
		$fieldsRequired = [
			'firstname' => "firstName",
			'lastname' => "lastName",
			'email' => "onAddEmail",
			'phone_number' => "onAddPhonenumber",
			'organisation' => false,
			'job_title' => false,
			'postcode' => "onAddPostcode",
		];
		$formFields = $form_data['fields'];

		$individualData = [];
		foreach ($formFields as $formField) {
			foreach (array_keys($fieldsRequired) as $requiredField) {
				if (str_starts_with($requiredField, $formField['key'])) {
					$borealisKey = $fieldsRequired[$requiredField];

					if ($borealisKey) {
						$individualData[$borealisKey] = $formField['value'];
					}
				}
			}
		}

		// Check if individual exists
		$individualId = $this->service->checkIndividualExists($individualData['onAddEmail']);

		// Create individual if not exists
		if (!$individualId) {
			$individualId = $this->service->createIndividual($individualData);
		}

		// Add Stakeholder to the Bellrock Engagement Plan
		$this->service->addIndividualToEngagementPlan($individualId, 1);

		// Add Stakeholder to the Broadshore Engagement Plan
		$this->service->addIndividualToEngagementPlan($individualId, 2);

		// Check is organisation exists
		if (isset($formFields['']) && $formFields['organisation']) {
			$organisationId = $this->service->checkOrganisationExists($formFields['organisation']);

			if (!$organisationId) {
				$organisationId = $this->service->createOrganisation([
					'organisationName' => $formFields['organisation']
				]);
			}

			$postData = [
				'baseIndividualId' => $individualId
			];

			if (isset($formFields['']) && $formFields['job_title']) {
				$postData['positionName'] = $formFields['job_title'];
			}

			$this->service->addIndividualToOrganisation($organisationId, $postData);
		}
	}

	public function send_supplier_registration_form_data_to_borealis($form_data)
	{
		$fieldsRequired = [
			'firstname' => "firstName",
			'lastname' => "lastName",
			'email' => "onAddEmail",
			'telephone_number' => "onAddPhonenumber",
			'company' => false,
			'company_website' => false,
			'company_description' => false,
			'tick_this_box_if_you_would_like_to_express_interest_in_all_our_uk_projects' => false,
			'project_phases_where_interest_is_expressed' => false,
			'services_where_interest_is_expressed' => false,
			'services_where_interest_is_expressed_other' => false,
			'primary_location_where_supplies_and_personnel_will_originate' => false,
			'scottish_supply_chain' => false,
			'registered_office_address' => "onAddAddress",
			'registered_office_postcode' => "onAddPostcode",
		];
		$formFields = $form_data['fields'];

		$individualData = [];
		foreach ($formFields as $formField) {
			foreach (array_keys($fieldsRequired) as $requiredField) {
				if (str_starts_with($requiredField, $formField['key'])) {
					$borealisKey = $fieldsRequired[$requiredField];

					if ($borealisKey) {
						$individualData[$borealisKey] = $formField['value'];
					}
				}
			}
		}

		$individualId = $this->service->checkIndividualExists($individualData['onAddEmail']);

		if (!$individualId) {
			$individualId = $this->service->createIndividual($individualData);
		}

		// Check if the Site URL is equal to bellrockwind.co.uk
		// If not, then it'll be broadshorewind.co.uk
		$siteUrl = get_site_url();
		if (str_contains($siteUrl, 'bellrockwind.co.uk')) {
			$primaryEngagementPlanId = 1;
			$secondaryEngagementPlanId = 2;
		} else {
			$primaryEngagementPlanId = 2;
			$secondaryEngagementPlanId = 1;
		}

		// Add Stakeholder to the Primary Engagement Plan
		$this->service->addIndividualToEngagementPlan($individualId, $primaryEngagementPlanId);

		if (isset($formFields['tick_this_box_if_you_would_like_to_express_interest_in_all_our_uk_projects']) && boolval($formFields['tick_this_box_if_you_would_like_to_express_interest_in_all_our_uk_projects'])) {
			// Add Stakeholder to the Secondary Engagement Plan
			$this->service->addIndividualToEngagementPlan($individualId, $secondaryEngagementPlanId);
		}

		$communicationText = "";
		if (isset($formFields['project_phases_where_interest_is_expressed']) && $formFields['project_phases_where_interest_is_expressed']) {
			$phases = explode(",", $formFields['project_phases_where_interest_is_expressed']);
			$phasesText = implode(", ", array_map(function ($phase) {
				return trim($phase);
			}, $phases));
			$communicationText .= "Project Phases where Interest is Expressed: {$phasesText}\n";
		}
		if (isset($formFields['services_where_interest_is_expressed']) && $formFields['services_where_interest_is_expressed']) {
			$services = explode(",", $formFields['services_where_interest_is_expressed']);
			$servicesText = implode(", ", array_map(function ($service) {
				return trim($service);
			}, $services));
			$communicationText .= "Services where Interest is Expressed: {$servicesText}\n";
		}
		if (isset($formFields['services_where_interest_is_expressed_other']) && $formFields['services_where_interest_is_expressed_other']) {
			$communicationText .= "Services where Interest is Expressed (Other): {$formFields['services_where_interest_is_expressed_other']}\n";
		}
		if (isset($formFields['primary_location_where_supplies_and_personnel_will_originate']) && $formFields['primary_location_where_supplies_and_personnel_will_originate']) {
			$communicationText .= "Primary Location where Supplies and Personnel will Originate: {$formFields['primary_location_where_supplies_and_personnel_will_originate']}\n";
		}
		if (isset($formFields['scottish_supply_chain']) && $formFields['scottish_supply_chain']) {
			$communicationText .= "Scottish Supply Chain: {$formFields['scottish_supply_chain']}\n";
		}

		$organisationId = $this->service->checkOrganisationExists($formFields['company']);

		if (!$organisationId) {
			$postData = [
				'organisationName' => $formFields['company'],
				'onAddAddress' => $formFields['registered_office_address'],
				'onAddPostcode' => $formFields['registered_office_postcode'],
				'email' => $formFields['email'],
				'phone' => $formFields['telephone_number'],
			];

			if ($formFields['company_description']) {
				// $postData['description'] = $communicationText;
				$postData['description'] = $formFields['company_description'];
			}

			$organisationId = $this->service->createOrganisation($postData);

			$this->service->addIndividualToOrganisation($organisationId, [
				'baseIndividualId' => $individualId
			]);
		}

		if (isset($formFields['company_website']) && $formFields['company_website']) {
			$this->service->addContactDataToOrganisation($organisationId, [
				'contactDetails' => $formFields['company_website'],
				'baseCatContactTypeId' => 4
			]);
		}
	}
}
