<?php

namespace Blue2\Scruffy\Controllers;

class StoreController
{
    public function __construct()
    {
        add_action('rest_api_init', function () {
            register_rest_route('simon-howie/v1', '/store-locations', [
                'methods' => 'POST',
                'callback' => [$this, 'getStoreLocations'],
            ]);
        });

        add_action('wp_dashboard_setup', [$this, 'uploadStoresCSV']);

        if (!wp_next_scheduled('sh_limited_products_remove')) 
        {
            wp_schedule_event( strtotime('00:05:00'), 'daily', 'sh_limited_products_remove' );
        }

        add_action( 'sh_limited_products_remove', [$this, 'removeLimitedProducts']);

        //we need to stop the store picking a date thats too far in the past
        add_action('acf/input/admin_footer', [$this, 'restrict_acf_date']);

        //add our JS for Ajax
        add_action('admin_footer', [$this, 'ajax_file_upload']);
        //Our ajax call
        add_action("wp_ajax_import_stock", [$this, 'importStock']);
        add_action("wp_ajax_nopriv_import_stock", [$this, 'importStock']);
    }

    public static function getStoreTerms()
    {
        $terms = get_terms(['taxonomy' => 'group']);
        foreach ($terms as $term) { ?>
            <option value="<?= $term->term_id ?>"><?= $term->name ?></option>
        <?php }
    }

    public static function outputSupermarketProductCategories()
    {
        $includeTerms = 'all';
        if (get_field('supermarket_category_taxonomy', 'option')) {
            $includeTerms=get_field('supermarket_category_taxonomy', 'option');
        }
        $supermarketTerm = get_term_by('name', 'supermarket', 'product_cat');
        $supermarketProductCategories = get_terms('product_cat', [
            'orderby' => 'name',
            'order' => 'asc',
            'hide_empty' => false,
            'parent' => $supermarketTerm->term_id,
            'include' => $includeTerms
        ]);
        foreach ($supermarketProductCategories as $category) { ?>
            <option value="<?= $category->term_id ?>"><?= $category->name ?></option>
        <?php }
    }

    public function getStoreLocations($request)
    {
        $params = $request->get_params()['requestParams'];
        if (!$params['search']) {
            return null;
        }
        $coordinates = $this->findCoordinates($params);
        if (!$coordinates) {
            return null;
        }

        $response = [
            'stores' => [],
            'products' => [],
        ];

        $args = [
            'post_type' => 'store',
            'status' => 'publish;',
            'posts_per_page' => -1,
        ];

        $stores = [];
        //Limit by store group
        if (isset($params['store']) && $params['store'] !== 0 && is_numeric($params['store'])) {
            $stores = $params['store'];
            if (!empty($stores)) {
                $taxQuery = [['taxonomy' => 'group', 'field' => 'term_id', 'terms' => $stores]];
                $args['tax_query'] = $taxQuery;
            }
        }

        if (isset($params['product_range']) && $params['product_range'] !== '0') {
            $productQueryArgs = [
                'post_type' => 'product',
                'ignore_sticky_posts' => 1,
                'posts_per_page' => -1,
                'fields' => 'ids',
                'tax_query' => [
                    [
                        'taxonomy' => 'product_cat',
                        'terms' => $params['product_range'],
                        'operator' => 'IN',
                    ],
                ],
            ];
            $products = new \WP_Query($productQueryArgs);

            if (!empty($products->posts)) {
                $baseMetaQuery = [
                    'relation' => 'OR',
                ];
                foreach ($products->posts as $productId) {
                    array_push($baseMetaQuery, [
                        'key' => 'products',
                        'value' => $productId,
                        'compare' => 'LIKE',
                    ]);

                    array_push($response['products'], [
                        'id' => $productId,
                        'permalink' => get_the_permalink($productId),
                        'image' => get_the_post_thumbnail_url($productId),
                        'name' => get_the_title($productId)
                    ]);
                }
                $args['meta_query'] = $baseMetaQuery;
            }
        }
        $allStores = new \WP_Query($args);
        if ($allStores->have_posts()) {
            $searchLat = $coordinates['lat'];
            $searchLng = $coordinates['lng'];

            while ($allStores->have_posts()) {
                $allStores->the_post();
                $location = get_field('location');
                $distance = $this->distanceCalculate($searchLat, $searchLng, $location['lat'], $location['lng']);
                $allowedDistance = $distance <= 15 ? true : false;

                if ($allowedDistance === true) {
                    $store = get_the_terms(get_the_ID(), 'group');
                    $thumbnail = get_field('image', 'group_' . $store['0']->term_id);
                    $thumbnail = wp_get_attachment_image($thumbnail, 'large');
                    $limitedTimeOffer = '';

                    if (have_rows('limited_offers')) {
                        while (have_rows('limited_offers')) : the_row();

                        $product = get_sub_field('limited_product');
                        $product_type = $product->ID;
                        $start = strtotime(get_sub_field('limited_time_start'));
                        $end = strtotime(get_sub_field('limited_time_end'));
                        $currentDate = strtotime(date("Y-m-d"));

                        if (($currentDate >= $start) && ($currentDate <= $end)) {
                            $limitedTimeOffer.='<div class="limited-time-container tool-tip-container product-'.$product_type.'"><span class="tool-tip">This product will be available in this store for a limited time only : <br><strong>'.date("d/m/Y", $start).'</strong> until <strong>'.date("d/m/Y", $end).'</strong></span><img src="'.get_template_directory_uri().'/dist/images/Limited-time-only.png"/></div>';
                        }
                        
                        endwhile;
                    }

                    $html = '
                        <div class="row storeResultEntry" id="' . get_the_ID() . '" data-store-id="' . get_the_ID() . '">
                            <div class="col-sm-5">
                                ' . $thumbnail . $limitedTimeOffer.'
                            </div>
                            <div class="col-sm-5">
                                <p>' . get_the_title() . '</p>
                                <p>' . $location['address'] . '</p>
                            </div>
                            <div class="col-2">
                                <button data-id="' . get_the_ID() . '" class="locate-store"><i class="fal fa-location"></i></button>
                            </div>
                        </div>';

                    $store = [
                        'id' => get_the_ID(),
                        'name' => get_the_title(),
                        'location' => $location,
                        'html' => $html,
                        'products' => get_field('products'),
                        'limited_offers' => get_field('limited_offers')
                    ];

                    array_push($response['stores'], $store);
                }
            }
        } else {
            return null;
        }

        return $response;
    }

    public static function findCoordinates($params)
    {
        $address = isset($params['search']) && !empty($params['search']) ? $params['search'] : "";
        $prepAddr = str_replace(' ', '+', $address);
        $prepAddr = $prepAddr . '+UK';
        $geocode = file_get_contents('https://maps.google.com/maps/api/geocode/json?address=' . $prepAddr . '&sensor=false&key=' . get_field('google_maps_key', 'option'));
        $output = json_decode($geocode);
        $latitude = $output->status === "OK" ? $output->results[0]->geometry->location->lat : false;
        $longitude = $output->status === "OK" ? $output->results[0]->geometry->location->lng : false;
        $coordinates = $latitude !== false && $longitude !== false ? ["lat" => $latitude, "lng" => $longitude] : false;

        return $coordinates;
    }

    public static function distanceCalculate($latSearch, $lngSearch, $latStore, $lngStore)
    {
        $theta = $lngSearch - $lngStore;
        $dist = sin(deg2rad($latSearch)) * sin(deg2rad($latStore)) + cos(deg2rad($latSearch)) * cos(deg2rad($latStore)) * cos(deg2rad($theta));
        $dist = acos($dist);
        $dist = rad2deg($dist);
        $miles = $dist * 60 * 1.1515;

        return $miles;
    }

    public function uploadStoresCSV()
    {
        wp_add_dashboard_widget(
            'storeImportWidget',
            'Store CSV Import',
            [$this, "runUpload"]
        );
    }

    public function removeLimitedProducts()
    {
        $stores = $this->getAllStores();

        foreach($stores as $store) {
            $fieldsToKeep = [];
            if (have_rows('limited_offers', $store)) {
                $limited_offers = get_field('limited_offers',$store);
                foreach ($limited_offers as $offer) {
                    $end = strtotime($offer['limited_time_end']);
                    $currentDate = strtotime(date("Y-m-d"));

                    if($end > $currentDate) {
                        // end is in the future, keep this
                        $fieldsToKeep[] = $offer;
                    }
                }
                
                update_field('limited_offers', $fieldsToKeep, $store);
            }
        }
    }

    public function runUpload()
    {
        ?>
        <form action="" method="post" enctype="multipart/form-data" id="importerForm">
            Select csv to upload:
            <input type="file" name="storeImport" id="storeImport">
            <br/>
            <input type="hidden" value="submit" name="submit">
            <input type="submit" value="Upload CSV">
        </form>

        <p class="upload_response"></p>
        <?php
    }

    public function importStock()
    {
        check_ajax_referer('csv-nonce', 'nonce');
        $csv = $_POST['csv'];
        $attachment_id = $_POST['id'];
        $update = 0;
        $insert = 0;
        $delete = 0;
        $date = date('d-m-Y H:i:s');
        $supermarketProducts = [];
        $csvData = [];
        $storeData = [];

        //we dont want a timeout
        ini_set('max_execution_time', '0');
        //set our memory size for a temporary boost
        ini_set('memory_limit','64M');

        $supermarketProducts = $this->getAllSuperMarketProducts();

        if(empty($supermarketProducts))
        {
            wp_send_json_error('Could not get the supermarket products');
        }

        $csvData = $this->getDataFromCsv($csv);

        if(empty($csvData))
        {
            wp_send_json_error('There is an error with your CSV Data');
        }

        $storeData = $this->getUniqueStoreData($csvData, $supermarketProducts);
        $csvStores = [];

        if($storeData)
        {
            foreach ($storeData as $key => $store) {
                $postId = $this->checkIfExist($key);
                if (!$postId) {
                    $insert++;
                    $postId = $this->insertStore($key, $this->geocode($store['concat_address']), $store);
                } else {
                    $update++;
                    $this->updateStore($postId, $store);
                }

                $csvStores[$postId] = $postId;
            }
        } else {
            wp_send_json_error('Could not retrieve Store Data');
        }
        
        if($csvStores) {
            //now lets delete any stores that arent in the CSV
            $stores = $this->getAllStores();
            foreach($stores as $store) {
                if (!in_array($store, $csvStores))
                {
                    $delete++;
                    $this->deleteStore($store);
                }
            }
        }

        $to = 'jonathan@blue2.co.uk, shane@blue2.co.uk, rachael@blue2.co.uk, marketing@simonhowiefoods.co.uk';
        $subject = 'CSV Import Results';

        $body = '';

        //lets make sure our timezone is local
        date_default_timezone_set('Europe/London');

        $time = date("H");
    
        /* If the time is less than 1200 hours, show good morning */
        if ($time < "12") {
            $body .= "Good morning Simon Howie" . '<br><br>';
        } else
        /* If the time is grater than or equal to 1200 hours, but less than 1700 hours, so good afternoon */
        if ($time >= "12" && $time < "17") {
            $body .= "Good afternoon Simon Howie" . '<br><br>';
        }  else
        /* Finally, show good night if the time is greater than or equal to 1700 hours */
        if ($time >= "17") {
            $body .= "Good evening Simon Howie" . '<br><br>';
        }

        $body .= 'Here are the results of your recent CSV upload that you started at ' . $date . '<br><br>';

        if( $insert > 0 ) {
            $body .= 'Number of stores added: ' . $insert . '<br><br>';
        }

        if( $update > 0 ) {
            $body .= 'Number of stores updated: ' . $update . '<br><br>';
        }

        if( $delete > 0 ) {
            $body .= 'Number of stores deleted: ' . $delete . '<br><br>';
        }

        $body .= 'If these numbers do not look correct please email Blue2 and we will have a look at the function'. '<br>';
        $body .= 'Regards '. '<br>';
        $body .= 'Blue2 '. '<br>';

        $headers = array('Content-Type: text/html; charset=UTF-8');

        if( wp_mail( $to, $subject, $body, $headers ) ) {
            wp_send_json_success( 'Email sent successfully, please check your email for results' );
        } else {
            wp_send_json_success( 'We couldnt send the email, but the CSV ran successfully.' );
        }  
    }

    /**
     * @return array
     */
    private function getAllSuperMarketProducts()
    {
        $products = wc_get_products([
            'limit' => -1,
            'category' => ['supermarket'],
        ]);
        $supermarketProducts = [];
        foreach ($products as $product) {
            $id = $product->get_id();
            $supermarketProducts[get_field('_sku', $id)] = $id;
        }
        return $supermarketProducts;
    }

    /**
     * @return array
     */
    private function getAllStores()
    {
        $stores = get_posts(array(
            'fields'          => 'ids',
            'posts_per_page'  => -1,
            'post_type' => 'store'
        ));

        return $stores;
    }

    /**
     * @param string $fileName
     * @return array
     */
    private function getDataFromCsv($fileName)
    {
        $stores = array_map('str_getcsv', file($fileName));
        $header = array_shift($stores);
        $csv = [];
        foreach ($stores as $row) {
            $csv[] = array_combine($header, $row);
        }
        return $csv;
    }

    /**
     * @param array $csvData
     * @param array $productSkuToId
     * @return array
     */
    private function getUniqueStoreData($csvData, $productSkuToId)
    {
        $stores = [];

        foreach ($csvData as $stockEntry) {
            $storeReference = $stockEntry['Supermarket'] . " " . $stockEntry['Postcode'] . " " . $stockEntry["Store number"];

            $productId = isset($productSkuToId[$stockEntry['SKU Number']]) ? $productSkuToId[$stockEntry['SKU Number']] : null;
            if ($productId !== null) {
                $stores[$storeReference]['products'][] = $productId;
            }
            $stores[$storeReference]['title'] = $stockEntry['Store Name'];
            $stores[$storeReference]['group'] = $stockEntry['Supermarket'];
            $stores[$storeReference]['store_number'] = $stockEntry["Store number"];
            $stores[$storeReference]['store_url'] = $stockEntry["Store Locator URL"];
            $stores[$storeReference]['phone_number'] = '';
            $stores[$storeReference]['concat_address'] = $stockEntry['Address Line 1'] . " " . $stockEntry['Address Line 2'] . " " . $stockEntry['Address Line 3'] . " " . $stockEntry['Address Line 4'] . " " . $stockEntry['Postcode'];
            if ($stockEntry['Limited'] === "1") {
                $startTime = \DateTime::createFromFormat('d/m/Y', $stockEntry['Limited Time Start']);
                $endTime = \DateTime::createFromFormat('d/m/Y', $stockEntry['Limited Time End']);

                //Bail out if time formats are incorrect
                if (!$startTime || !$endTime) {
                    echo $productId . ' Limited not added due to incorrect date format';
                    continue;
                }
                $stores[$storeReference]['limited_offers'][] = [
                    'limited_product' => $productId,
                    'limited_time_start' => $startTime->format('Ymd'),
                    'limited_time_end' => $endTime->format('Ymd')
                ];
            }
        }
        return $stores;
    }

    /**
     * @param string $reference
     * @return int|bool
     */
    private function checkIfExist($reference)
    {
        $existingPost = get_posts([
            'post_type' => 'store',
            'meta_key' => 'reference',
            'meta_value' => $reference,
        ]);
        return count($existingPost) > 0 ? $existingPost[0]->ID : false;
    }

    private function insertStore($reference, $fullAddress, $store)
    {
        $postId = wp_insert_post([
            'post_title' => $store['title'],
            'post_status' => 'publish',
            'post_type' => 'store',
        ]);

        wp_set_object_terms($postId, $store['group'], 'group');

        add_post_meta($postId, 'location', $fullAddress, true);
        add_post_meta($postId, '_location', 'field_5c77fb63e1c1e', true);

        add_post_meta($postId, 'store_number', $store['store_number'], true);
        add_post_meta($postId, '_store_number', 'field_5c7cf5689bfc3', true);

        add_post_meta($postId, 'store_url', $store['store_url'], true);
        add_post_meta($postId, '_store_url', 'field_5c7cf5459bfc2', true);

        add_post_meta($postId, 'phone_number', $store['phone_number'], true);
        add_post_meta($postId, '_phone_number', 'field_5c7cf5759bfc4', true);

        add_post_meta($postId, 'reference', $reference, true);
        add_post_meta($postId, '_reference', 'field_5c8b708d919cc', true);

        if (isset($store['products'])) {
            add_post_meta($postId, 'products', $store['products'], true);
            add_post_meta($postId, '_products', 'field_5e31896c0938d', true);
        }

        if (isset($store['limited_offers'])) {
            foreach ($store['limited_offers'] as $limitedProduct) {
                add_row('limited_offers', $limitedProduct, $postId);
            }
        }

        $types = get_the_terms( $postId, 'group' );
        
        $store_type = $types[0]->name;

        //echo $store_type . ' ' . get_the_title($postId) . " has been created<br/>";

        return $postId;
    }

    private function deleteStore($id)
    {
        $store = get_the_title($id);
        $types = get_the_terms( $id, 'group' );
        wp_delete_post($id, false);

        $store_type = $types[0]->name;

        //echo $store_type . ' ' . $store . " has been deleted <br/>";
    }

    /**
     * @param string $address
     * @return array
     */
    public function geocode($address)
    {
        $googleKey = get_field('google_maps_key', 'option');
        $prepAddr = str_replace(' ', '+', $address);
        $geocode = file_get_contents('https://maps.google.com/maps/api/geocode/json?address=' . $prepAddr . '&sensor=false&key=' . $googleKey);
        $output = json_decode($geocode);
        $latitude = $output->results[0]->geometry->location->lat;
        $longitude = $output->results[0]->geometry->location->lng;

        return [
            "address" => $address,
            "lat" => $latitude,
            "lng" => $longitude,
        ];
    }

    private function updateStore($postId, $store)
    {
        $fieldsToUpdate = [
            'store_url',
            'phone_number',
            'products',
            'limited_offers'
        ];

        $types = get_the_terms( $postId, 'group' );

        foreach ($fieldsToUpdate as $field) {
            if (isset($store[$field])) {
                //check if our field is products
                if( $field == 'products') {
                    //if we got here then we have products for this store, start by deleting old products
                    delete_field($field, $postId);
                }
                
                update_field($field, $store[$field], $postId);
            }
        }
        
        $store_type = $types[0]->name;

        //echo $store_type . ' ' . get_the_title($postId) . " has been updated <br/>";
    }

    public function restrict_acf_date() 
    {
        ?>
        <script type="text/javascript">
        (function($) {
            acf.add_filter('date_picker_args', function( args, $field ) {
                // do something to args
                args['minDate'] = '-1y';

                return args;
            });
        })(jQuery);
        </script>
        <?php       
    }

    public function ajax_file_upload()
    {
        ?>
            <script>
                jQuery(function($){
                    jQuery( document ).ready(function() {
                        $( "#importerForm" ).submit(function( event ) {
                            $('.upload_response').html('<span>Uploading your CSV file, please wait</span>');
                            upload_csv();
                            event.preventDefault();
                        });
                    });
                    
                    function upload_csv()
                    {
                        var formData = new FormData();
                        formData.append("action", "upload-attachment");
                                                    
                        var fileInputElement = document.getElementById( "storeImport" );
                        formData.append( "async-upload" , fileInputElement.files[0]);
                        formData.append( "name" , fileInputElement.files[0].name);
                                                    
                        //also available on page from _wpPluploadSettings.defaults.multipart_params._wpnonce
                        <?php $my_nonce = wp_create_nonce("media-form"); ?>
                        formData.append("_wpnonce", "<?php echo $my_nonce; ?>");
                                                
                        var xhr = new XMLHttpRequest();
                        xhr.onreadystatechange=function()
                        {
                            if ( xhr.readyState == 4 && xhr.status == 200 )
                            {
                                var response = JSON.parse( xhr.responseText );
                                
                                if( response.success == true )
                                {
                                    $('.upload_response').html('<span style="color:green">CSV uploaded successfully, processing your CSV information</span>');
                                    //give a brief timeout to let things finish and show an update message
                                    setTimeout(function() { 
                                        import_stock(response.data.link, response.data.id);
                                    }, 8000);
                                    
                                }
                                else
                                {
                                    $('.upload_response').html('<span style="color:red">There was an error while uploading your CSV, please try again</span>');
                                }
                            }
                        }
                        
                        xhr.open("POST","async-upload.php",true);
                        xhr.send(formData);
                    }

                    function import_stock( csv, id )
                    {
                        var data = {};
                            
                        data[ 'action' ] = 'import_stock';
                        data[ 'nonce' ] = '<?php echo wp_create_nonce( 'csv-nonce' ); ?>';
                        data[ 'csv' ] = csv;
                        data[ 'id' ] = id;

                        $.ajax({
                            type: 'POST',
                            url: '<?php echo admin_url( 'admin-ajax.php' ); ?>',
                            data: data,
                            beforeSend: () => {
                                $('.upload_response').html('<span>We are currently processing your CSV, please wait <br><br>Please do not refresh this page or navigate away</span>');
                            },
                            success: response => {
                                if(response.success = false) {
                                    $('.upload_response').html('<span style="color:red">There was an error while processing your CSV, please try again <br><br> Error: ' + response.data + '</span>');
                                }
                                else if(response.success = true) {
                                    $('.upload_response').html('<span style="color:green">Your CSV was processed successfully <br><br>' + response.data + '</span>');
                                }
                            },
                            error: () => {
                                $('.upload_response').html('<span style="color:red">There was an error while processing your CSV. <br>This could be due to a cloudflare issue, please wait to see if you receive an email, if nothing is received within 30 minutes, try again.</span>');
                            },
                            complete: () => {
                            }
                        });
                    }
                })
            </script>
        <?php
    }
}
