/* global google */

import loadScript from './utils/externalScriptLoader';
import { GOOGLE_API_KEY } from './utils/constants';
import axios from 'axios';

import MarkerOrange from '../images/map-marker-orange.png';

const simulateClick = function(elem) {
    // Create our event (with options)
    const evt = new MouseEvent('click', {
        bubbles: true,
        cancelable: true,
        view: window
    });
    // If cancelled, don't dispatch our event
    const canceled = !elem.dispatchEvent(evt);
    console.log(canceled);
};
let productToActivate = '';

class StoreMap {
    map = null;
    geocoder = null;
    defaultCenter = { lat: 56.3904758, lng: -3.4843572 };
    activeLocation = null;
    stores = [];
    markers;

    constructor() {
        this.markers = {};
        loadScript(
            `https://maps.googleapis.com/maps/api/js?key=${GOOGLE_API_KEY}&libraries=geometry&region=GB`
        ).then(() => this.initMap());
        this.indicateLoading();
        document
            .getElementById('map-search-form')
            .addEventListener('submit', event => this.formSubmit(event));
        this.swiper = document.querySelector('.swiper-container').swiper;

        document
            .getElementById('reset')
            .addEventListener('click', this.resetEverything);

        document
            .getElementById('reset')
            .addEventListener('click', this.resetScroll);

        document
            .getElementById('product-range')
            .addEventListener('change', this.deactivateProductFilter);

        document
            .getElementById('individual-product-filter')
            .addEventListener('change', e => {
                this.updateActiveSwiperSlide();
                this.filterMarkersAndResults(parseInt(e.target.value,10));
            });

        setInterval(function() {
            if (document.getElementById('search-input').value.length > 0) {
                document.getElementById('locationPropmpt').classList.add('d-none');
            } else {
                document.getElementById('locationPropmpt').classList.remove('d-none');
            }
        }, 2000);

        const userSelection = document.getElementsByClassName('add-location-prompt');

        for (let i = 0; i < userSelection.length; i++) {
            userSelection[i].addEventListener('click', this.submitFromSwiper);
        }
    }
    indicateLoading = () => {
        document.getElementById('map-contain').classList.add('loading');
    };

    locateStore(storeId) {
        this.map.setZoom(14);
        this.map.panTo(this.markers[storeId].position);
    }

    disableLoading = () => {
        document.getElementById('map-contain').classList.remove('loading');
    };

    initMap() {
        const UK = {
            north: 59.81,
            south: 49.72,
            west: -13.23,
            east: 0.67
        };
        this.fetchData('');
        this.map = new google.maps.Map(document.getElementById('map'), {
            center: this.defaultCenter,
            zoom: 10,
            disableDefaultUI: true,
            restriction: {
                latLngBounds: UK,
                strictBounds: false
            }
        });
        this.geocoder = new google.maps.Geocoder();

        let calculatedCenter = {};

        if (navigator.geolocation) {
            navigator.geolocation.getCurrentPosition(
                position => {
                    calculatedCenter = {
                        lat: position.coords.latitude,
                        lng: position.coords.longitude
                    };
                    this.map.setCenter(calculatedCenter);
                    this.map.setZoom(10);
                    this.activeLocation = new google.maps.LatLng(
                        position.coords.latitude,
                        position.coords.longitude
                    );
                },
                () => {
                    this.activeLocation = new google.maps.LatLng(
                        this.defaultCenter.lat,
                        this.defaultCenter.lng
                    );
                    this.getAllStores(this.activeLocation).then(data => {
                        this.stores = data;
                        this.addMarkersToMap();
                        this.addResults();
                        this.registerButtonEvents();
                        this.disableLoading();
                    });
                }
            );
        } else {
            this.activeLocation = new google.maps.LatLng(
                this.defaultCenter.lat,
                this.defaultCenter.lng
            );
        }
    }

    registerButtonEvents() {
        const buttons = document.getElementsByClassName('locate-store');
        Array.from(buttons).forEach(element => {
            element.addEventListener('click', () => {
                this.locateStore(element.dataset.id);
                this.highlightResult(element.dataset.id);
                const marker = this.markers[element.dataset.id];
                setTimeout(() => {
                    if (marker.getAnimation() !== null) {
                        marker.setAnimation(null);
                    } else {
                        marker.setAnimation(google.maps.Animation.BOUNCE);
                    }
                }, 1000);
                setTimeout(() => {
                    marker.setAnimation(null);
                }, 4000);
            });
        });
    }

    async getAllStores(requestParams = null) {
        return await axios
            .post('/wp-json/simon-howie/v1/store-locations', {
                requestParams
            })
            .then(response => {
                return response.data;
            })
            .catch(error => {
                return error;
            });
    }

    addMarkersToMap() {
        if (this.stores.length === 0) {
            return;
        }

        for (const store of this.stores) {
            const { lat, lng } = store.location;
            const marker = new google.maps.Marker({
                position: {
                    lat: parseFloat(lat),
                    lng: parseFloat(lng)
                },
                animation: google.maps.Animation.DROP,
                icon: MarkerOrange
            });

            marker.setMap(this.map);
            store.visible = true;

            this.markers[store.id] = marker;
            marker.addListener('click', () => {
                this.highlightResult(store.id);
            });
            this.map.setZoom(10);
        }
    }

    highlightResult = id => {
        const results = document.getElementsByClassName('result');
        Array.from(results).forEach(result => {
            result.childNodes[1].style.backgroundColor = 'transparent';
        });
        const result = document.getElementById(id);

        result.scrollIntoView({
            behavior: 'smooth',
            block: 'nearest'
        });
        result.style.backgroundColor = 'rgba(234, 118, 0, 0.5)';

        if (window.innerWidth <= 992) {
            document.getElementById('map').scrollIntoView({
                behavior: 'smooth',
                block: 'nearest'
            });
        }
    };

    addResults() {
        if (this.stores.length === 0) {
            return;
        }

        for (const store of this.stores) {
            if (store.visible !== true) {
                continue;
            }
            const template = document.createElement('div');
            template.className = 'result';
            template.innerHTML = store.html;
            document.getElementById('result-container').appendChild(template);
        }

        if (productToActivate !== '') {
            setTimeout(function() {
                $(`.swiper-slide[data-id='${productToActivate}']`).addClass('slideToActivate');
                console.log(productToActivate);
                // const mySlide = $('.slideToActivate')
                //    .attr('data-swiper-slide-index');
                // this.swiper.slideTo(mySlide - 1);
                const activeProductSlides = document.getElementsByClassName('product--active');
                let i = 0;
                for (i = 0; i < activeProductSlides.length; i++) {
                    activeProductSlides[i].classList.remove('product--active');
                }
                const activeSlide = document.querySelector('.slideToActivate');
                console.log(activeSlide);
                activeSlide.classList.remove('slideToActivate');

                simulateClick(activeSlide);
            }, 500);
        }
    }

    returnNoResults() {
        const form = document.getElementById('map-search-form');
        const values = Object.values(form).reduce((obj, field) => {
            obj[field.name] = field.value;
            return obj;
        }, {});
        if (this.stores.length === 0 && values.search !== '') {
            const template = document.createElement('div');
            template.className = 'result';
            template.innerHTML =
                '<p class="no-results">Sorry, there were no results found in your area.</p>';
            document.getElementById('result-container').appendChild(template);
        }
    }

    formSubmit(event) {
        document
            .getElementById('individual-product-filter-container')
            .classList.remove('individual-product-filter-container--active');
        event.preventDefault();
        this.indicateLoading();
        this.resetResults();
        const form = document.getElementById('map-search-form');
        const values = Object.values(form).reduce((obj, field) => {
            obj[field.name] = field.value;
            return obj;
        }, {});
        const location = `${values.search}, uk`;
        this.geocodeAddress(location);
        this.fetchData(values, true);

        if (window.innerWidth <= 992) {
            document.getElementById('result-container').scrollIntoView({
                behavior: 'smooth',
                block: 'nearest'
            });
        } else {
            /*
            document.getElementById('map').scrollIntoView({
                behavior: 'smooth',
                block: 'nearest'
            });
            */
            const tgtEl = $('#map-search-form');
            const scrollToHere = tgtEl.offset().top;
            $('html, body').animate({
                scrollTop: scrollToHere
            }, 500);
            const accent = document.getElementById('where-to-buy-accent');
            accent.classList.add('accent--scrolled');
        }
    }

    geocodeAddress(value) {
        this.geocoder.geocode({ address: value }, (results, status) => {
            if (status === 'OK') {
                this.activeLocation = results[0].geometry.location;
                this.map.setCenter(results[0].geometry.location);
            } else {
                console.log(
                    `Geocode was not successful for the following reason: ${status}`
                );
            }
        });
    }

    fetchData(requestData, updateSwiper = false) {
        this.indicateLoading();
        try {
            this.getAllStores(requestData).then(data => {
                if (data) {
                    this.stores = data.stores;
                    this.products = data.products;
                    this.addMarkersToMap();
                    this.addResults();
                    this.registerButtonEvents();
                    if (updateSwiper) {
                        this.updateSwiper();
                    }
                } else {
                    this.returnNoResults();
                }
                this.disableLoading();
            });
        } catch (error) {
            this.disableLoading();
        }
    }

    updateSwiper = () => {
        if (
            this.swiper === null ||
            this.products === null ||
            this.products.length === 0
        ) {
            return;
        }

        const productsSelect = document.getElementById('individual-product-filter');
        while (productsSelect.options.length > 0) {
            productsSelect.remove(0);
        }
        productsSelect.options[productsSelect.options.length] = new Option('-- Select a product --', 'select');

        this.swiper.removeAllSlides();
        this.products.forEach(product => {
            this.swiper.appendSlide(
                `<div class="swiper-slide map-filter" data-id="${product.id}">
                    <img src="${product.image}" alt="" data-id="${product.id}">
                </div>`
            );
            productsSelect.options[productsSelect.options.length] = new Option(`${product.name}`, `${product.id}`);
        });
        this.swiper.update();
        const newSlides = document.getElementsByClassName('map-filter');
        Array.from(newSlides).forEach(slide => {
            slide.onclick = this.filterMap;
        });
        document
            .getElementById('individual-product-filter-container')
            .classList.add('individual-product-filter-container--active');
    };

    updateActiveSwiperSlide = () => {
        const productValue = document
            .getElementById('individual-product-filter')
            .value;
        if (productValue !== 'select') {
            $(`.swiper-slide[data-id='${productValue}']`).addClass('slideToActivate');
            const mySlide = $('.slideToActivate')
                .attr('data-swiper-slide-index');
            /*
            if (mySlide < 1) {
                mySlide = 1000;
            }
            */
            this.swiper.slideTo(mySlide - 1);
            const activeProductSlides = document.getElementsByClassName('product--active');
            let i = 0;
            for (i = 0; i < activeProductSlides.length; i++) {
                activeProductSlides[i].classList.remove('product--active');
            }

            const activeSlide = document.querySelector('.slideToActivate');
            activeSlide.classList.remove('slideToActivate');

            simulateClick(activeSlide);
        }
    }

    filterMap = e => {
        e.preventDefault();
        if (!e.target.dataset.id) {
            return;
        }
        const activeProductSlides = document.getElementsByClassName('product--active');
        let i = 0;
        for (i = 0; i < activeProductSlides.length; i++) {
            activeProductSlides[i].classList.remove('product--active');
        }
        e.target.closest('.map-filter').classList.add('product--active');

        // Set 'limited time offer' tooltips.
        const resultLimitedOffers = document.getElementsByClassName('limited-offer--visible');
        i = 0;
        for (i = 0; i < resultLimitedOffers.length * 2; i++) {
            resultLimitedOffers[0].classList.remove('limited-offer--visible');
        }
        const results = document.getElementById('result-container');
        const activeProductOffers = results.getElementsByClassName(`product-${e.target.dataset.id}`);
        i = 0;
        for (i = 0; i < activeProductOffers.length; i++) {
            activeProductOffers[i].classList.add('limited-offer--visible');
        }

        // Get id from data attribute
        const productId = parseInt(e.target.dataset.id, 10);
        this.filterMarkersAndResults(productId);
    };

    filterMarkersAndResults = productId => {
        const storesWithProduct = [];

        // Update Select value
        document.getElementById('individual-product-filter').value = productId;

        if (this.stores.length === 0) {
            return;
        }
        // Foreach store, check which have the selected product
        this.stores.forEach(store => {
            const foundProduct = store.products.find(id => {
                return id === productId;
            });

            // If store stocks selected product add to array
            if (foundProduct) {
                storesWithProduct.push(store.id);
            }
        });

        document.querySelectorAll('.storeResultEntry').forEach(store =>  {
            const found = storesWithProduct.includes(parseInt(store.dataset.storeId, 10));

            store.style.display = found ? 'flex' : 'none';
            store.parentElement.style.background = found ? 'white' : 'none';
        });

        // Foreach marker set visibility based on whether or not the store stocks filtered item
        Object.keys(this.markers).forEach(key => {
            const visibility = storesWithProduct.includes(parseInt(key, 10));
            this.markers[key].setVisible(visibility);
        });
    };

    resetResults = () => {
        if (this.markers) {
            Object.keys(this.markers).forEach(key => {
                this.markers[key].setMap(null);
            });
        }

        this.markers = {};
        this.stores = [];
        document.getElementById('result-container').innerHTML =
            '<p class="default">Please select above to view results</p>';
        const accent = document.getElementById('where-to-buy-accent');
        accent.classList.remove('accent--scrolled');
    };

    resetEverything = () => {
        location.reload();
    }

    resetScroll = () => {
        const scrollTo = $('#top').offset().top;
        $('html, body').animate({
            scrollTop: scrollTo
        }, 500);
        const accent = document.getElementById('where-to-buy-accent');
        accent.classList.remove('accent--scrolled');
    }

    activateProductFilter = () => {
        const myValue = document
            .getElementById('product-range')
            .value;
        if (myValue === '0') {
            document
                .getElementById('individual-product-filter-container')
                .classList.remove('individual-product-filter-container--active');
        } else {
            document
                .getElementById('individual-product-filter-container')
                .classList.add('individual-product-filter-container--active');
        }
    }

    deactivateProductFilter = () => {
        document
            .getElementById('individual-product-filter-container')
            .classList.remove('individual-product-filter-container--active');
    }

    submitFromSwiper = e => {
        const productRangeId = parseInt(e.target.getAttribute('data-range'), 10);
        productToActivate = parseInt(e.target.getAttribute('data-id'), 10);
        document.getElementById('product-range').value = productRangeId;
        const submitBtn = document.getElementById('submit-search');
        simulateClick(submitBtn);
        document
            .getElementById('individual-product-filter-container')
            .classList.add('individual-product-filter-container--active');
    }

    displayAllMarkers = () => {
        Object.keys(this.markers).forEach(key => {
            this.markers[key].setVisible(true);
        });
    };
}

export default StoreMap;
