/* global google */

import loadScript from './utils/externalScriptLoader';
import { GOOGLE_API_KEY } from './utils/constants';
import axios from 'axios';

import MarkerOrange from '../images/map-marker-orange.png';

class StoreMap {
    map = null;
    geocoder = null;
    defaultCenter = { lat: 56.3904758, lng: -3.4843572 };
    activeLocation = null;
    stores = [];
    markers;

    constructor() {
        this.markers = {};
        loadScript(`https://maps.googleapis.com/maps/api/js?key=${GOOGLE_API_KEY}&libraries=geometry&region=GB`)
            .then(() => this.initMap());
        this.indicateLoading();
        document.getElementById('map-search-form')
            .addEventListener('submit', event => this.formSubmit(event));
    }

    indicateLoading = () => {
        document.getElementById('map-contain')
            .classList
            .add('loading');
    };

    locateStore(storeId) {
        this.map.setZoom(14);
        this.map.panTo(this.markers[storeId].position);
    }

    disableLoading = () => {
        document.getElementById('map-contain')
            .classList
            .remove('loading');
    };

    initMap() {
        const UK = {
            north: 59.81,
            south: 49.72,
            west: -13.23,
            east: 0.67,
        };
        this.fetchData('');
        this.map = new google.maps.Map(document.getElementById('map'), {
            center: this.defaultCenter,
            zoom: 10,
            disableDefaultUI: true,
            restriction: {
                latLngBounds: UK,
                strictBounds: false,
            },
        });
        this.geocoder = new google.maps.Geocoder();

        let calculatedCenter = {};

        if (navigator.geolocation) {
            // console.log(navigator.geolocation);
            navigator.geolocation.getCurrentPosition(position => {
                calculatedCenter = {
                    lat: position.coords.latitude,
                    lng: position.coords.longitude
                };
                this.map.setCenter(calculatedCenter);
                this.map.setZoom(10);
                this.activeLocation = new google.maps.LatLng(position.coords.latitude, position.coords.longitude);
            },
            () => {
                this.activeLocation = new google.maps.LatLng(this.defaultCenter.lat, this.defaultCenter.lng);
                // console.log(this.activeLocation);
                this.getAllStores(this.activeLocation)
                    .then(data => {
                        this.stores = data;
                        this.addMarkersToMap();
                        this.addResults();
                        this.registerButtonEvents();
                        this.disableLoading();
                    });
            });
        } else {
            this.activeLocation = new google.maps.LatLng(this.defaultCenter.lat, this.defaultCenter.lng);
        }
    }

    registerButtonEvents() {
        const buttons = document.getElementsByClassName('locate-store');
        Array.from(buttons)
            .forEach(element => {
                element.addEventListener('click', () => {
                    this.locateStore(element.dataset.id);
                    this.highlightResult(element.dataset.id);
                    const marker = this.markers[element.dataset.id];
                    // console.log(marker);
                    setTimeout(() => {
                        if (marker.getAnimation() !== null) {
                            marker.setAnimation(null);
                        } else {
                            marker.setAnimation(google.maps.Animation.BOUNCE);
                        }
                    }, 1000);
                    setTimeout(() => { marker.setAnimation(null); }, 4000);
                });
            });
    }

    async getAllStores(requestParams = null) {
        return await axios.post('/wp-json/simon-howie/v1/store-locations', {
            requestParams
        })
            .then(response => {
                return response.data;
            })
            .catch(error => {
                return error;
            });
    }

    addMarkersToMap() {
        for (const store of this.stores) {
            const { lat, lng } = store.location;
            // const distance = google.maps.geometry.spherical.computeDistanceBetween
            // (this.activeLocation, new google.maps.LatLng(lat, lng));
            const marker = new google.maps.Marker({
                position: {
                    lat: parseFloat(lat),
                    lng: parseFloat(lng)
                },
                animation: google.maps.Animation.DROP,
                icon: MarkerOrange
            });

            marker.setMap(this.map);
            store.visible = true;

            this.markers[store.id] = marker;
            marker.addListener('click', () => {
                this.highlightResult(store.id);
            });
            this.map.setZoom(10);
        }
    }

    highlightResult = id => {
        const results = document.getElementsByClassName('result');
        Array.from(results).forEach(result => {
            result.childNodes[1].style.backgroundColor = 'transparent';
        });
        const result = document.getElementById(id);

        result.scrollIntoView({
            behavior: 'smooth',
            block: 'nearest'
        });
        result.style.backgroundColor = 'rgba(234, 118, 0, 0.5)';
    };

    addResults() {
        for (const store of this.stores) {
            if (store.visible !== true) {
                continue;
            }
            const template = document.createElement('div');
            template.className = 'result';
            template.innerHTML = store.html;
            document.getElementById('result-container')
                .appendChild(template);
        }
    }
    returnNoResults() {
        const form = document.getElementById('map-search-form');
        const values = Object.values(form)
            .reduce((obj, field) => {
                obj[field.name] = field.value;
                return obj;
            }, {});
        // console.log(values);
        if (this.stores.length === 0 && values.search !== '') {
            const template = document.createElement('div');
            template.className = 'result';
            template.innerHTML = '<p class="no-results">Sorry, there were no results found in your area.</p>';
            document.getElementById('result-container')
                .appendChild(template);
        }
    }

    formSubmit(event) {
        event.preventDefault();
        this.indicateLoading();
        this.resetResults();
        const form = document.getElementById('map-search-form');
        const values = Object.values(form)
            .reduce((obj, field) => {
                obj[field.name] = field.value;
                return obj;
            }, {});
        const location = `${values.search}, uk`;
        this.geocodeAddress(location);
        this.fetchData(values);

        document.getElementById('map').scrollIntoView({
            behavior: 'smooth',
            block: 'nearest'
        });
    }

    geocodeAddress(value) {
        this.geocoder.geocode({ address: value }, (results, status) => {
            if (status === 'OK') {
                this.activeLocation = results[0].geometry.location;
                this.map.setCenter(results[0].geometry.location);
            } else {
                console.log(`Geocode was not successful for the following reason: ${status}`);
            }
        });
    }

    fetchData(requestData) {
        this.indicateLoading();
        try {
            this.getAllStores(requestData)
                .then(data => {
                    this.stores = data;
                    this.addMarkersToMap();
                    this.addResults();
                    this.registerButtonEvents();
                    this.disableLoading();
                });
        } catch (error) {
            this.disableLoading();
        }
        try {
            this.getAllStores(requestData)
                .then(data =>  {
                    this.stores = data;
                    this.returnNoResults();
                });
        } catch (error) {
            this.disableLoading();
        }
    }

    resetResults() {
        Object.keys(this.markers)
            .forEach(key => {
                this.markers[key].setMap(null);
            });

        this.markers = {};
        this.stores =  [];
        document.getElementById('result-container').innerHTML = '';
    }
}

export default StoreMap;
