<?php
namespace Blue2\Scruffy;
use Blue2\Scruffy\Controllers\MaintenanceController;
use Blue2\Scruffy\Controllers\SiteController;

class Site extends \Blue2\Common\Site
{
    public function __construct()
    {
        parent::__construct();

        $this->registerShortCodes();
        $this->registerFilters();
        $this->registerActions();
        $this->setupControllers();
    }

    public function no_redirect_on_404($redirect_url)
    {
        if(is_404())
        {
            return false;
        }
        return $redirect_url;
    }

    public function overwriteCPTRequest($query) {
        // If query doesn't contain locations, just return the query
        if (strpos($query->request, 'locations') === false) {
            return $query;
        } 

        if ( count($query->query_vars) > 0 && empty($query->query_vars['post_type'])) {
            // The request contains the string that was requested.
            $pageName = basename($query->request);
    
            // Check if post with the requested slug exists in your custom post type
            $plotResult = get_posts(array(
                'post_type' => 'plot',
                'name' => $pageName
            ));

            $experienceResult = get_posts(array(
                'post_type' => 'experience',
                'name' => $pageName
            ));

            if (count($experienceResult) < 1 && count($plotResult) < 1 ) {
                //never matched any of our requests
                return $query;
            } 
            
            if($plotResult) {
                $new_query = array(
                    'page' => '',
                    'plot' => $pageName,
                    'post_type' => 'plot',
                    'name' => $pageName
                );

                $query->query_vars = $new_query;
                return $query;
            } elseif($experienceResult) {
                $new_query = array(
                    'page' => '',
                    'experience' => $pageName,
                    'post_type' => 'experience',
                    'name' => $pageName
                );

                $query->query_vars = $new_query;
                return $query;
            }
            else {
                return $query;
            }
    
        } else {
            return $query;
        }
    }

    public function overwriteExperiencePage($query) {
        if (strpos($query->request, 'locations') === false) {
            return $query;
        } 

        if ( count($query->query_vars) > 0 && empty($query->query_vars['post_type'])) {
            // The request contains the string that was requested.
            $pageName = basename($query->request);
    
            // Check if post with the requested slug exists in your custom post type
            $result= get_page_by_title( ucfirst($pageName) );

            if (count($result) < 1) {
                //never matched any of our requests
                return $query;
            } 
            
            if($result) {
                $new_query = array(
                    'post_type'  => 'page', 
                    'name' => ucfirst($pageName) 
                );

                $query->query_vars = $new_query;
                return $query;
            }
            else {
                return $query;
            }
    
        } else {
            return $query;
        }
    }

    public function registerMenus()
    {
        register_nav_menus([
            'main_menu' => 'Main Menu',
            'footer_menu' => 'Footer Menu',
            'quick_links' => 'Quick Links',
            'help_links' => 'Help Links',
            'legal_links' => 'Footer Legal Links'
        ]);
    }

    public function tidyExcerpt()
    {
        return ' &hellip; <a href="' . get_permalink() . '">Continued</a>';
    }

    private function registerShortCodes()
    {
        // Empty
    }

    private function registerFilters(){
        add_filter('redirect_canonical', [$this,'no_redirect_on_404']);

        add_filter( 'ninja_forms_render_options', [$this, 'render_park_locations'] ,10,2);

        add_filter( 'post_type_link', [$this,'change_plot_link'], 1, 3 );

        add_filter( 'register_post_type_args', [$this, 'change_post_slugs'], 10, 2 );

        add_filter('excerpt_more', [$this, 'tidyExcerpt']);

        add_filter( 'get_the_archive_title', [$this,'mlv_archive_title'] );
    }

    private function registerActions() {
        add_action('init', [$this, 'registerPostTypes']);

        add_action('after_setup_theme', [$this, 'registerMenus']);

        add_action('init', [$this, 'customImageSizes']);

        add_action('init', [$this, 'registerCustomTaxonomy']);

        add_action('init', [$this, 'customAdminPages']);

        add_action('parse_request', [$this, 'overwriteCPTRequest'], 10, 1);

        add_action('parse_request', [$this, 'overwriteExperiencePage'], 10, 1);

        if (isset($_GET['lodge_search']) && $_GET['lodge_search'] == 1){
            add_filter('pre_get_posts',[$this,'lodge_Search_filter']);
        }
    }

    private function setupControllers(){
        new MaintenanceController();
        new SiteController();
    }

    public function registerPostTypes()
    {
        Post\Event::registerPostType();
        Post\Lodge::registerPostType();
        Post\Plot::registerPostType();
        Post\Experience::registerPostType();
        Post\Team::registerPostType();
    }

    public function registerCustomTaxonomy() 
    {
        $labels = array(
            'name' => _x( 'Locations', 'taxonomy general name' ),
            'singular_name' => __( 'Locations', 'taxonomy singular name' ),
            'search_items' =>  __( 'Search Locations' ),
            'popular_items' => __( 'Popular Locations' ),
            'all_items' => __( 'All Locations' ),
            'parent_item' => null,
            'parent_item_colon' => null,
            'edit_item' => __( 'Edit Location' ), 
            'update_item' => __( 'Update Location' ),
            'add_new_item' => __( 'Add New Location' ),
            'new_item_name' => __( 'New Location Name' ),
            'separate_items_with_commas' => __( 'Separate locations with commas' ),
            'add_or_remove_items' => __( 'Add or remove locations' ),
            'choose_from_most_used' => __( 'Choose from the most used locations' ),
            'menu_name' => __( 'Locations' ),
        );
        
        register_taxonomy('locations',array('lodge','plot', 'experience'), array(
            'hierarchical' => true,
            'labels' => $labels,
            'show_ui' => true,
            'show_in_rest' => true,
            'show_admin_column' => true,
            'update_count_callback' => '_update_post_term_count',
            'query_var' => true,
            'rewrite' => array( 'slug' => 'locations' ),
          ));
    }

    public function customImageSizes()
    {
        add_image_size('banner', '1920', '800', true);
        add_image_size('card', '500', '300', true);
        add_image_size('article', '530', '375', true);
        add_image_size('slider', '850', '850', true);
        add_image_size('team-member', 360, 440, ['center', 'center']);
    }

    public function change_plot_link( $post_link, $id = 0 ) {
        $post = get_post( $id );
        if( $post->post_type == 'plot' || $post->post_type == 'experience' ) 
        {
           if ( is_object( $post ) ) {
              $terms = wp_get_object_terms( $post->ID, array('locations') );
              if ( $terms ) {
                 return str_replace( '%cat%', $terms[0]->slug, $post_link );
             }
          }
        }
        return   $post_link ;
    }

    public function change_post_slugs( $args, $post_type ) {

        if ( $post_type == 'plot') {
            $args['rewrite']['slug'] = 'locations/%cat%';
        }

        if ( $post_type == 'experience') {
            $args['rewrite']['slug'] = 'locations/%cat%/experiences';
        }

        return $args;
    }

    public function customAdminPages()
    {        
        $terms = get_terms( array(
            'taxonomy' => 'locations',
            'hide_empty' => false,
        ) );

        // Add parent.
        $parent = acf_add_options_page(array(
            'page_title'  => __('Location Settings'),
            'menu_title'  => __('Location Settings'),
            'redirect'    => true,
        ));

        foreach($terms as $term) {
            if (function_exists('acf_add_options_page')) {
                // Add sub page.
                $options = acf_add_options_page(array(
                    'page_title'  => __($term->name . ' Settings'),
                    'menu_title'  => __($term->name),
                    'parent_slug' => $parent['menu_slug']
                ));

                $termObjects = [];
                $acfLocations = [];

                $termObject = array(
                    'param' => 'options_page',
                    'operator' => '==',
                    'value' => 'acf-options-' . $term->slug,
                );

                $termObjects[] = $termObject;

                $acfLocations[] = $termObjects;
            }

            //add our options to our new parent pages
            if( function_exists('acf_add_local_field_group') ):
                acf_add_local_field_group(array(
                    'key' => $term->slug . '-group',
                    'title' => 'Location Information',
                    'fields' => array(
                        array(
                            'key' => $term->slug . '-site-plan',
                            'label' => 'Site Plan',
                            'name' => '',
                            'type' => 'tab',
                            'instructions' => '',
                            'required' => 0,
                            'conditional_logic' => 0,
                            'wrapper' => array(
                                'width' => '',
                                'class' => '',
                                'id' => '',
                            ),
                            'placement' => 'top',
                            'endpoint' => 0,
                        ),
                        array(
                            'key' => $term->slug . '-site-map',
                            'label' => 'Site Map',
                            'name' => $term->slug . '-site-map',
                            'type' => 'image',
                            'instructions' => 'Upload the site plan',
                            'required' => 0,
                            'conditional_logic' => 0,
                            'wrapper' => array(
                                'width' => '',
                                'class' => '',
                                'id' => '',
                            ),
                            'return_format' => 'url',
                            'preview_size' => 'medium',
                            'library' => 'all',
                            'min_width' => '',
                            'min_height' => '',
                            'min_size' => '',
                            'max_width' => '',
                            'max_height' => '',
                            'max_size' => '',
                            'mime_types' => '',
                        ),
                        array(
                            'key' => $term->slug . '-3d-walkthrough',
                            'label' => '3d Walk Through',
                            'name' => '',
                            'type' => 'tab',
                            'instructions' => '',
                            'required' => 0,
                            'conditional_logic' => 0,
                            'wrapper' => array(
                                'width' => '',
                                'class' => '',
                                'id' => '',
                            ),
                            'placement' => 'top',
                            'endpoint' => 0,
                        ),
                        array(
                            'key' => $term->slug . '-matterport-code',
                            'label' => 'Matterport Code',
                            'name' => $term->slug . '-matterport-code',
                            'type' => 'text',
                            'instructions' => 'Enter the link to the matterport video',
                            'required' => 0,
                            'conditional_logic' => 0,
                            'wrapper' => array(
                                'width' => '',
                                'class' => '',
                                'id' => '',
                            ),
                            'default_value' => '',
                            'tabs' => 'all',
                            'toolbar' => 'full',
                            'media_upload' => 1,
                            'delay' => 0,
                        ),
                        array(
                            'key' => $term->slug . '-location-tab',
                            'label' => 'Location',
                            'name' => '',
                            'type' => 'tab',
                            'instructions' => '',
                            'required' => 0,
                            'conditional_logic' => 0,
                            'wrapper' => array(
                                'width' => '',
                                'class' => '',
                                'id' => '',
                            ),
                            'placement' => 'top',
                            'endpoint' => 0,
                        ),
                        array(
                            'key' => $term->slug . '-google-maps',
                            'label' => 'Location',
                            'name' => $term->slug . '-location',
                            'type' => 'wysiwyg',
                            'instructions' => 'Paste in your Google Maps embed code',
                            'required' => 0,
                            'conditional_logic' => 0,
                            'wrapper' => array(
                                'width' => '',
                                'class' => '',
                                'id' => '',
                            ),
                            'default_value' => '',
                            'tabs' => 'all',
                            'toolbar' => 'full',
                            'media_upload' => 1,
                            'delay' => 0,
                        ),
                    ),
                    'location' => $acfLocations,
                    'menu_order' => 0,
                    'position' => 'normal',
                    'style' => 'default',
                    'label_placement' => 'top',
                    'instruction_placement' => 'label',
                    'hide_on_screen' => '',
                    'active' => true,
                    'description' => '',
                ));
            endif;
        }
    }
    
    public function lodge_Search_filter($query) {
        //only do this on search
        if (is_search()) {
            //first check for the first type if not set then its the default post.
            $post_type = 'plot';

            if ($query->get('post_type') == 'nav_menu_item')
                return $query;

            if ($query->is_search) {
                $query->set('post_type', $post_type );
            };
    
            if (isset($_GET['resort'])){
                $query->set('locations', $_GET['resort']);
            }

            $metaQuery = [
                'relation' => 'AND'
            ];

            if(isset($_GET['bedrooms'])) {
                $metaQuery[] = [
                    'key' => 'number_of_bedrooms',
                    'value' => intval($_GET['bedrooms']),
                    'compare' => '=='
                ];
            }
            if(isset($_GET['type'])) {
                $metaQuery[] = [
                    'key' => 'lodge_type',
                    'value' => $_GET['type'],
                    'compare' => 'LIKE'
                ];
            }

            $metaQuery[] = [
                'key' => 'plot_status',
                'value' => 'sale',
                'compare' => 'LIKE'
            ];

            $query->set('meta_query', $metaQuery);
        }
    
        return $query;
    }

    public function render_park_locations($options,$settings) {
        if( $settings['key'] == 'park_location_1624029241106' ){
            
            $terms = get_terms( array(
                'taxonomy' => 'locations',
                'hide_empty' => false,
            ) );

            foreach($terms as $term) {
                $options[] = array('label' => $term->name, 'value' => $term->name);
            }
        }
        return $options;
    }

    public function mlv_archive_title( $title ) {
        if ( is_category() ) {
            $title = single_cat_title( '', false );
        } elseif ( is_tag() ) {
            $title = single_tag_title( '', false );
        } elseif ( is_author() ) {
            $title = '<span class="vcard">' . get_the_author() . '</span>';
        } elseif ( is_post_type_archive() ) {
            $title = post_type_archive_title( '', false );
        } elseif ( is_tax() ) {
            $title = single_term_title( '', false );
        }
    
        return $title;
    }
}  