<?php

namespace Blue2\Scruffy;

use Blue2\Scruffy\Controllers\ClanTartanController;
use Blue2\Scruffy\Controllers\PostController;
use Blue2\Scruffy\Controllers\UserController;
use Blue2\Scruffy\Controllers\WooCommerceController;
use Blue2\Scruffy\Controllers\BuildAnOutfitController;
use Blue2\Scruffy\Taxonomy\Colour;
use Blue2\Scruffy\Taxonomy\Composition;
use Blue2\Scruffy\Taxonomy\FabricStyle;
use Blue2\Scruffy\Taxonomy\FabricType;
use Blue2\Scruffy\Taxonomy\Usage;

class Site extends \Blue2\Common\Site
{
    const AOS_CLASSES = [
        'data-aos="fade-up-scale-up"',
        'data-aos="fade-down-scale-up"',
    ];

    public function __construct()
    {
        parent::__construct();

        add_action('after_setup_theme', [$this, 'registerMenus']);
        add_action('pre_get_posts', [$this, 'stockistQueryUpdate']);

        add_filter('excerpt_more', [$this, 'tidyExcerpt']);

        $this->registerShortCodes();
        $this->registerFilters();
        $this->registerActions();
        $this->setupControllers();

        add_action('admin_init', [$this, 'deactivatePlugins']);
        add_action('wp_loaded', [$this, 'addSettingsPage']);

        add_filter('acf/fields/google_map/api', [$this, 'acfApiKey']);

        add_filter('posts_join', [$this, 'tax_search_join']);
        add_filter('posts_where', [$this, 'tax_search_where']);
        add_filter('posts_groupby', [$this, 'tax_search_groupby']);

        add_image_size('banner', 1920);
        add_image_size('overlay-image', 1100);
        add_image_size('prod-thumbnail', 300, 300, ['center', 'top']);

        if (function_exists('get_field')) {
            add_theme_support('soil-google-analytics', get_field('google_analytics_code', 'option'));
        }

    }

    private function registerShortCodes()
    {
        add_shortcode('socialmedia', [$this, 'getSocialMedia']);
    }

    private function registerFilters()
    {
        add_filter('excerpt_length', function ($length) {
            return 20;
        });

        add_filter('excerpt_more', [$this, 'changeExcerptReadMore']);

    }

    function changeExcerptReadMore($more) {
        global $post;
        return '<a href="'. get_permalink($post->ID) . '"><br>Read more <i class="far fa-arrow-right"></i></a>';
    }


    private function registerActions()
    {
        add_action('init', [$this, 'registerPostTypes']);
        add_action('init', [$this, 'registerTaxonomies']);
        add_action('wp_dashboard_setup', [$this, 'importStockistsWidget']);


    }

    private function setupControllers()
    {
        new WooCommerceController();
        new UserController();
        new BuildAnOutfitController();
        new ClanTartanController();
    }

    function importStockistsWidget()
    {
        global $wp_meta_boxes;
        wp_add_dashboard_widget('importStockists', 'Import Stockists', [$this, 'importStockistsForm']);
    }

    function importStockistsForm()
    {
        echo
        '<form action="" method="post" enctype="multipart/form-data" name="importStockist" id="importStockist">
            <input type="file" name="stockistCSV" id="stockistCSV" accept=".csv">
            <input type="submit" value="Upload Stockists" name="submit">
        </form>';


        if (isset($_POST['submit']) && ($_FILES['stockistCSV']['name'] != "")) {
            $this->importStockists();
        }
    }

    function importStockists()
    {
        ini_set('max_execution_time', 300);
        $stores = $this->getStockistsFromCSV($_FILES['stockistCSV']['tmp_name']);
        $googleKey = get_field('google_maps_key', 'option');

        foreach ($stores as $store) {
            $reference = $store['Customer Name'] . " " . $store['E-mail'];
            $postId = $this->checkIfExist($reference);
            if ($postId === false) {
                $fullAddress = $this->geocode($googleKey, $store['Customer Name'] . ', ' . $store['Address Line1'] . ', ' . $store['Address Line2'] . ', ' . $store['Address Line3'] . ', ' . $store['Address Line4'] . ', ' . $store['Postcode']);
                $this->insertStockist($reference, $fullAddress, $store);
            } else {
                $this->updateStockist($postId, $store);
            }
        }
    }

    private function getStockistsFromCSV($fileName)
    {
        $stores = array_map('str_getcsv', file($fileName));
        $header = array_shift($stores);
        $csv = [];
        foreach ($stores as $row) {
            $csv[] = array_combine($header, $row);
        }

        return $csv;
    }

    private function checkIfExist($reference)
    {
        $existingPost = get_posts([
            'post_type' => 'stockist',
            'meta_key' => 'reference',
            'meta_value' => $reference,
        ]);
        $postId = count($existingPost) > 0 ? $existingPost[0]->ID : false;

        return $postId;
    }

    public function geocode($key, $address)
    {
        $prepAddr = str_replace(' ', '+', $address);
        $geocode = file_get_contents('https://maps.google.com/maps/api/geocode/json?address=' . $prepAddr . '&sensor=false&key=' . $key);
        $output = json_decode($geocode);
        $latitude = $output->results[0]->geometry->location->lat;
        $longitude = $output->results[0]->geometry->location->lng;

        $fullAddress = [
            "address" => $address,
            "lat" => $latitude,
            "lng" => $longitude,
        ];

        return $fullAddress;
    }

    private function insertStockist($reference, $fullAddress, $store)
    {
        $postId = wp_insert_post([
            'post_title' => $store['Customer Name'],
            'post_status' => 'publish',
            'post_type' => 'stockist',
        ]);

        add_post_meta($postId, 'location', $fullAddress, true);
        add_post_meta($postId, '_location', 'field_5c77fb63e1c1e', true);

        add_post_meta($postId, 'website_url', $store['connected url'], true);
        add_post_meta($postId, '_website_url', 'field_5c7cf5459bfc2', true);

        add_post_meta($postId, 'telephone_number', $store['Phone Number'], true);
        add_post_meta($postId, '_telephone_number', 'field_5c7cf5759bfc4', true);

        add_post_meta($postId, 'reference', $reference, true);
        add_post_meta($postId, '_reference', 'field_5c8b708d919cc', true);

        echo $postId . " has been created<br/>";
    }

    private function updateStockist($postId, $store)
    {
        update_field('website_url', $store['connected url'], $postId);
        update_field('telephone_number', $store['Phone Number'], $postId);
        echo $postId . " has been updated <br/>";
    }

    public function acfApiKey($api)
    {

        $api['key'] = 'AIzaSyAGsKZ5PP8iURMpB4kRQKcpLQps-szGeqw';

        return $api;
    }

    public function deactivatePlugins()
    {
        if (WP_DEBUG === true) {
            deactivate_plugins('wordfence/wordfence.php');
        }
    }

    function stockistQueryUpdate($query)
    {

        if (!is_admin() && is_main_query() && is_post_type_archive('stockist')) {
            $query->set('posts_per_page', '200');
            $query->set('orderby', 'title');
            $query->set('order', 'ASC' );
        }

    }


    public function addSettingsPage()
    {
        acf_add_options_sub_page([
            'title' => 'Stockist Settings',
            'parent' => 'edit.php?post_type=stockist',
            'capability' => 'manage_options',
        ]);

        acf_add_options_sub_page([
            'title' => 'Distributor Settings',
            'parent' => 'edit.php?post_type=distributor',
            'capability' => 'manage_options',
        ]);
    }

    public function registerMenus()
    {
        register_nav_menus([
            'main_menu' => 'Main Menu',
            'push_menu' => 'Push Menu',
            'footer_menu_one' => 'Footer Menu One',
            'footer_menu_two' => 'Footer Menu Two',
            'footer_menu_three' => 'Footer Menu Three',
            'footer_menu_four' => 'Footer Menu Four',
        ]);
    }

    public function tidyExcerpt()
    {
        return '<span class="read-more"><a href="' . get_permalink() . '">Read More</a></span>';
    }

    public function registerPostTypes()
    {
        Post\Event::registerPostType();
        Post\CaseStudy::registerPostType();
        Post\Stockist::registerPostType();
        Post\Distributor::registerPostType();
    }

    public function registerTaxonomies()
    {
        new Colour();
        new FabricStyle();
    }

    public function getSocialMedia()
    {
        $socialmedia = '<nav class="social-media social-media-links col-md">';

        if (have_rows('social_media_channels', 'option')) {
            $socialmedia .= '<p>';
            while (have_rows('social_media_channels', 'option')) {
                the_row();
                $socialmedia .= '<a href="' . get_sub_field("full_url") . '" target="_blank" rel="noopener"><i class="' . get_sub_field("fa5_icon_class") . '"></i></a>';
            }
            $socialmedia .= '</p>';
        }

        $socialmedia .= '</nav>';
        return $socialmedia;
    }

    public function tax_search_join($join)
    {
        global $wpdb;

        if (!is_admin() && is_search()) {
            $join .= "
                INNER JOIN
                {$wpdb->term_relationships} ON {$wpdb->posts}.ID = {$wpdb->term_relationships}.object_id
                INNER JOIN
                {$wpdb->term_taxonomy} ON {$wpdb->term_taxonomy}.term_taxonomy_id = {$wpdb->term_relationships}.term_taxonomy_id
                INNER JOIN
                {$wpdb->terms} ON {$wpdb->terms}.term_id = {$wpdb->term_taxonomy}.term_id
            ";
        }

        return $join;
    }

    public function tax_search_where($where)
    {
        global $wpdb;

        if (!is_admin() && is_search()) {
            // add the search term to the query
            $where .= " OR
            (
            {$wpdb->term_taxonomy}.taxonomy LIKE 'product_cat'
            AND
            {$wpdb->terms}.name Like '%" . $wpdb->esc_like(get_query_var('s')) . "%'
            ) ";
        }

        return $where;
    }

    public function tax_search_groupby($groupby)
    {
        global $wpdb;
        if (!is_admin() && is_search()) {
            $groupby = "{$wpdb->posts}.ID";
        }
        return $groupby;
    }
}
