<?php
namespace Blue2\Scruffy;

use Blue2\Scruffy\Controllers\ProductController;

class Site extends \Blue2\Common\Site
{
    public function __construct()
    {
        parent::__construct();

        add_action('after_setup_theme', [$this, 'registerMenus']);
        add_action('after_setup_theme', [$this, 'customImageSizes']);
        add_filter('excerpt_more', [$this, 'tidyExcerpt']);
        add_action('admin_menu', [$this, 'customAdminPages']);
        add_action('pre_get_posts', [$this, 'orderDefaultQueryBy']);

        $this->registerShortCodes();
        $this->registerFilters();
        $this->registerActions();
        $this->setupControllers();
    }

    public function registerMenus()
    {
        register_nav_menus([
            'top_english_menu' => 'Top Menu',
            'english_menu' => 'Menu',
            'footer_english_menu' => 'Footer Menu',
            'footer_english_terms_menu' => 'Footer T&C',
        ]);
    }

    public function addLanguageQueryVar()
    {
        add_rewrite_tag('%language%', '([\w\-]+)');
    }

    public function tidyExcerpt()
    {
        return ' &hellip; <a href="' . get_permalink() . '">Continued</a>';
    }

    private function registerShortCodes()
    {
        add_shortcode('dynamic_topic', [$this, 'dynamicTopic']);
    }

    private function registerFilters()
    {
        add_filter('acf/fields/google_map/api', [$this,'acf_google_map_api']);
    }

    private function registerActions()
    {
        add_action('init', [$this, 'registerPostTypes']);
        add_action('wp_enqueue_scripts', [$this, 'localizeScripts'], 101);
        add_action('init', [$this, 'registerTaxonomies']);
        add_action('init', [$this, 'addLanguageQueryVar']);
    }

    private function setupControllers()
    {
        new ProductController();
    }

    public function registerPostTypes()
    {
        Post\Product::registerPostType();
        Post\Person::registerPostType();
    }

    public function customAdminPages()
    {
        if (function_exists('acf_add_options_page')) {
            acf_add_options_page([
                'title' => 'Post archive settings',
                'parent' => 'edit.php',
                'capability' => 'edit_posts',
//                'post_id' => 'post_archive_settings'
            ]);

            acf_add_options_page([
                'title' => 'Person archive settings',
                'parent' => 'edit.php?post_type=person',
                'capability' => 'edit_posts',
//                'post_id' => 'person_archive_settings'
            ]);

            acf_add_options_page([
                'title' => 'Product archive settings',
                'parent' => 'edit.php?post_type=product',
                'capability' => 'edit_posts',
//                'post_id' => 'product_archive_settings'
            ]);
        }
    }
    public function registerTaxonomies()
    {
        Post\Product::registerTaxonomy();
    }

    public function customImageSizes()
    {
        add_image_size('fullwidth', 1920); // height will be recalculated to keep the dimensions
        add_image_size('featured-post', 750, 350, true);
    }

    public function localizeScripts()
    {
        wp_localize_script('b2/js', 'ajaxObject', [ 'ajax_url' => admin_url('admin-ajax.php')]);
    }

    public function orderDefaultQueryBy($query)
    {
        if (!is_admin() && $query->is_main_query()) {
            if (is_post_type_archive('product') || is_tax('product-category')) {
                $query->set('orderby', 'meta_value_num');
                $query->set('meta_key', 'product_order');
                $query->set('order', 'ASC');
                $query->set('posts_per_page', 100);
            } elseif (is_post_type_archive('person')) {
                $query->set('orderby', 'meta_value_num');
                $query->set('meta_key', 'person_order');
                $query->set('order', 'ASC');
                $query->set('posts_per_page', 100);
            }
        }
    }

    public function dynamicTopic()
    {
        if (isset($_GET['enquiry-topic']) && $_GET['enquiry-topic'] !== '') {
            return $_GET['enquiry-topic'];
        }
    }

    public static function getPageLanguage()
    {
        if (defined( 'ICL_LANGUAGE_CODE' )) {
            $language = ICL_LANGUAGE_CODE;
            switch ($language) {
                case 'es':
                    $pageLanguage = 'spanish';
                    break;
                case 'zh-hans':
                    $pageLanguage = 'chinese';
                    break;
                case 'pt-br':
                    $pageLanguage = 'portuguese';
                    break;
                default:
                    $pageLanguage = 'english';
            }
        } else {
            $pageLanguage = get_field('language');
        }
        return $pageLanguage;
    }

    public static function inlineSVG($location, $class, $label = '', $hidden = false)
    {
        $path = get_template_directory() . $location;
        $ariaValue = $hidden == 1 ? 'true' : 'false';
        $file = file_get_contents($path);
        $file = str_replace(
            '<?xml version="1.0" encoding="UTF-8"?>',
            '',
            $file
        );
        echo str_replace(
            '<svg',
            '<svg class="' . $class . '" role="img" aria-label="' . $label . '" aria-hidden="' . $ariaValue .'"',
            $file
        );
    }

    public function acf_google_map_api( $api ){
        $api['key'] = get_field('google_maps_api_key', 'option');
        return $api;
    }
}
