<?php
/**
 * The public-facing functionality of the plugin.
 *
 * @link       https://wpgeodirectory.com
 * @since      1.0.0
 *
 * @package    WPInv_SagePay
 * @subpackage WPInv_SagePay/includes
 */

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the dashboard-specific stylesheet and JavaScript.
 *
 * @package    WPInv_SagePay
 * @subpackage WPInv_SagePay/admin
 * @author     GeoDirectory <info@wpgeodirectory.com>
 */
class WPInv_SagePay_Public {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $name    The ID of this plugin.
	 */
	private $name;
    
    /**
	 * The domain specified for this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $domain    The domain identifier for this plugin.
	 */
	private $domain;
    
	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;
    
	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @var      string    $name       The name of the plugin.
	 * @var      string    $version    The version of this plugin.
	 */
	public function __construct( $name, $domain, $version ) {
		$this->name = $name;
        $this->domain = $domain;
		$this->version = $version;
	}

	/**
	 * Register the stylesheets for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {
	}

	/**
	 * Register the stylesheets for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {
	}

    public function checkout_cc_form( $invoice_id ) {
        return false;
    }
    
    public function process_payment( $purchase_data ) {
        if ( ! wp_verify_nonce( $purchase_data['gateway_nonce'], 'wpi-gateway' ) ) {
            wp_die( __( 'Nonce verification has failed', $this->domain ), __( 'Error', $this->domain ), array( 'response' => 403 ) );
        }
        
        // Collect payment data
        $payment_data = array(
            'price'         => $purchase_data['price'],
            'date'          => $purchase_data['date'],
            'user_email'    => $purchase_data['user_email'],
            'invoice_key'   => $purchase_data['invoice_key'],
            'currency'      => wpinv_get_currency(),
            'items'         => $purchase_data['items'],
            'user_info'     => $purchase_data['user_info'],
            'cart_details'  => $purchase_data['cart_details'],
            'gateway'       => 'sagepay',
            'status'        => 'wpi-pending'
        );
        
        // Record the pending payment
        $invoice = wpinv_get_invoice( $purchase_data['invoice_id'] );
        $username = wpinv_get_option( 'sagepay_username' );
        $encrypt_password = wpinv_get_option( 'sagepay_encrypt_password' );

        if ( !empty( $invoice ) && $username && $encrypt_password ) {
            $customer_name = stripslashes( html_entity_decode( $invoice->get_user_full_name(), ENT_COMPAT, 'UTF-8' ) );
            $first_name = stripslashes( html_entity_decode( $invoice->get_first_name(), ENT_COMPAT, 'UTF-8' ) );
            $last_name = stripslashes( html_entity_decode( $invoice->get_last_name(), ENT_COMPAT, 'UTF-8' ) );
            $address = stripslashes( html_entity_decode( $invoice->get_address(), ENT_COMPAT, 'UTF-8' ) );
            $city = stripslashes( html_entity_decode( $invoice->user_info['city'], ENT_COMPAT, 'UTF-8' ) );
            $state = stripslashes( html_entity_decode( $invoice->user_info['state'], ENT_COMPAT, 'UTF-8' ) );
            $country = stripslashes( html_entity_decode( $invoice->user_info['country'], ENT_COMPAT, 'UTF-8' ) );
            $phone = stripslashes( html_entity_decode( $invoice->user_info['phone'], ENT_COMPAT, 'UTF-8' ) );
            $postcode = stripslashes( html_entity_decode( $invoice->user_info['zip'], ENT_COMPAT, 'UTF-8' ) );
            
            $return_url = add_query_arg( array( 'payment-confirm' => 'sagepay', 'invoice-id' => $invoice->ID ), wpinv_get_success_page_uri() );
            $cancel_url = wpinv_get_failed_transaction_uri( '?invoice-id=' . $invoice->ID );
            
            // loading the class
            require_once( WPINV_SAGEPAY_DIR . 'includes/SagePay.class.inc' );
            
            //asigning values to the object
            $sagePay = new SagePay();

            $sagePay->setVendorTxCode( $invoice->ID );
            $sagePay->setCurrency( $payment_data['currency'] );
            $sagePay->setAmount( wpinv_sanitize_amount( $invoice->get_total() ) );
            $sagePay->setDescription( stripslashes( html_entity_decode( $invoice->title, ENT_COMPAT, 'UTF-8' ) ) );
            $sagePay->setCustomerName( $customer_name );
            $sagePay->setCustomerEMail( $invoice->get_email() );
            $sagePay->setBillingSurname( $last_name );
            $sagePay->setBillingFirstnames( $first_name );
            $sagePay->setBillingAddress1( $address );
            $sagePay->setBillingCity( $city );
            $sagePay->setBillingPostCode( $postcode );
            $sagePay->setBillingState( $state );
            $sagePay->setBillingCountry( strtolower( $country ) );
            $sagePay->setBillingPhone( $phone );
            $sagePay->setDeliverySameAsBilling();
            $sagePay->setEncryptPassword( $encrypt_password );

            $sagePay->setSuccessURL( $return_url );
            $sagePay->setFailureURL( $cancel_url );
            
            $params = array();
            $params['VPSProtocol'] = '3.00';
            $params['TxType'] = 'PAYMENT';
            $params['Vendor'] = $username;
            $params['Crypt'] = $sagePay->getCrypt();
            $params = apply_filters( 'wpinv_sagepay_form_extra_parameters', $params, $invoice );
        
            $redirect_text = __( 'Redirecting to SagePay payment site, click on button if not redirected.', $this->domain );
            $redirect_text = apply_filters( 'wpinv_sagepay_redirect_text', $redirect_text, $invoice );
        ?>
<div class="wpi-sagepay-form" style="padding:20px;font-family:arial,sans-serif;text-align:center;color:#555">
<?php do_action( 'wpinv_sagepay_form_before', $invoice ); ?>
<h3><?php echo $redirect_text ;?></h3>
<form action="<?php echo $this->sagepay_payment_url(); ?>" name="wpi_sagepay_form" method="POST">
    <?php foreach ( $params as $param => $value ) { ?>
    <input type="hidden" value="<?php echo esc_attr( $value );?>" name="<?php echo esc_attr( $param );?>">
    <?php } ?>
    <?php do_action( 'wpinv_sagepay_form_parameters', $invoice ); ?>
    <input type="submit" name="wpi_sagepay_submit" value="<?php esc_attr_e( 'Proceed to SagePay', $this->domain ) ;?>">
</form>
<script type="text/javascript">document.wpi_sagepay_form.submit();</script>
<?php do_action( 'wpinv_sagepay_form_after', $invoice ); ?>
</div>
        <?php
        } else {
            wpinv_record_gateway_error( __( 'Payment Error', $this->domain ), sprintf( __( 'Payment failed while processing a SagePay payment. Payment data: %s', $this->domain ), json_encode( $payment_data ) ), $invoice );
            // If errors are present, send the user back to the purchase page so they can be corrected
            wpinv_send_back_to_checkout( '?payment-mode=' . $purchase_data['post_data']['wpi-gateway'] );
        }
    }
    
    public function handle_ipn() {
        // loading the class
        require_once( WPINV_SAGEPAY_DIR . 'includes/SagePay.class.inc' );
        
        if ( isset( $_REQUEST['crypt'] ) ) {
            $sagePay = new SagePay();
            $response = $sagePay->decode( $_REQUEST['crypt'] );
            
            if ( !empty( $response ) && !empty( $_REQUEST['invoice-id'] ) ) {
                $invoice_id = (int)$_REQUEST['invoice-id'];
                $invoice = wpinv_get_invoice( $invoice_id );
                
                if ( !empty( $invoice ) ) {
                    // Empty the shopping cart
                    wpinv_empty_cart();
                        
                    // Check status of response
                    if ( isset( $response['Status'] ) && $response['Status'] === 'OK' ) {
                        wpinv_update_payment_status( $invoice_id, 'publish' );
                        wpinv_set_payment_transaction_id( $invoice_id, $response['TxAuthNo'] );
                        wpinv_insert_payment_note( $invoice_id, sprintf( __( 'SagePay Transaction ID: %s', $this->domain ) , $response['TxAuthNo'] ) );
                        
                        wpinv_send_to_success_page( array( 'invoice_key' => $invoice->get_key() ) );
                    } else if( isset( $response['Status'] ) && $response['Status'] === 'ABORT' ) {                
                        wpinv_update_payment_status( $invoice_id, 'wpi-cancelled' );
                        
                        wpinv_record_gateway_error( 'error_transaction_failed', __( 'Transaction has been cancelled!', $this->domain ) );
                    } else {
                        wpinv_update_payment_status( $invoice_id, 'wpi-failed' );
                        
                        if ( isset( $response['StatusDetail'] ) ) {
                            wpinv_record_gateway_error( 'error_transaction_failed', __( 'Transaction Failed. ' . $response['StatusDetail'], $this->domain ) );
                        } else {
                            wpinv_record_gateway_error( 'error_transaction_failed', __( 'Transaction Failed with ' . $response['Status'] . ' status for Unknown Reason.', $this->domain ) );
                        }
                    }
                    
                    wpinv_send_to_failed_page( '?invoice-id=' . $invoice_id );
                } else {
                    wpinv_set_error( 'error_transaction_failed', __( 'Invoice the are you going to pay does not exists!', $this->domain ) );
                }
            } else {
                wpinv_set_error( 'error_transaction_failed', __( 'Invalid transaction request found!', $this->domain ) );
            }
            
            wpinv_send_back_to_checkout( '?payment-mode=' . $purchase_data['post_data']['wpi-gateway'] );
        }
    }
    
    public function sagepay_payment_url() {
        $redirect = wpinv_is_test_mode( 'sagepay' ) ? 'https://test.sagepay.com/gateway/service/vspform-register.vsp' 
                                                    : 'https://live.sagepay.com/gateway/service/vspform-register.vsp';
        
        return apply_filters( 'wpinv_sagepay_redirect', $redirect );
    }
}
