<?php
/**
 * This file processes the data sent to th endpoint and creates the job post.
 * It returns a message back to the send with either success (including the new job URL ) or failure.
 *
 * @package WP_Job_Manager_Broadbean
 */

// Start by getting the stored user name and password used to the feed.
$username = wpjmbb_get_username();
$password = wpjmbb_get_password();

// set logging to false - filterable.
$logging = wpjmbb_get_debugging();

/**
 * Get the contents of the feed provided by adcourier.
 * if you want to test the feed, by pulling an XML feed from a specific url.
 * you can use the wpjmbb_xml_feed_url filter.
 */
$xml_content = file_get_contents( apply_filters( 'wpjmbb_xml_feed_url', 'php://input' ) );

// if we have no xml content.
if ( false === $xml_content || '' === $xml_content ) {

	// echo an error.
	header( 'HTTP/1.1 401 Unauthorized' );
	esc_html_e( 'Feed contains no valid XML data.', 'wp-job-manager-broadbean' );
	die();

}

/**
 * Turn the raw posted data into a more usable object.
 * Each xml node is now an property of the object.
 */
$xml_params = simplexml_load_string( $xml_content );

// allow the job data delivered to be filtered.
$xml_params = apply_filters( 'wpjmbb_xml_job_data', $xml_params );

/**
 * Before we go any further - lets authenticate.
 * Check the username / password sent matches that stored.
 */
if ( wp_strip_all_tags( (string) $xml_params->username ) !== $username || wp_strip_all_tags( (string) $xml_params->password !== $password ) ) {

	// echo an error.
	header( 'HTTP/1.1 401 Unauthorized' );
	esc_html_e( 'Error: Sorry username and/or password are not valid.', 'wp-job-manager-broadbean' );
	die();

}

// if the current license is not valid.
if ( 'valid' !== wpjmbb_check_license() ) {

	// echo an error.
	header( 'HTTP/1.1 401 Unauthorized' );
	esc_html_e( 'Error: License failure.', 'wp-job-manager-broadbean' );
	die();

}

/* get the command for this job - either add/delete/edit */
$command = wp_strip_all_tags( (string) $xml_params->command );

// we are now authenticated there check which command we should be doing.
if ( 'add' === $command || '' === $command ) {

	/**
	 * Fires after the feed is authenticated but before any further processing of the XML.
	 *
	 * @param $xml_params are the parameters in the sent XML as a PHP object
	 */
	do_action( 'wpjmbb_feed_authenticated', $xml_params );

	/**
	 * Prepare the taxonomy terms to add.
	 * Defaults are for job_listing_category and job_listing_type taxonomies.
	 * Start by building a filterable array of taxonomy to handle.
	 */

	// disprove later when checking if there are any to add.
	$have_tax_terms = false;

	// get the taxonomy objects for jobs - just their names.
	$job_taxonomies = get_object_taxonomies(
		'job_listing',
		'names'
	);

	/* if we have taxonomies */
	if ( ! empty( $job_taxonomies ) ) {

		/* set up holding array to fill with terms*/
		$tax_terms = array();

		/**
		 * Loop through each of the taxonomies we have preparing it for adding to the post.
		 * This result in array like so:
		 * array(
		 *		'job_listing_category'	=> array(
		 *				'General',
		 *				'IT'
		 *		),
		 *		'job_listing_type'		=> array(
		 *				'Temporary'
		 *		)
		 * )
		 */
		foreach ( $job_taxonomies as $taxonomy ) {

			/* check the xml has content */
			if ( ! empty( $xml_params->$taxonomy ) ) {

				/* add the prepared terms to our terms array */
				$tax_terms[ $taxonomy ] = wpjmbb_prepare_terms( $xml_params->$taxonomy, $taxonomy );

				$have_tax_terms = true;

			}
		} // End foreach().
	} // End if().

	// default job post author is guest.
	$author = 0;

	/* start by getting the author */
	$sent_author = wp_strip_all_tags( $xml_params->job_author );

	// check if the sent author is a number.
	if ( is_numeric( $sent_author ) ) {

		// set the author of this new job to this user id.
		$author = absint( $sent_author );

	} else {

		// author is not set to integer or a wp user id so lets check it is a valid email.
		if ( is_email( $sent_author ) ) {

			// check for a WordPress user with this email address.
			$user = get_user_by( 'email', $sent_author );

			// if we have a user returned.
			if ( false !== $user ) {

				// set the author as the users user id.
				$author = $user->ID;

			}
		}
	}

	$wpjmbb_insert_post_args = array(
		'post_type'    => 'job_listing',
		'post_title'   => wp_strip_all_tags( (string) $xml_params->job_title ),
		'post_content' => wp_kses_post( $xml_params->job_description ),
		'post_status'  => 'publish',
		'post_author'  => absint( $author ),
	);

	// get any job with this reference already.
	$existing_job = wpjmbb_get_job_by_reference( wp_strip_all_tags( (string) $xml_params->job_reference ) );

	// if we have a job with this reference we should be editing it.
	if ( false !== $existing_job ) {

		// add the id of the job into the insert post array.
		$wpjmbb_insert_post_args['ID'] = $existing_job;

	}

	/**
	 * Lets now insert the actual post into WordPress.
	 * uses the wp_insert_post function to do this.
	 * if this works it will return the post id of the job added.
	 */
	$job_post_id = wp_insert_post(
		apply_filters(
			'wpjmbb_new_job_post_args',
			$wpjmbb_insert_post_args,
			$xml_params
		)
	);

	/**
	 * Lets check that the job was added.
	 * checking for a job id present in the variable.
	 */
	if ( 0 !== $job_post_id ) {

		/**
		 * If wpbb logging is set to true the plugin will save the raw incoming XML feed.
		 * as post meta for this job with the key being _wpjmbb_raw_bb_feed.
		 * it also adds the date before the XML output.
		 */
		if ( true === $logging ) {

			$debug_content = array();

			// combine the XML with the current date stamp.
			$debug_content['sent_date'] = date( 'd:m:Y H:i:s' );
			$debug_content['sent_xml'] = $xml_content;

			// lets save the raw posted data in post meta for this job.
			add_post_meta(
				$job_post_id, // this is id of the job we have just added.
				'_wpjmbb_raw_feed', // this is the meta key to store the post meta in.
				$debug_content, // this is value to store - sent from broadbean/logicmelon.
				true
			);

		} // End if().

		/**
		 * Handle assign taxonomy terms to the newly added job post.
		 * check we have a job type to add.
		 */
		if ( true === $have_tax_terms ) {

			$terms_added = array();

			/**
			 * Job was added successfully.
			 * start by looping through the tax terms ids to add to this job.
			 */
			foreach ( $tax_terms as $tax => $terms ) {

				$terms_added[ $tax ] = wp_set_post_terms(
					(int) $job_post_id,
					$terms,
					$tax
				);

				/**
				 * Fires after the term has been added.
				 *
				 * @param (int) $job_post_id is the post id for the added job.
				 * @param (array) $terms terms to be added.
				 * @param (string) $tax taxonomy of the term.
				 */
				do_action( 'wpjmbb_job_term_added', $job_post_id, $terms, $tax );

			}
		}

		/* is the sidebload image function around */
		if ( ! function_exists( 'media_handle_sideload' ) ) {

			require_once( ABSPATH . 'wp-admin/includes/media.php' );
			require_once( ABSPATH . 'wp-admin/includes/file.php' );
			require_once( ABSPATH . 'wp-admin/includes/image.php' );

		}

		/**
		 * Get all of the fields added by the core plugin.
		 * and third party plugins.
		 */
		$job_fields = wpjmbb_get_fields();

		// prepare the job fields array.
		$job_fields = wpjmbb_prepare_job_fields( $job_fields );

		// check we have job fields to process.
		if ( ! empty( $job_fields ) ) {

			// loop through each field.
			foreach ( $job_fields as $meta_key => $field ) {

				// remove the underscrore from the first part of the key.
				$bb_field = ltrim( $meta_key, '_' );

				// change any dashes for underscore.
				$bb_field = str_replace( '-', '_', $bb_field );

				// lets check that the field sent is not empty.
				if ( '' !== $xml_params->$bb_field ) {

					// get the meta value.
					$value = (string) wp_strip_all_tags( $xml_params->$bb_field );

					// check we have a type set.
					if ( isset( $field['type'] ) ) {

						// if this is an upload field.
						if ( 'file' === $field['type'] ) {

							// get the url from the posted xml field.
							$url = $xml_params->$bb_field;

							// perform a download on the url.
							$tmp = download_url( $url );

							// check for any download errors - for example if the url is a 404.
							if ( ! is_wp_error( $tmp ) ) {

								$file_array = array(
									'name' => basename( $url ),
									'tmp_name' => $tmp,
								);

								// Check for download errors
								if ( is_wp_error( $tmp ) ) {
									@unlink( $file_array[ 'tmp_name' ] );
									return $tmp;
								}

								$id = media_handle_sideload( $file_array, $job_post_id );

								// Check for handle sideload errors.
								if ( is_wp_error( $id ) ) {
									@unlink( $file_array['tmp_name'] );
									return $id;
								}

								// get the url from the newly upload file.
								$value = wp_get_attachment_url( $id );

								// if this is the company logo.
								if ( 'company_logo' === $bb_field ) {

									/* set the attachment as the featured image */
									update_post_meta( $job_post_id, '_thumbnail_id', $id );

								}
							} else { // downloading the url has failed.

								// set the value to an empty string - save this as post meta instead of value sent.
								$value = '';

							} // End if().
						} // End if().
					} // End if().

					// add the meta data.
					update_post_meta( $job_post_id, $meta_key, $value );

					/**
					 * Fires after the field has been added.
					 *
					 * @param (int) $job_post_id is the post id for the added job
					 * @param (array) $field term to be added
					 */
					do_action( 'wpjmbb_job_field_added', $job_post_id, $field );

				} // End if().
			} // End foreach().
		} // End if().

		// Fires after a job has been added and after tax and meta data saved.
		do_action( 'wpjmbb_job_added', $job_post_id, $xml_params );

		// added or updated.
		if ( false !== $existing_job ) {
			$new_command = 'updated';
		} else {
			$new_command = 'added';
		}

		// everything appears to have worked therefore lets output a success message.
		echo apply_filters(
			'wpjmbb_job_added_success_message',
			'Success: This Job has been ' . $new_command . ' and has a post ID of ' . $job_post_id . '. The permalink to this job is: ' . get_permalink( $job_post_id ),
			$job_post_id
		);

	} else { // call to wp_insert_post returned zero and the job was not added.

		// output a error to the indicate the problem.
		echo apply_filters(
			'wpjmbb_job_added_failure_message',
			'Error: There was an error, the job was not published.',
			$job_post_id
		);

	} // End if().
} // End if().

// if the command is to delete this job.
elseif ( 'delete' === $command ) {

	// get the sent job reference.
	$job_reference = wp_strip_all_tags( (string) $xml_params->job_reference );

	// get the job id of the job that has the sent reference.
	$job_id = wpjmbb_get_job_by_reference( $job_reference );

	/* if we have a job matching the sent reference */
	if ( false !== $job_id ) {

		// setup string of deleted posts.
		$deleted_posts = '<p>Post or posts deleted. The following post or posts were deleted: ';

		// delete the post.
		$deleted = wp_delete_post( $job_id );

		// check whether the post was deleted.
		if ( false !== $deleted ) {

			/* clear the wpjm cache for job listings */
			WP_Job_Manager_Cache_Helper::get_transient_version( 'get_job_listings', true );

			/* add to delete posts string */
			$deleted_posts .= $job_id . ' | Job Reference: ' . $job_reference;

			/* output confirmation message */
			echo $deleted_posts . '</p>';

			// fires after the job is deleted and passed the deleted posts post object.
			do_action( 'wpjmbb_job_trashed', $deleted );

		} else { // No delete took place.

			// echo an error.
			header( 'HTTP/1.1 500 Unauthorized' );
			esc_html_e( 'Delete failed. This jobs either does not exist, or it is already in the trash.', 'wp-job-manager-broadbean' );
			die();

		}
	} else {

		// echo an error.
		header( 'HTTP/1.1 500 Unauthorized' );
		esc_html_e( 'Error: A delete command was sent, however the job to delete cannot be found. The job reference sent, does not match a job on this site.', 'wp-job-manager-broadbean' );
		die();

	}
}

// stop any further loading.
die();
