<?php
/**
 * Plugin Name: WooCommerce Royal Mail
 * Plugin URI: https://woocommerce.com/products/royal-mail/
 * Description: Offer Royal Mail shipping rates automatically to your customers. Prices according to <a href="https://www.royalmail.com/sites/royalmail.com/files/2020-02/royal-mail-our-prices-valid-from-23-march-2020.pdf">the 2020 price guide</a>.
 * Version: 2.5.31
 * Author: WooCommerce
 * Author URI: https://woocommerce.com/
 * Text Domain: woocommerce-shipping-royalmail
 * Domain Path: /languages
 * Copyright: © 2020 WooCommerce
 * License: GNU General Public License v3.0
 * License URI: http://www.gnu.org/licenses/gpl-3.0.html
 * WC tested up to: 4.4
 * WC requires at least: 2.6
 * Tested up to: 5.5
 *
 * Woo: 182719:03839cca1a16c4488fcb669aeb91a056
 *
 * @package WC_RoyalMail
 */

/**
 * WooCommerce fallback notice.
 *
 * @since 2.5.24
 * @return void
 */
function woocommerce_shipping_royalmail_missing_wc_notice() {
	/* translators: %s WC download URL link. */
	echo '<div class="error"><p><strong>' . sprintf( esc_html__( 'Royalmail requires WooCommerce to be installed and active. You can download %s here.', 'woocommerce-shipping-royalmail' ), '<a href="https://woocommerce.com/" target="_blank">WooCommerce</a>' ) . '</strong></p></div>';
}

/**
 * Only load the plugin if WooCommerce is activated
 */
if ( ! class_exists( 'WC_RoyalMail' ) ) :
	define( 'WOOCOMMERCE_SHIPPING_ROYALMAIL_VERSION', '2.5.31' ); // WRCS: DEFINED_VERSION.

	/**
	 * Main Royal Mail class
	 */
	class WC_RoyalMail {
		/**
		 * Plugin's version.
		 *
		 * @since 2.5.0 introduced.
		 *
		 * @var string
		 */
		public $version = WOOCOMMERCE_SHIPPING_ROYALMAIL_VERSION;

		/**
		 * Constructor.
		 */
		public function __construct() {
			add_action( 'admin_init', array( $this, 'maybe_install' ), 5 );
			add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), array( $this, 'plugin_action_links' ) );
			add_action( 'woocommerce_shipping_init', array( $this, 'shipping_init' ) );
			add_filter( 'woocommerce_shipping_methods', array( $this, 'shipping_methods' ) );
			add_action( 'admin_notices', array( $this, 'environment_check' ) );
		}

		/**
		 * Check environment.
		 *
		 * Hooked into `admin_notices` so it'll render admin notice if there's
		 * a failed check.
		 *
		 * @access public
		 * @return void
		 */
		public function environment_check() {

			if ( 'GBP' !== get_woocommerce_currency() ) {
				echo '<div class="error">
					<p>' . sprintf( __( 'Royal Mail requires that the <a href="%s">currency</a> is set to Pound sterling.', 'woocommerce-shipping-royalmail' ), admin_url( 'admin.php?page=wc-settings&tab=general' ) ) . '</p>
				</div>';
			}

			if ( 'GB' !== WC()->countries->get_base_country() ) {
				echo '<div class="error">
					<p>' . sprintf( __( 'Royal Mail requires that the <a href="%s">base country/region</a> is set to United Kingdom.', 'woocommerce-shipping-royalmail' ), admin_url( 'admin.php?page=wc-settings&tab=general' ) ) . '</p>
				</div>';
			}
		}

		/**
		 * Add plugin action links to the plugins page.
		 *
		 * @param array $links Links.
		 *
		 * @return array Links.
		 */
		public function plugin_action_links( $links ) {
			$plugin_links = array(
				'<a href="https://woocommerce.com/my-account/create-a-ticket?broken=primary&select=182719">' . __( 'Support', 'woocommerce-shipping-royalmail' ) . '</a>',
				'<a href="https://www.woocommerce.com/products/royal-mail">' . __( 'Docs', 'woocommerce-shipping-royalmail' ) . '</a>',
			);
			return array_merge( $plugin_links, $links );
		}

		/**
		 * Load our shipping class.
		 */
		public function shipping_init() {
			require_once dirname( __FILE__ ) . '/includes/class-wc-royalmail-privacy.php';
			include_once dirname( __FILE__ ) . '/includes/class-wc-shipping-royalmail.php';
		}

		/**
		 * Add our shipping method to woocommerce.
		 *
		 * @param array $methods Shipping methods.
		 *
		 * @return array Shipping methods.
		 */
		public function shipping_methods( $methods ) {
			$methods['royal_mail'] = 'WC_Shipping_Royalmail';
			return $methods;
		}

		/**
		 * Checks the plugin version.
		 *
		 * @since 2.5.0
		 * @version 2.5.0
		 *
		 * @return bool
		 */
		public function maybe_install() {
			// Only need to do this for versions less than 2.5.0 to migrate
			// settings to shipping zone instance.
			$doing_ajax = defined( 'DOING_AJAX' ) && DOING_AJAX;
			if ( ! $doing_ajax
				&& ! defined( 'IFRAME_REQUEST' )
				&& version_compare( get_option( 'wc_royal_mail_version' ), '2.5.0', '<' ) ) {

				$this->install();

			}

			return true;
		}

		/**
		 * Update/migration script.
		 *
		 * @since 2.5.0
		 * @version 2.5.0
		 */
		public function install() {
			// Get all saved settings and cache it.
			$royal_mail_settings = get_option( 'woocommerce_royal_mail_settings', false );

			// If settings exists.
			if ( $royal_mail_settings ) {
				global $wpdb;

				// Unset un-needed settings.
				unset( $royal_mail_settings['enabled'] );
				unset( $royal_mail_settings['availability'] );
				unset( $royal_mail_settings['countries'] );

				// First add it to the "rest of the world" zone when no Royal Mail
				// instance.
				if ( ! $this->is_zone_has_royal_mail( 0 ) ) {
					$wpdb->query( $wpdb->prepare( "INSERT INTO {$wpdb->prefix}woocommerce_shipping_zone_methods ( zone_id, method_id, method_order, is_enabled ) VALUES ( %d, %s, %d, %d )", 0, 'royal_mail', 1, 1 ) );
					// Add settings to the newly created instance to options table.
					$instance = $wpdb->insert_id;
					add_option( 'woocommerce_royal_mail_' . $instance . '_settings', $royal_mail_settings );
				}
			}
			update_option( 'wc_royal_mail_version', $this->version );
		}

		/**
		 * Helper method to check whether given zone_id has royal_mail method instance.
		 *
		 * @since 2.5.0
		 * @version 2.5.0
		 *
		 * @param int $zone_id Zone ID.
		 *
		 * @return bool True if given zone_id has royal_mail method instance.
		 */
		public function is_zone_has_royal_mail( $zone_id ) {
			global $wpdb;

			return (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(instance_id) FROM {$wpdb->prefix}woocommerce_shipping_zone_methods WHERE method_id = 'royal_mail' AND zone_id = %d", $zone_id ) ) > 0;
		}
	}
endif;

add_action( 'plugins_loaded', 'woocommerce_shipping_royalmail_init' );

/**
 * Initializes extension.
 *
 * @since 2.5.24
 * @return void
 */
function woocommerce_shipping_royalmail_init() {
	load_plugin_textdomain( 'woocommerce-shipping-royalmail', false, plugin_basename( dirname( __FILE__ ) ) . '/languages' );

	if ( ! class_exists( 'WooCommerce' ) ) {
		add_action( 'admin_notices', 'woocommerce_shipping_royalmail_missing_wc_notice' );
		return;
	}

	new WC_RoyalMail();
}
