<?php
/**
 * First class rate.
 *
 * @package WC_RoyalMail/Rate
 */

/**
 * RoyalMail_Rate_First_Class class.
 *
 * Updated on 2020-03-23 as per https://www.royalmail.com/sites/royalmail.com/files/2020-07/royal-mail-our-prices-1-september-2020.pdf
 * See UK Standard page 6.
 */
class RoyalMail_Rate_First_Class extends RoyalMail_Rate {

	/**
	 * ID/Name of rate.
	 *
	 * @var string
	 */
	protected $rate_id = 'first_class';

	const COMPENSATION_UP_TO_VALUE = 20;

	/**
	 * Pricing bands
	 *
	 * Key is size (e.g. 'letter') and value is an array where key is weight in
	 * gram and value is the price (in penny).
	 *
	 * @var array
	 */
	protected $bands = array(
		'2020' => array(
			'letter'              => array(
				100 => 76,
			),
			'large-letter'        => array(
				100 => 115,
				250 => 164,
				500 => 214,
				750 => 295,
			),
			'small-parcel-wide'   => array(
				1000 => 370,
				2000 => 557,
			),
			'small-parcel-deep'   => array(
				1000 => 370,
				2000 => 557,
			),
			'small-parcel-bigger' => array(
				1000 => 370,
				2000 => 557,
			),
			'medium-parcel'       => array(
				1000  => 590,
				2000  => 902,
				5000  => 1585,
				10000 => 2190,
				20000 => 3340,
			),
		),
	);

	/**
	 * Shipping boxes.
	 *
	 * @var array
	 */
	protected $boxes = array(
		'letter' => array(
			'length'   => 240, // Max L in mm.
			'width'    => 165, // Max W in mm.
			'height'   => 5,   // Max H in mm.
			'weight'   => 100, // Max Weight in grams.
		),
		'large-letter' => array(
			'length'   => 353,
			'width'    => 250,
			'height'   => 25,
			'weight'   => 750,
		),
		'small-parcel-wide' => array(
			'length' => 450,
			'width'  => 350,
			'height' => 80,
			'weight' => 2000,
		),
		'small-parcel-deep' => array(
			'length' => 350,
			'width'  => 250,
			'height' => 160,
			'weight' => 2000,
		),
		'small-parcel-bigger' => array(
			'length' => 450,
			'width'  => 350,
			'height' => 160,
			'weight' => 2000,
		),
		'medium-parcel' => array(
			'length'   => 610,
			'width'    => 460,
			'height'   => 460,
			'weight'   => 20000,
		),
	);

	/**
	 * Get quotes for this rate.
	 *
	 * @param  array  $items to be shipped.
	 * @param  string $packing_method the method selected.
	 * @param  string $destination Address to ship to.
	 * @return array
	 */
	public function get_quotes( $items, $packing_method, $destination, $boxes = array(), $instance_id = '' ) {
		$class_quote           = 0;
		$packages              = $this->get_packages( $items, $packing_method );
		$options               = $this->get_instance_options( $instance_id );
		$compensation_optional = ( ! empty( $options['compensation_optional'] ) && 'yes' === $options['compensation_optional'] );

		if ( $packages ) {
			foreach ( $packages as $package ) {
				if ( $package->value > self::COMPENSATION_UP_TO_VALUE && ! $compensation_optional ) {
					return false; // Max. compensation is 20.
				}

				$quote = 0;

				if ( ! $this->get_rate_bands( $package->id ) ) {
					return false; // Unpacked item.
				}

				$this->debug( __( 'RoyalMail First Class:', 'woocommerce-shipping-royalmail' ) . ' <pre>' . print_r( $package, true ) . '</pre>' );


				$bands = $this->get_rate_bands( $package->id );

				$matched = false;

				foreach ( $bands as $band => $value ) {
					if ( is_numeric( $band ) && $package->weight <= $band ) {
						$quote += $value;
						$matched = true;
						break;
					}
				}

				if ( ! $matched ) {
					return null;
				}

				$class_quote += $quote;
			}
		}

		// Return pounds.
		$quotes                = array();
		$quotes['first-class'] = $class_quote / 100;

		return $quotes;
	}
}
