<?php
/**
 * Second class rate.
 *
 * @package WC_RoyalMail/Rate
 */

/**
 * RoyalMail_Rate_Second_Class class.
 *
 * Updated on 2020-03-23 as per https://www.royalmail.com/sites/royalmail.com/files/2020-07/royal-mail-our-prices-1-september-2020.pdf
 * See UK Standard page 6.
 */
class RoyalMail_Rate_Second_Class extends RoyalMail_Rate {

	/**
	 * ID/Name of rate.
	 *
	 * @var string
	 */
	protected $rate_id = 'second_class';

	const COMPENSATION_UP_TO_VALUE = 20;

	/**
	 * Pricing bands
	 *
	 * Key is size (e.g. 'letter') and value is an array where key is weight in
	 * gram and value is the price (in penny).
	 *
	 * @var array
	 */
	protected $bands = array(
		'2020' => array(
			'letter'              => array(
				100 => 65,
			),
			'large-letter'        => array(
				100 => 88,
				250 => 140,
				500 => 183,
				750 => 248,
			),
			'small-parcel-wide'   => array(
				1000 => 310,
				2000 => 310,
			),
			'small-parcel-deep'   => array(
				1000 => 310,
				2000 => 310,
			),
			'small-parcel-bigger' => array(
				1000 => 310,
				2000 => 310,
			),
			'medium-parcel'       => array(
				1000  => 520,
				2000  => 520,
				5000  => 899,
				10000 => 2025,
				20000 => 2855,
			),
		),
	);

	/**
	 * Shipping boxes.
	 *
	 * @var array $boxes
	 */
	protected $boxes = array(
		'letter' => array(
			'length'   => 240, // Max L in mm.
			'width'    => 165, // Max W in mm.
			'height'   => 5,   // Max H in mm.
			'weight'   => 100, // Max Weight in grams.
		),
		'large-letter' => array(
			'length'   => 353,
			'width'    => 250,
			'height'   => 25,
			'weight'   => 750,
		),
		'small-parcel-wide' => array(
			'length' => 450,
			'width'  => 350,
			'height' => 80,
			'weight' => 2000,
		),
		'small-parcel-deep' => array(
			'length' => 350,
			'width'  => 250,
			'height' => 160,
			'weight' => 2000,
		),
		'small-parcel-bigger' => array(
			'length' => 450,
			'width'  => 350,
			'height' => 160,
			'weight' => 2000,
		),
		'medium-parcel' => array(
			'length'   => 610,
			'width'    => 460,
			'height'   => 460,
			'weight'   => 20000,
		),
	);

	/**
	 * Cost for signed for delivery.
	 *
	 * @var array
	 */
	protected $signed_for_cost = array(
		'2019' => 120,
		'2020' => 130,
	);

	/**
	 * Cost for signed for delivery of a package.
	 *
	 * @var array
	 */
	protected $signed_for_package_cost = array(
		'2019' => 100,
	);

	/**
	 * Get quotes for this rate.
	 *
	 * @param array  $items to be shipped.
	 * @param string $packing_method selected.
	 * @param string $destination address.
	 *
	 * @return array
	 */
	public function get_quotes( $items, $packing_method, $destination, $boxes = array(), $instance_id = '' ) {
		$class_quote           = false;
		$recorded_quote        = false;
		$packages              = $this->get_packages( $items, $packing_method );
		$options               = $this->get_instance_options( $instance_id );
		$compensation_optional = ( ! empty( $options['compensation_optional'] ) && 'yes' === $options['compensation_optional'] );

		if ( $packages ) {
			foreach ( $packages as $package ) {
				if ( $package->value > self::COMPENSATION_UP_TO_VALUE && ! $compensation_optional ) {
					return false; // Max. compensation is 20.
				}

				$quote = 0;

				if ( ! $this->get_rate_bands( $package->id ) ) {
					return false; // Unpacked item.
				}

				$this->debug( __( 'RoyalMail Second Class:', 'woocommerce-shipping-royalmail' ) . ' <pre>' . print_r( $package, true ) . '</pre>' );


				$bands = $this->get_rate_bands( $package->id );

				$matched = false;

				foreach ( $bands as $band => $value ) {
					if ( is_numeric( $band ) && $package->weight <= $band ) {
						$quote += $value;
						$matched = true;
						break;
					}
				}

				if ( ! $matched ) {
					return;
				}

				$class_quote    += $quote;

				if ( 'letter' === $package->id || 'large-letter' === $package->id ) {
					$recorded_quote += $quote + $this->get_signed_cost( 'signed_for_cost' );
				} else {
					$recorded_quote += $quote + $this->get_signed_cost( 'signed_for_package_cost' );
				}
			}
		}

		// Return pounds.
		$quotes                        = array();
		$quotes['second-class']        = $class_quote / 100;

		return $quotes;
	}
}
