<?php
/**
 * First class rate.
 *
 * @package WC_RoyalMail/Rate
 */

/**
 * RoyalMail_Rate_First_Class_Signed class.
 *
 * Updated on 2020-09-04 as per https://www.royalmail.com/sites/royalmail.com/files/2020-07/royal-mail-our-prices-1-september-2020.pdf
 * See UK Signed page 5.
 */
class RoyalMail_Rate_First_Class_Signed extends RoyalMail_Rate_First_Class {

	/**
	 * ID/Name of rate.
	 *
	 * @var string
	 */
	protected $rate_id = 'first_class_signed';

	const COMPENSATION_UP_TO_VALUE = 50;

	/**
	 * Cost for signed for delivery.
	 *
	 * @var array
	 */
	protected $signed_for_cost = array(
		'2019' => 120,
		'2020' => 130,
	);

	/**
	 * Cost for signed for delivery of a package.
	 *
	 * @var array
	 */
	protected $signed_for_package_cost = array(
		'2019' => 100,
	);

	/**
	 * Get quotes for this rate.
	 *
	 * @param  array  $items to be shipped.
	 * @param  string $packing_method the method selected.
	 * @param  string $destination Address to ship to.
	 * @return array
	 */
	public function get_quotes( $items, $packing_method, $destination, $boxes = array(), $instance_id = '' ) {
		$class_quote           = 0;
		$packages              = $this->get_packages( $items, $packing_method );
		$options               = $this->get_instance_options( $instance_id );
		$compensation_optional = ( ! empty( $options['compensation_optional'] ) && 'yes' === $options['compensation_optional'] );

		if ( $packages ) {
			foreach ( $packages as $package ) {
				if ( $package->value > self::COMPENSATION_UP_TO_VALUE && ! $compensation_optional ) {
					return false; // Max. compensation is 50.
				}

				$quote = 0;

				if ( ! $this->get_rate_bands( $package->id ) ) {
					return false; // Unpacked item.
				}

				$this->debug( __( 'RoyalMail First Class Signed:', 'woocommerce-shipping-royalmail' ) . ' <pre>' . print_r( $package, true ) . '</pre>' );

				$bands = $this->get_rate_bands( $package->id );

				$matched = false;

				foreach ( $bands as $band => $value ) {
					if ( is_numeric( $band ) && $package->weight <= $band ) {
						$quote += $value;
						$matched = true;
						break;
					}
				}

				if ( ! $matched ) {
					return null;
				}

				if ( 'letter' === $package->id || 'large-letter' === $package->id ) {
					$class_quote += $quote + $this->get_signed_cost( 'signed_for_cost' );
				} else {
					$class_quote += $quote + $this->get_signed_cost( 'signed_for_package_cost' );
				}
			}
		}

		// Return pounds.
		$quotes                       = array();
		$quotes['first-class-signed'] = $class_quote / 100;

		return $quotes;
	}
}
