<?php
/**
 * International-Tracked-and-Signed rate.
 *
 * @package WC_RoyalMail/Rate
 */

/**
 * RoyalMail_Rate_International_Tracked_Signed class.
 *
 * Updated on 2020-09-04 as per https://www.royalmail.com/sites/royalmail.com/files/2020-07/royal-mail-our-prices-1-september-2020.pdf
 * See International Tracked & Signed page 11.
 */
class RoyalMail_Rate_International_Tracked_Signed extends RoyalMail_Rate {

	/**
	 * ID/Name of rate
	 *
	 * @var string
	 */
	protected $rate_id = 'international_tracked_signed';

	/**
	 * List of countries that support Tracked and Signed service.
	 *
	 * @see http://www.royalmail.com/sites/default/files/Royal-Mail-International-Tracking-Signature-Services-List-April2017.pdf
	 *
	 * @since 2.5.4
	 * @version 2.5.4
	 *
	 * @var array
	 */
	protected $supported_countries = array(
		'AX',
		'AD',
		'AR',
		'AT',
		'BB',
		'BY',
		'BE',
		'BZ',
		'BG',
		'KH',
		'CA',
		'KY',
		'CK',
		'HR',
		'CY',
		'CZ',
		'DK',
		'EC',
		'FO',
		'FI',
		'FR',
		'GE',
		'DE',
		'GI',
		'GR',
		'GL',
		'HK',
		'HU',
		'IS',
		'ID',
		'IE',
		'IT',
		'JP',
		'LV',
		'LB',
		'LI',
		'LT',
		'LU',
		'MY',
		'MT',
		'MD',
		'NL',
		'NZ',
		'PL',
		'PT',
		'RO',
		'RU',
		'SM',
		'RS',
		'SG',
		'SK',
		'SI',
		'KR',
		'ES',
		'SE',
		'CH',
		'TH',
		'TO',
		'TT',
		'TR',
		'UG',
		'AE',
		'US',
		'VA',
	);

	/**
	 * Pricing bands - Europe, Zone 1, Zone 2, Zone 3 (previously Zone 1)
	 *
	 * @var array
	 */
	protected $bands = array(
		'2020' => array(
			'letter'       => array(
				10  => array( 645, 645, 645, 645 ),
				20  => array( 645, 685, 685, 685 ),
				100 => array( 690, 775, 780, 775 ),
			),
			'large-letter' => array(
				100 => array( 850, 925, 980, 935 ),
				250 => array( 895, 1035, 1140, 1055 ),
				500 => array( 980, 1215, 1390, 1245 ),
				750 => array( 1020, 1395, 1665, 1440 ),
			),
			'packet'       => array(
				100  => array( 985, 1110, 1200, 1288 ),
				250  => array( 990, 1200, 1325, 1434 ),
				500  => array( 1155, 1575, 1750, 1868 ),
				750  => array( 1260, 1815, 2045, 2123 ),
				1000 => array( 1355, 2080, 2370, 2468 ),
				1250 => array( 1400, 2290, 2640, 2783 ),
				1500 => array( 1475, 2430, 2890, 3048 ),
				1750 => array( 1510, 2510, 3050, 3165 ),
				2000 => array( 1535, 2600, 3215, 3321 ),
			),
		),
	);

	/**
	 * Shipping boxes
	 *
	 * @var array
	 */
	protected $default_boxes = array(
		'letter' => array(
			'length'   => 240, // Max L in mm.
			'width'    => 165, // Max W in mm.
			'height'   => 5,   // Max H in mm.
			'weight'   => 100,  // Max Weight in grams.
		),
		'large-letter' => array(
			'length'   => 353,
			'width'    => 250,
			'height'   => 25,
			'weight'   => 750,
		),
		'long-parcel' => array(
			'length'   => 600,
			'width'    => 150,
			'height'   => 150,
			'weight'   => 2000,
		),
		'square-parcel' => array(
			'length'   => 300,
			'width'    => 300,
			'height'   => 300,
			'weight'   => 2000,
		),
		'parcel' => array(
			'length'   => 450,
			'width'    => 225,
			'height'   => 225,
			'weight'   => 2000,
		),
	);

	/**
	 * Fixed compensation
	 *
	 * @var string
	 */
	private $compensation    = '250';

	/**
	 * Get quotes for this rate.
	 *
	 * @since 2.5.4
	 * @version 2.5.4
	 *
	 * @param  array  $items to be shipped.
	 * @param  string $packing_method the method selected.
	 * @param  string $destination Address to ship to.
	 * @param  array  $boxes User-defined boxes.
	 * @return array
	 */
	public function get_quotes( $items, $packing_method, $destination, $boxes = array() ) {
		if ( ! in_array( $destination, $this->supported_countries ) ) {
			return array();
		}

		$class_quote = false;

		if ( ! empty( $boxes ) ) {
			$this->boxes = array();

			foreach ( $boxes as $key => $box ) {
				$this->boxes[ $key ] = array(
					'length'     => $box['inner_length'],
					'width'      => $box['inner_width'],
					'height'     => $box['inner_height'],
					'box_weight' => $box['box_weight'],
					'weight'     => 2000,
				);
			}
		} else {
			$this->boxes = $this->default_boxes;
		}

		$zone     = $this->get_zone( $destination );
		$packages = $this->get_packages( $items, $packing_method );

		if ( $packages ) {
			foreach ( $packages as $package ) {

				if ( 'letter' !== $package->id && 'large-letter' !== $package->id ) {
					$package->id = 'packet';
				}

				if ( 'packet' === $package->id && 900 < ( $package->length + $package->width + $package->height ) ) {
					return false; // Exceeding parcels requirement, unpacked.
				}

				if ( ! $this->get_rate_bands( $package->id ) ) {
					return false; // Unpacked item.
				}

				$this->debug( __( 'International tracked and signed package:', 'woocommerce-shipping-royalmail' ) . ' <pre>' . print_r( $package, true ) . '</pre>' );

				$bands   = $this->get_rate_bands( $package->id );
				$quote   = 0;
				$matched = false;

				foreach ( $bands as $band => $value ) {
					if ( $package->weight <= $band ) {
						switch ( $zone ) {
							case 'EU':
							case 'EUR':
								$quote += $value[0];
								break;
							case '1':
								$quote += $value[1];
								break;
							case '2':
								$quote += $value[2];
								break;
							case '3':
								// Fallback to zone 1 for older prices.
								$quote += isset( $value[3] ) ? $value[3] : $value[1];
								break;
						}
						$matched = true;
						break;
					}
				}

				if ( ! $matched ) {
					return;
				}

				$class_quote  += $quote;

				if ( $package->value > 50 ) {
					$class_quote += $this->compensation;
				}
			}
		}

		// Return pounds.
		$quotes = array();
		$quotes['international-tracked-signed'] = $class_quote / 100;

		return $quotes;
	}
}
