<?php
/**
 * International Standard rate.
 *
 * @package WC_RoyalMail/Rate
 */

/**
 * RoyalMail_Rate_International_Standard class.
 *
 * Updated on 2020-09-04 as per https://www.royalmail.com/sites/royalmail.com/files/2020-07/royal-mail-our-prices-1-september-2020.pdf
 * See International Standard page 10.
 */
class RoyalMail_Rate_International_Standard extends RoyalMail_Rate {

	/**
	 * ID/Name of rate
	 *
	 * @var string
	 */
	protected $rate_id = 'international_standard';

	/**
	 * Pricing bands - Europe, Zone 1, Zone 2, Zone 3 (previously Zone 1)
	 *
	 * @var array
	 */
	protected $bands = array(
		'2020' => array(
			'letter'       => array(
				10  => array( 145, 145, 145, 145 ),
				20  => array( 145, 170, 170, 170 ),
				100 => array( 170, 250, 255, 250 ),
			),
			'large-letter' => array(
				100 => array( 300, 375, 425, 380 ),
				250 => array( 425, 560, 665, 575 ),
				500 => array( 510, 785, 965, 815 ),
				750 => array( 610, 1045, 1330, 1090 ),
			),
			'packet'       => array(
				100  => array( 515, 625, 730, 838 ),
				250  => array( 535, 735, 875, 984 ),
				500  => array( 725, 1125, 1320, 1518 ),
				750  => array( 850, 1400, 1645, 1773 ),
				1000 => array( 965, 1680, 1980, 2118 ),
				1250 => array( 1055, 1925, 2300, 2433 ),
				1500 => array( 1160, 2150, 2610, 2698 ),
				1750 => array( 1240, 2250, 2780, 2815 ),
				2000 => array( 1285, 2395, 3000, 2971 ),
			),
		),
	);

	/**
	 * Shipping boxes
	 *
	 * @var array
	 */
	protected $default_boxes = array(
		'letter' => array(
			'length'   => 240, // Max L in mm.
			'width'    => 165, // Max W in mm.
			'height'   => 5,   // Max H in mm.
			'weight'   => 100,  // Max Weight in grams.
		),
		'large-letter' => array(
			'length'   => 353,
			'width'    => 250,
			'height'   => 25,
			'weight'   => 750,
		),
		'long-parcel' => array(
			'length'   => 600,
			'width'    => 150,
			'height'   => 150,
			'weight'   => 2000,
		),
		'square-parcel' => array(
			'length'   => 300,
			'width'    => 300,
			'height'   => 300,
			'weight'   => 2000,
		),
		'parcel' => array(
			'length'   => 450,
			'width'    => 225,
			'height'   => 225,
			'weight'   => 2000,
		),
	);

	/**
	 * Get quotes for this rate
	 *
	 * @param  array  $items to be shipped.
	 * @param  string $packing_method the method selected.
	 * @param  string $destination Address to ship to.
	 * @param  array  $boxes User-defined boxes.
	 * @param int $instance_id.
	 * @return array
	 */
	public function get_quotes( $items, $packing_method, $destination, $boxes = array(), $instance_id = '' ) {
		$standard_quote = false;

		if ( ! empty( $boxes ) ) {
			$this->boxes = array();

			foreach ( $boxes as $key => $box ) {
				$this->boxes[ $key ] = array(
					'length'     => $box['inner_length'],
					'width'      => $box['inner_width'],
					'height'     => $box['inner_height'],
					'box_weight' => $box['box_weight'],
					'weight'     => 2000,
				);
			}
		} else {
			$this->boxes = $this->default_boxes;
		}

		$zone                  = $this->get_zone( $destination );
		$packages              = $this->get_packages( $items, $packing_method );
		$options               = $this->get_instance_options( $instance_id );
		$compensation_optional = ( ! empty( $options['compensation_optional'] ) && 'yes' === $options['compensation_optional'] );

		if ( $packages ) {
			foreach ( $packages as $package ) {
				if ( $package->value > 20 && ! $compensation_optional ) {
					return false; // Max. compensation is 20.
				}

				if ( 'letter' !== $package->id && 'large-letter' !== $package->id ) {
					$package->id = 'packet';
				}

				if ( 'packet' === $package->id && 900 < ( $package->length + $package->width + $package->height ) ) {
					return false; // Exceeding parcels requirement, unpacked.
				}

				if ( ! $this->get_rate_bands( $package->id ) ) {
					return false; // Unpacked item.
				}

				$this->debug( __( 'International package:', 'woocommerce-shipping-royalmail' ) . ' <pre>' . print_r( $package, true ) . '</pre>' );

				$bands   = $this->get_rate_bands( $package->id );
				$quote   = 0;
				$matched = false;

				foreach ( $bands as $band => $value ) {
					if ( $package->weight <= $band ) {
						switch ( $zone ) {
							case 'EU':
							case 'EUR':
								$quote += $value[0];
								break;
							case '1':
								$quote += $value[1];
								break;
							case '2':
								$quote += $value[2];
								break;
							case '3':
								// Fallback to zone 1 for older prices.
								$quote += isset( $value[3] ) ? $value[3] : $value[1];
								break;
						}
						$matched = true;
						break;
					}
				}

				if ( ! $matched ) {
					return;
				}

				$standard_quote += $quote;
			}
		}

		// Return pounds.
		$quotes = array();
		$quotes['international-standard'] = $standard_quote / 100;

		return $quotes;
	}
}
