<?php
/**
 * International Signed rate.
 *
 * @package WC_RoyalMail/Rate
 */

/**
 * RoyalMail_Rate_International_Signed class.
 *
 * Updated on 2020-03-23 as per https://www.royalmail.com/sites/royalmail.com/files/2020-02/royal-mail-our-prices-valid-from-23-march-2020.pdf
 * See Parcelforce WorldWide page 14.
 */
class RoyalMail_Rate_International_Signed extends RoyalMail_Rate {

	/**
	 * ID/Name of rate
	 *
	 * @var string
	 */
	protected $rate_id = 'international_signed';

	/**
	 * List of countries that support Signed service.
	 *
	 * @see http://www.royalmail.com/sites/default/files/Royal-Mail-International-Tracking-Signature-Services-List-April2017.pdf
	 *
	 * @since 2.5.4
	 * @version 2.5.4
	 *
	 * @var array
	 */
	protected $supported_countries = array(
		'AF',
		'AL',
		'DZ',
		'AO',
		'AI',
		'AG',
		'AM',
		'AW',
		'AU',
		'AZ',
		'BS',
		'BH',
		'BD',
		'BJ',
		'BM',
		'BT',
		'BO',
		'BQ',
		'BA',
		'BW',
		'BR',
		'IO',
		'VG',
		'BN',
		'BF',
		'BI',
		'CM',
		'CV',
		'CF',
		'TD',
		'CL',
		'CN',
		'CX',
		'CO',
		'KM',
		'CG',
		'CD',
		'CR',
		'CU',
		'CW',
		'DJ',
		'DM',
		'DO',
		'EG',
		'SV',
		'GQ',
		'ER',
		'EE',
		'ET',
		'FK',
		'FJ',
		'GF',
		'PF',
		'TF',
		'GA',
		'GM',
		'GH',
		'GD',
		'GP',
		'GT',
		'GN',
		'GW',
		'GY',
		'HT',
		'HN',
		'IN',
		'IR',
		'IQ',
		'IL',
		'CI',
		'JM',
		'JO',
		'KZ',
		'KE',
		'KI',
		'KW',
		'KG',
		'LA',
		'LS',
		'LR',
		'LY',
		'MO',
		'MK',
		'MG',
		'YT',
		'MW',
		'MV',
		'ML',
		'MQ',
		'MR',
		'MU',
		'MX',
		'MN',
		'ME',
		'MS',
		'MA',
		'MZ',
		'MM',
		'NA',
		'NR',
		'NP',
		'NC',
		'NI',
		'NE',
		'NG',
		'NU',
		'KP',
		'NO',
		'OM',
		'PK',
		'PW',
		'PA',
		'PG',
		'PY',
		'PE',
		'PH',
		'PN',
		'PR',
		'QA',
		'RE',
		'RW',
		'ST',
		'SA',
		'SN',
		'SC',
		'SL',
		'SB',
		'ZA',
		'SS',
		'LK',
		'BQ',
		'SH',
		'KN',
		'LC',
		'MF',
		'SX',
		'VC',
		'SD',
		'SR',
		'SZ',
		'SY',
		'TW',
		'TJ',
		'TZ',
		'TL',
		'TG',
		'TK',
		'TN',
		'TM',
		'TC',
		'TV',
		'UA',
		'UY',
		'UZ',
		'VU',
		'VE',
		'VN',
		'WF',
		'EH',
		'WS',
		'YE',
		'ZM',
		'ZW',
	);

	/**
	 * Pricing bands - EU, ZONE 1, Zone 2.
	 *
	 * @var array
	 */
	protected $bands = array(
		'2020' => array(
			'letter'       => array(
				10  => array( 645, 645, 645 ),
				20  => array( 645, 685, 685 ),
				100 => array( 690, 775, 780 ),
			),
			'large-letter' => array(
				100 => array( 850, 925, 980 ),
				250 => array( 895, 1035, 1140 ),
				500 => array( 980, 1215, 1390 ),
				750 => array( 1020, 1395, 1665 ),
			),
			'packet'       => array(
				100  => array( 985, 1110, 1200 ),
				250  => array( 990, 1200, 1325 ),
				500  => array( 1155, 1575, 1750 ),
				750  => array( 1260, 1815, 2045 ),
				1000 => array( 1355, 2080, 2370 ),
				1250 => array( 1400, 2290, 2640 ),
				1500 => array( 1475, 2430, 2890 ),
				1750 => array( 1510, 2510, 3050 ),
				2000 => array( 1535, 2600, 3215 ),
			),
		),
	);

	/**
	 * Shipping boxes
	 *
	 * @var array
	 */
	protected $default_boxes = array(
		'letter' => array(
			'length'   => 240, // Max L in mm.
			'width'    => 165, // Max W in mm.
			'height'   => 5,   // Max H in mm.
			'weight'   => 100,  // Max Weight in grams.
		),
		'large-letter' => array(
			'length'   => 353,
			'width'    => 250,
			'height'   => 25,
			'weight'   => 750,
		),
		'long-parcel' => array(
			'length'   => 600,
			'width'    => 150,
			'height'   => 150,
			'weight'   => 2000,
		),
		'square-parcel' => array(
			'length'   => 300,
			'width'    => 300,
			'height'   => 300,
			'weight'   => 2000,
		),
		'parcel' => array(
			'length'   => 450,
			'width'    => 225,
			'height'   => 225,
			'weight'   => 2000,
		),
	);

	/**
	 * Get quotes for this rate
	 *
	 * @param  array  $items to be shipped.
	 * @param  string $packing_method the method selected.
	 * @param  string $destination Address to ship to.
	 * @param  array  $boxes User-defined boxes.
	 *
	 * @return array
	 */
	public function get_quotes( $items, $packing_method, $destination, $boxes = array() ) {
		if ( ! in_array( $destination, $this->supported_countries ) ) {
			return array();
		}

		if ( ! empty( $boxes ) ) {
			$this->boxes = array();

			foreach ( $boxes as $key => $box ) {
				$this->boxes[ $key ] = array(
					'length'     => $box['inner_length'],
					'width'      => $box['inner_width'],
					'height'     => $box['inner_height'],
					'box_weight' => $box['box_weight'],
					'weight'     => 2000,
				);
			}
		} else {
			$this->boxes = $this->default_boxes;
		}

		$packages    = $this->get_packages( $items, $packing_method );
		$class_quote = false;

		if ( $packages ) {
			foreach ( $packages as $package ) {
				$quote = $this->get_quote( $package, $destination );

				// Do not return a quote if one of the packages exceeds a limitation.
				if ( false === $quote ) {
					return array();
				}

				$class_quote += $quote;
			}
		}

		// Return pounds.
		$quotes = array();
		$quotes['international-signed'] = $class_quote / 100;

		return $quotes;
	}

	/**
	 * Get quote.
	 *
	 * @since 2.5.1
	 * @version 2.5.1
	 *
	 * @param array  $package     Package to ship.
	 * @param string $destination Destination.
	 *
	 * @return bool|int|void
	 */
	public function get_quote( $package, $destination ) {

		$zone = $this->get_zone( $destination );

		if ( 'letter' !== $package->id && 'large-letter' !== $package->id ) {
			$package->id = 'packet';
		}

		if ( 'packet' === $package->id && 900 < ( $package->length + $package->width + $package->height ) ) {
			return false; // Exceeding parcels requirement, unpacked.
		}

		if ( ! $this->get_rate_bands( $package->id ) ) {
			return false; // Unpacked item.
		}

		$this->debug( __( 'International signed package:', 'woocommerce-shipping-royalmail' ) . ' <pre>' . print_r( $package, true ) . '</pre>' );

		$bands   = $this->get_rate_bands( $package->id );
		$quote   = 0;
		$matched = false;

		foreach ( $bands as $band => $value ) {
			if ( $package->weight <= $band ) {
				switch ( $zone ) {
					case 'EU' :
					case 'EUR' :
						$quote += $value[0];
						break;
					case '1' :
						$quote += $value[1];
						break;
					case '2' :
						$quote += $value[2];
						break;
				}
				$matched = true;
				break;
			}
		}

		if ( ! $matched ) {
			return;
		}
		return $quote;
	}
}
