<?php
/**
 * Woo_Ecards_Pro Class file.
 * Plugin Name: WooCommerce eCards Pro
 * Plugin URI: http://www.flippercode.com/
 * Description: This woocommerce plugin allows you to create and sale ecards on your site as regular woocommerce products. Allows site visitors to purchase ecards from your store and let them send it to their defined email-addresses on any schedule date.
 * Author: flippercode
 * Author URI: http://www.flippercode.com/
 * Version: 2.0.1
 * Text Domain: woocommerce-ecards-pro
 * Domain Path: /lang/
 */

if ( ! class_exists( 'FC_Plugin_Base' ) ) {
	$pluginClass = plugin_dir_path( __FILE__ ) . '/core/class.plugin.php';
	if ( file_exists( $pluginClass ) ) {
		include( $pluginClass );
	}
}

if ( ! class_exists( 'Woo_Ecards_Pro' ) && class_exists( 'FC_Plugin_Base' ) ) {

	/**
	 * Main plugin class
	 *
	 * @author Flipper Code <hello@flippercode.com>
	 * @package Woocommerce Ecards Pro
	 */
	class Woo_Ecards_Pro extends FC_Plugin_Base {

		/**
		 * List of Class Vars & Modules.
		 *
		 * @var array
		 */
		private $isEcard;
		private $extraCartFields;
		protected $dboptions;

		/**
		 * Intialize variables, files and call actions.
		 *
		 * @var array
		 */
		public function __construct() {

			$wooInstalled = in_array( 'woocommerce/woocommerce.php', get_option( 'active_plugins' ) );
			if ( ! function_exists( 'is_plugin_active_for_network' ) ) {
				require_once( ABSPATH . '/wp-admin/includes/plugin.php' );
			}
			$networkActive = ( is_plugin_active_for_network( 'woocommerce/woocommerce.php' ) ) ? true : false;
			$is_woocommerce_not_installed = ( ! $wooInstalled && ! $networkActive ) ? true : false;

			if ( $is_woocommerce_not_installed ) {
				add_action( 'admin_notices', array( $this, 'wcep_woocommerce_missing' ) );
			} else {

				parent::__construct( $this->_plugin_definition() );
				$this->dboptions = maybe_unserialize( get_option( 'wp-ecards-pro' ) );
				$this->extraCartFields = apply_filters( 'ecp_extra_cart_fields', array( 'recipient_name', 'recipient_email_address', 'ecard_delivery_date', 'default_subject', 'ecard_heading', 'ecard_message', 'sender_name', 'sender_email_address', 'copy_to_self' ) );
				$this->register_hooks();

			}

		}

		function _plugin_definition() {

			$this->pluginPrefix = 'ecp';
			$pluginClasses = array( 'wecp-template.php', 'ecp-controller.php', 'ecp-model.php', 'mandrill.php', 'ecp-fresh-settings.php' );
			$pluginModules = array( 'overview', 'sendecards', 'settings', 'tools' );
			$pluginCssFilesFrontEnd = array( 'frontend.css' );
			$pluginCssFilesBackendEnd = array( 'backend.css' );
			$pluginJsFilesFrontEnd = array( 'ecard-frontend.js' );
			$pluginJsFilesBackEnd = array( 'select2.js', 'backend.js' );
			$pluginData = array(
				'childFileRefrence' => __FILE__,
				'childClassRefrence' => __CLASS__,
				'pluginPrefix' => 'ecp',
				'pluginDirectory' => plugin_dir_path( __FILE__ ),
				'pluginTextDomain' => 'woocommerce-ecards-pro',
				'pluginURL' => plugin_dir_url( __FILE__ ),
				'dboptions' => 'wp-ecards-pro',
				'controller' => 'ECP_Controller',
				'model' => 'ECP_Model',
				'pluginLabel' => esc_html__( 'WooCommerce eCards Pro', 'woocommerce-ecards-pro' ),
				'pluginClasses' => $pluginClasses,
				'pluginmodules' => $pluginModules,
				'pluginmodulesprefix' => 'ECP_Model_',
				'pluginCssFilesFrontEnd' => $pluginCssFilesFrontEnd,
				'pluginCssFilesBackEnd' => $pluginCssFilesBackendEnd,
				'pluginJsFilesFrontEnd' => $pluginJsFilesFrontEnd,
				'pluginJsFilesBackEnd' => $pluginJsFilesBackEnd,
				'isWoocommerceExtention' => true,
				'loadCustomizer' => false,
				'pluginDirectoryBaseName' => basename( dirname( __FILE__ ) ),
			);

			return $pluginData;
		}

		public function register_hooks() {

			add_action( 'plugins_loaded', array( $this, 'load_plugin_languages' ) );
			add_action( 'deliver_woo_ecards', array( $this, 'deliver_woo_ecards' ) );
			add_action( 'woocommerce_before_add_to_cart_button', array( $this, 'add_exta_cart_fields' ) );
			add_action( 'woocommerce_add_to_cart_validation', array( $this, 'apply_validations_to_extrafields' ), 10, 3 );
			add_action( 'woocommerce_add_cart_item_data', array( $this, 'save_extra_card_order_info' ), 10, 2 );
			add_filter( 'woocommerce_get_item_data', array( $this, 'render_meta_on_cart_and_checkout' ), 10, 2 );
			add_action( 'woocommerce_add_order_item_meta', array( $this, 'ecard_order_meta_handler' ), 1, 3 );
			add_action( 'wp_enqueue_scripts', array( $this, 'load_frontend_resources' ), 1, 3 );
			add_action( 'wp_footer', array( $this, 'hook_in_foot' ), 1, 3 );
			add_action( 'woocommerce_thankyou', array( $this, 'on_ecard_order_placing' ) );
			add_action( 'wp_head', array( $this, 'check_for_ecard' ), 10, 2 );
			add_filter( 'woocommerce_product_single_add_to_cart_text', array( $this, 'woo_custom_cart_button_text' ), 10, 2 );
			add_filter( 'woocommerce_product_add_to_cart_text', array( $this, 'woo_archive_custom_cart_button_text' ), 10, 2 );
			add_filter( 'woocommerce_attribute_label', array( $this, 'woocommerce_attribute_label' ), 10, 3 );
			add_filter( 'cron_schedules', array( $this, 'my_cron_schedules' ) );
			add_action( 'woocommerce_after_order_itemmeta', array( $this, 'ecard_order_extra_meta' ), 10, 3 );
			add_action( 'wp_ajax_get_current_ecard_info', array( $this, 'get_current_ecard_info' ) );
			add_action( 'wp_ajax_nopriv_get_current_ecard_info', array( $this, 'get_current_ecard_info' ) );
			add_action( 'wp_ajax_send_ecards_to_users', array( $this, 'send_ecards_to_users' ) );
			add_action( 'wp_ajax_nopriv_send_ecards_to_users', array( $this, 'send_ecards_to_users' ) );
			add_action( 'wp_ajax_wep_update_ecard_meta', array( $this, 'wep_update_ecard_meta' ) );
			add_action( 'wp_ajax_nopriv_wep_update_ecard_meta', array( $this, 'wep_update_ecard_meta' ) );

			if ( is_admin() ) {

				add_action( 'load-post.php', array( $this, 'register_ecards_metabox' ) );
				add_action( 'load-post-new.php', array( $this, 'register_ecards_metabox' ) );
				add_action( 'save_post', array( $this, 'save_ecards_metabox' ), 10, 2 );
				add_action( 'admin_head', array( $this, 'hook_in_admin_head' ) );
				add_action( 'admin_footer', array( $this, 'hook_in_admin_footer' ) );
				if ( isset( $this->dboptions['display_ecard_order_listing'] ) && ! empty( $this->dboptions['display_ecard_order_listing'] ) ) {
					add_filter( 'manage_shop_order_posts_columns', array( $this, 'manage_shop_order_columns' ), 20 );
					add_action( 'manage_shop_order_posts_custom_column', array( $this, 'custom_woocommerce_order_columns' ), 20, 2 );
				}
				if ( isset( $this->dboptions['display_column_for_ecard_product'] ) && ! empty( $this->dboptions['display_column_for_ecard_product'] ) ) {
					add_filter( 'manage_product_posts_columns', array( $this, 'manage_product_columns' ), 20 );
					add_action( 'manage_product_posts_custom_column', array( $this, 'custom_woocommerce_product_columns' ), 20, 2 );
				}
			}

		}

		function manage_product_columns( $existing_columns ) {

			$existing_columns['ecard_product_info']    = esc_html__( 'eCard Product', 'woocommerce-ecards-pro' );
			$existing_columns = apply_filters( 'wecp-woo-products-columns', $existing_columns );
			return $existing_columns;

		}
		
		function custom_woocommerce_product_columns( $column, $post_id ) {

			global $post;

			switch ( $column ) {
				case 'ecard_product_info':
					$hasEcard = get_post_meta( $post_id, 'convert_into_ecard', true );
					if ( empty( $hasEcard ) ) {
						echo '<div class="wecp-notify">' . esc_html__( 'No', 'woocommerce-ecards-pro' ) . '</div>';
					} else {
						echo '<div class="wecp-success">' . esc_html__( 'Yes', 'woocommerce-ecards-pro' ) . '</div>';
					}
					break;

				default:
					break;
			}
		}
		
		function custom_woocommerce_order_columns( $column, $post_id ) {

			global $post;

			switch ( $column ) {
				case 'ecard_info':
					$hasEcard = get_post_meta( $post_id, 'convert_into_ecard', true );
					echo esc_html__( 'Order Has eCard : ', 'woocommerce-ecards-pro' );
					if ( empty( $hasEcard ) ) {
						echo '<div class="wecp-notify">' . esc_html__( 'No', 'woocommerce-ecards-pro' ) . '</div>';
					} else {
						echo '<div class="wecp-success">' . esc_html__( 'Yes', 'woocommerce-ecards-pro' ) . '</div>';
					}
					if ( $hasEcard ) {
						$allEcardSend = get_post_meta( $post_id, 'ecard_mail_send', true );
						echo esc_html__( 'All Delivered : ', 'woocommerce-ecards-pro' );
						if ( empty( $allEcardSend ) ) {
							echo '<div class="wecp-notify">' . esc_html__( 'No', 'woocommerce-ecards-pro' ) . '</div>';
						} else {
							echo '<div class="wecp-success">' . esc_html__( 'Yes', 'woocommerce-ecards-pro' ) . '</div>';
						}
					}
					break;

				default:
					break;
			}
		}


		function manage_shop_order_columns( $existing_columns ) {

			$existing_columns['ecard_info']    = esc_html__( 'eCard Information', 'woocommerce-ecards-pro' );
			$existing_columns = apply_filters( 'wecp-woo-columns', $existing_columns );
			return $existing_columns;

		}


		function ecard_order_extra_meta( $item_id, $item, $_product ) {

			global $post;
			$allecards = get_post_meta( $post->ID, 'all_ecards_in_order', true );
			$receiver = wc_get_order_item_meta( $item_id, 'recipient_email_address' );
			if ( $allecards && count( $allecards ) > 0 ) {
				$p_id = isset( $_product ) ? $_product->get_id() : '';
				foreach ( $allecards as $key => $ecards_in_order ) {
					if ( $ecards_in_order['ecard_id'] == $p_id && $receiver == $ecards_in_order['recipient_email_address'] ) {

						$deliverytime = $allecards[ $key ]['ecard_deliverery_time'];
						$send = ( $allecards[ $key ]['ecard_send'] ) ? true : false;
						if ( $send ) {
							$deliveryTime = date( 'Y-m-d H:i:s', $deliverytime );
							echo '<p class="ecard-extra-info">' . esc_html__( 'eCard Delivery Status', 'woocommerce-ecards-pro' ) . ' : <b>Sent</b></p>';
							echo '<p class="ecard-extra-info">' . esc_html__( 'eCard Delivery Date & Time', 'woocommerce-ecards-pro' ) . ': ';
							echo '<b>' . $deliveryTime . '</b></p>';

						} else {
							echo '<p class="ecard-extra-info">' . esc_html__( 'eCard Delivery Status', 'woocommerce-ecards-pro' ) . ' : <span>' . esc_html__( 'Not Sent Yet', 'woocommerce-ecards-pro' ) . '</span></p>';
						}
						break;
					}
				}
			}

		}

		function get_current_ecard_info() {

			// #1 Nonce security verification
			// #2 Authorization of request verification
			$nonce = sanitize_text_field( $_POST['nonce'] );
			if ( wp_verify_nonce( $nonce, 'ecp-call-nonce' ) && current_user_can( 'administrator' ) ) {

				$id = sanitize_text_field( wp_unslash( $_POST['ecard_id'] ) );
				$ecardImage = wp_get_attachment_image_src( get_post_thumbnail_id( $id ), 'full' );
				$defaultmsg = get_post_meta( $id, 'ecard_default_email' );
				$response = array(
					'img' => esc_url( $ecardImage[0] ),
					'title' => 'Dear {username}',
					'msg' => $defaultmsg,
				);

			} else {
				$response = array( 'status' => '0' );
			}
			echo json_encode( $response );
			exit;

		}

		function define_admin_menu() {

			$pagehook = add_menu_page(
				esc_html__( 'Woocommerce eCards Pro', 'woocommerce-ecards-pro' ),
				esc_html__( 'Woocommerce eCards Pro', 'woocommerce-ecards-pro' ),
				'ecp_admin_overview',
				ECP_SLUG,
				array( $this, 'processor' ),
				esc_url( ECP_IMAGES . 'fc-small-logo.png' )
			);

			return $pagehook;
		}

		function frontend_script_localisation() {

			$ecp_settings = array();
			if ( is_product() ) {
				$ecp_settings['is_product'] = true;
			}
			$ecp_settings['nonce'] = wp_create_nonce( 'ajax-nonce' );
			wp_localize_script( 'ecard-frontend.js', 'ecp_settings', $ecp_settings );
		}

		function backend_script_localisation() {

			$ecp_js_lang = array();
			$ecp_js_lang['ajax_url'] = esc_url( admin_url( 'admin-ajax.php' ) );
			$ecp_js_lang['nonce'] = wp_create_nonce( 'ecp-call-nonce' );
			wp_localize_script( 'backend.js', 'settings_obj', $ecp_js_lang );
			wp_localize_script( 'flippercode-ui.js', 'settings_obj', $ecp_js_lang );

		}

		function my_cron_schedules( $schedules ) {

			if ( ! isset( $schedules['deliver_woo_ecards'] ) ) {
				$schedules['deliver_woo_ecards'] = array(
					'interval' => 1 * 60,
					'display' => esc_html__( 'After Every One Minute', 'woocommerce-ecards-pro' ),
				);
			}
			   return $schedules;
		}

		function wcep_woocommerce_missing() {
			?>
				<div class="notice notice-error">
				<p><?php esc_html_e( 'WooCommerce is required for woocommerce ecards pro plugin. Please install and configure woocommerce first.', 'woocommerce-ecards-pro' ); ?></p>
				</div>
			<?php
		}

		/**
		 * Load plugin language file.
		 */
		function load_plugin_languages() {

			load_plugin_textdomain( 'woocommerce-ecards-pro', false, basename( dirname( __FILE__ ) ) . '/lang/' );
		}

		function woocommerce_attribute_label( $label, $name, $product ) {

			if ( $name == 'recipient_email_address' ) {
				$label = isset( $this->dboptions['recipient_email_address'] ) ? $this->dboptions['recipient_email_address'] : '';
			} else if ( $name == 'recipient_name' ) {
				$label = isset( $this->dboptions['recipient_name'] ) ? $this->dboptions['recipient_name'] : '';
			} else if ( $name == 'ecard_delivery_date' ) {
				$label = isset( $this->dboptions['ecard_delivery_date'] ) ? $this->dboptions['ecard_delivery_date'] : '';
			} else if ( $name == 'default_subject' ) {
				$label = isset( $this->dboptions['default_subject'] ) ? $this->dboptions['default_subject'] : '';
			} else if ( $name == 'ecard_heading' ) {
				$label = isset( $this->dboptions['ecard_heading'] ) ? $this->dboptions['ecard_heading'] : '';
			} else if ( $name == 'ecard_message' ) {
				$label = isset( $this->dboptions['ecard_message'] ) ? $this->dboptions['ecard_message'] : '';
			} else if ( $name == 'sender_name' ) {
				$label = isset( $this->dboptions['sender_name'] ) ? $this->dboptions['sender_name'] : '';
			} else if ( $name == 'sender_email_address' ) {
				$label = isset( $this->dboptions['sender_email_address'] ) ? $this->dboptions['sender_email_address'] : '';
			} else if ( $name == 'copy_to_self' ) {
				$label = esc_html__( 'Copy To Self', 'woocommerce-ecards-pro' );
			}
			return $label;
		}



		function setup_order_meta_for_ecards( $order_id ) {

			$ecardsinfo = get_post_meta( $order_id, 'all_ecards_in_order', true );
			if ( ! isset( $ecardsinfo ) || $ecardsinfo == '' || ! is_array( $ecardsinfo ) ) {

				$order = new WC_Order( $order_id );
				$totalEcards = array();
				foreach ( $order->get_items() as $key => $item ) {
					$recepient_email_address = wc_get_order_item_meta( $key, 'recipient_email_address' );
					if ( ! empty( $recepient_email_address ) ) {
						$totalEcards[] = array(
							'ecard_id' => $item['product_id'],
							'ecard_send' => false,
							'ecard_deliverery_time' => '',
							'recipient_email_address' => $recepient_email_address,
						);
					}
				}
				if ( count( $totalEcards ) > 0 ) {
					update_post_meta( $order_id, 'all_ecards_in_order', $totalEcards );
				}
			}

		}

		function is_order_has_only_ecards( $order_id ) {

			$order = new WC_Order( $order_id );
			$ecard_count = 0;
			$order_has_all_ecard = false;
			foreach ( $order->get_items() as $key => $item ) {
				$recepient_email_address = wc_get_order_item_meta( $key, 'recipient_email_address' );
				if ( ! empty( $recepient_email_address ) ) {
					$ecard_count++; }
			}
			if ( count( $order->get_items() ) == $ecard_count ) {
				$order_has_all_ecard = true;
			}
			return $order_has_all_ecard;

		}

		function send_ecard_instantly( $order_id ) {

			$today = date( 'Y-m-d' );
			$order = new WC_Order( $order_id );
			$ecards_for_now = array();
			$allmeta = get_post_meta( $order_id );
			foreach ( $order->get_items() as $key => $lineItem ) {

				$recepient_email_address = wc_get_order_item_meta( $key, 'recipient_email_address' );
				$delivery_date = wc_get_order_item_meta( $key, 'ecard_delivery_date' );
				$productID = $lineItem['product_id'];
				$dateNotSpecified = ( empty( $delivery_date ) ) ? true : false;
				$specifiedFortoday = ( ! empty( $delivery_date ) && $delivery_date == $today ) ? true : false;

				if ( $dateNotSpecified || $specifiedFortoday ) {

					$ecardImage = wp_get_attachment_image_src( get_post_thumbnail_id( $productID ), 'full' );
					$currentEcard = array(
						'ecard_id' => $productID,
						'recipient_name' => $lineItem['recipient_name'],
						'recipient' => $recepient_email_address,
						'from' => $allmeta['_billing_email'][0],
						'ecard_message' => $lineItem['ecard_message'],
						'ecard_image' => $ecardImage[0],
						'order_id' => $order_id,
						'default_subject' => $lineItem['default_subject'],
					);

					if ( isset( $lineItem['sender_name'] ) ) {
						 $currentEcard['sender_name'] = $lineItem['sender_name'];
					} else {
						$currentEcard['sender_name'] = trim( $allmeta['_billing_first_name'][0] . ' ' . $allmeta['_billing_last_name'][0] );
					}

					if ( isset( $lineItem['sender_email_address'] ) ) {
						$currentEcard['from'] = $lineItem['sender_email_address'];
					}
					if ( isset( $lineItem['copy_to_self'] ) ) {
						$currentEcard['copy_to_self'] = $lineItem['copy_to_self'];
					}
					$ecards_for_now[] = $currentEcard;

				}
			}

			if ( count( $ecards_for_now ) > 0 ) {
				$this->do_ecards_delivery( $ecards_for_now );
			}

		}

		function on_ecard_order_placing( $order_id ) {

			$order = new WC_Order( $order_id );
			$order_has_ecard = false;
			foreach ( $order->get_items() as $key => $item ) {
				$recepient_email_address = wc_get_order_item_meta( $key, 'recipient_email_address' );
				if ( ! empty( $recepient_email_address ) ) {
					$order_has_ecard = true;
					break; }
			}

			if ( $order_has_ecard ) {

				$this->setup_order_meta_for_ecards( $order_id );
				$payment_method = get_post_meta( $order_id, '_payment_method', true );
				if ( $payment_method == 'paypal' ) {
					$this->send_ecard_instantly( $order_id );
				}
				update_post_meta( $order_id, 'convert_into_ecard', 'yes' );
				$total_ecards_sent = get_post_meta( $order_id, 'total_ecard_send', true );
				$total_ecards_sent = ( ! empty( $total_ecards_sent ) ) ? $total_ecards_sent : 0;
				update_post_meta( $order_id, 'total_ecard_send', $total_ecards_sent );
				if ( $this->is_order_has_only_ecards( $order_id ) && $payment_method == 'paypal' && $this->dboptions['autocomplete_ecard_orders'] ) {
					$order->update_status( 'completed' );
				}
			}

		}

		function woo_custom_cart_button_text( $text, $product ) {

			if ( $this->isEcard && ! empty( $this->dboptions['button_labels']['add_to_cart_product'] ) ) {
				return $this->dboptions['button_labels']['add_to_cart_product'];
			}
			return esc_html__( 'Add To Cart', 'woocommerce-ecards-pro' );

		}

		function woo_archive_custom_cart_button_text( $var, $instance ) {

			global $product;
			$ecard = esc_attr( get_post_meta( $product->get_id(), 'convert_into_ecard', true ) );
			if ( ( $ecard == 'yes' ) && ! empty( $this->dboptions['button_labels']['add_to_cart_shop'] ) ) {
				return $this->dboptions['button_labels']['add_to_cart_shop'];
			}
			return esc_html__( 'Add To Cart', 'woocommerce-ecards-pro' );

		}

		function check_for_ecard() {

			global $woocommerce;
			$items = $woocommerce->cart->get_cart();
			$ecard_info = array();
			foreach ( $items as $item => $values ) {

				if ( isset( $values['ecard_message'] ) && ! empty( $values['ecard_message'] ) ) {
					$ecard_info[] = array(
						'ecard_id' => $values['product_id'],
						'recipient_name' => $values['recipient_name'],
						'ecard_message' => $values['ecard_message'],
						'recipient_email_address' => $values['recipient_email_address'],
					);

				}
			}
			if ( count( $ecard_info ) > 0 ) {
				?>
			<script>
			var all_ecards_in_cart = <?php echo json_encode( $ecard_info ); ?>;
			</script>
				<?php
			}
			if ( isset( $this->dboptions['ecard_custom_css'] ) && ! empty( $this->dboptions['ecard_custom_css'] ) ) {
				$custom_css = $this->dboptions['ecard_custom_css'];
			} else {
				$custom_css = ''; }
			?>
			<style>
				.update_ecard_msg_btn{
					background: #f1c40f!important;color:#212F3D!important;margin:10px auto 0!important;font-weight:bold!important;font-family: sans-serif!important;font-size: 13px!important;line-height: 1.1!important;text-align: center;text-decoration: none;display: inline-block;border-radius: 3px;font-weight: bold!important;border:none!important;padding: 15px!important;
				}
			<?php echo esc_html( $custom_css ); ?>	
			</style>
			<?php
			global $post;
			$ecard = esc_attr( get_post_meta( $post->ID, 'convert_into_ecard', true ) );
			$this->isEcard = ( $ecard == 'yes' ) ? true : false;

		}

		function save_ecards_metabox( $post_id, $post ) {

			// Nonce Verification
			if ( ! isset( $_POST['conver_ecard_nonce'] ) || ! wp_verify_nonce( $_POST['conver_ecard_nonce'], basename( __FILE__ ) ) ) {
				return $post_id;
			}

			// Authorization
			$post_type = get_post_type_object( $post->post_type );
			if ( ! current_user_can( $post_type->cap->edit_post, $post_id ) ) {
				return $post_id;
			}

			$convert_into_ecard = ( isset( $_POST['convert_into_ecard'] ) ? sanitize_text_field( $_POST['convert_into_ecard'] ) : '' );
			update_post_meta( $post_id, 'convert_into_ecard', $convert_into_ecard );

			if ( ! empty( $_POST['convert_into_ecard'] ) && ( sanitize_text_field( $_POST['convert_into_ecard'] ) == 'yes' ) ) {
				$ecard_default_subject = ( isset( $_POST['ecard_default_subject'] ) ? sanitize_text_field( $_POST['ecard_default_subject'] ) : '' );
				$ecard_default_email = ( isset( $_POST['ecard_default_email'] ) ? ( sanitize_text_field( $_POST['ecard_default_email'] ) ) : '' );
				update_post_meta( $post_id, 'ecard_default_subject', $ecard_default_subject );
				update_post_meta( $post_id, 'ecard_default_email', $ecard_default_email );
			}

		}

		function register_ecards_metabox() {
			add_action( 'add_meta_boxes', array( $this, 'setup_ecards_metabox' ) );   }

		function setup_ecards_metabox() {

			add_meta_box(
				'covert-product-ecard',
				esc_html__( 'WooCommerce eCards Pro', 'woocommerce-ecards-pro' ),
				array( $this, 'render_ecards_metabox' ),
				'product',
				'side',
				'high'
			);

		}


		function render_ecards_metabox( $object, $box ) {

			// Nonce field creation with custom metabox form.
			wp_nonce_field( basename( __FILE__ ), 'conver_ecard_nonce' );

			$ecard = esc_attr( get_post_meta( $object->ID, 'convert_into_ecard', true ) );
			$ecard_sub = sanitize_text_field( get_post_meta( $object->ID, 'ecard_default_subject', true ) );
			if ( empty( $ecard_sub ) ) {
				$ecard_sub = esc_html__( 'You have received a new eCard.', 'woocommerce-ecards-pro' );
			}
			$ecard_msg = esc_textarea( get_post_meta( $object->ID, 'ecard_default_email', true ) );
			?>
		  <p><?php esc_html_e( 'Convert To eCard ?', 'woocommerce-ecards-pro' ); ?> <input class="widefat" type="checkbox" name="convert_into_ecard" id="convert_into_ecard" value="yes" size="30" <?php checked( $ecard, 'yes' ); ?>/></p>
		  <p><?php esc_html_e( 'Default eCard\'s Email Subject', 'woocommerce-ecards-pro' ); ?> <input class="widefat" type="text" name="ecard_default_subject" id="ecard_default_subject" value="<?php echo esc_html( $ecard_sub ); ?>" size="30" /></p>
		  <p><?php esc_html_e( 'Default Greeting Message', 'woocommerce-ecards-pro' ); ?> <textarea class="widefat" name="ecard_default_email" id="ecard_default_email" rows="6" ><?php echo esc_textarea( $ecard_msg ); ?></textarea></p>
		  <p><?php esc_html_e( 'You can convert this product to an eCard. eCard will be sent to the recipient when the order status is complete.', 'woocommerce-ecards-pro' ); ?><br /></p>
		  <p><?php esc_html_e( 'You can use {sender_name}, {sender_email}, {recipient_email}, {site_url},{website_name} placeholder in your subject & message. Placeholder values will be replaced with dynamic information collected from eCard Form on frontend just before sending Ecard. Important: Senders related placeholders will be processed and displayed in eCard preview to customers only when you are allowing them to enter sender\'s related information on frontend eCard product form.', 'woocommerce-ecards-pro' ); ?><br /></p>
		  
			<?php
		}

		function hook_in_foot() {

			if ( is_product() || is_cart() || is_checkout() ) {

				?>
			<script>
				<?php
				if ( is_product() ) {
					$page = 'product';
				} else if ( is_cart() ) {
					$page = 'cart';
				} else if ( is_checkout() ) {
					$page = 'checkout';
				}
				?>
				var currentpage = "<?php echo esc_html( $page ); ?>";
				var loader_url = "<?php echo esc_url( plugins_url( 'assets/images/Preloader_3.gif', __FILE__ ) ); ?>";
				var website_name = "<?php echo get_bloginfo( 'name' ); ?>";
				var website_url = "<?php echo esc_url( home_url() ); ?>";
				var ecards_ajax_url = "<?php echo esc_url( admin_url( 'admin-ajax.php' ) ); ?>"
				
			</script>
				<?php
				if ( is_product() ) {
					global $post;
					$ecardImage = wp_get_attachment_image_src( get_post_thumbnail_id( $post->ID ), 'full' );
					$info = array(
						'ecard_image' => $ecardImage[0],
						'ecard_id' => $post->ID,
					);

				} else {

					global $woocommerce;
					$items = $woocommerce->cart->get_cart();
					if ( $items ) {

						$cartItemDetails = '';
						
						foreach ( $items as $item => $values ) {
							
							$values['recipient_email_address'] = ( is_array( $values ) && array_key_exists( 'recipient_email_address', $values ) && isset( $values['recipient_email_address'] ) && ! empty( $values['recipient_email_address'] ) ) ? $values['recipient_email_address'] : '';
							$_product = wc_get_product( $values['product_id'] );
							$fullImageSrc = wp_get_attachment_image_src( get_post_thumbnail_id( $_product->get_id() ), 'full' );
							$ecard_heading = ( ! empty( $values['ecard_heading'] ) ) ? $values['ecard_heading'] : '';
							$ecard_msg = ( ! empty( $values['ecard_message'] ) ) ? $values['ecard_message'] : '';
							$title = ( ! empty( $values['recipient_name'] ) ) ? $values['recipient_name'] : '';
							$ecard_title = $title;
							$cartItemDetails .= '<div class="cart_items_details" data-ecard-id="' . $_product->get_id() . '"  data-ecard-recepient="' . $values['recipient_email_address'] . '" data-ecard-img-src="' . $fullImageSrc[0] . '" data-ecard-msg="' . $ecard_msg . '" data-ecard-heading="' . $ecard_heading . '" data-ecard-title="' . html_entity_decode( $ecard_title ) . '">';

						}
						$cartItemDetails .= '</div>';
						echo $cartItemDetails;
					}

					$info = array(
						'ecard_image' => '',
						'ecard_id' => null,
					);

				}
				$ecardPreviewModal = $this->get_preview_markup( $info );
				echo $ecardPreviewModal;
				?>
				<?php
			}
		}

		function wep_update_ecard_meta() {

			$response = array( 'status' => '0' );

			// Check for nonce security
			$nonce = sanitize_text_field( $_POST['nonce'] );
			if ( wp_verify_nonce( $nonce, 'ajax-nonce' ) ) {

				global $woocommerce;
				$ecardToUpdate = sanitize_text_field( $_POST['product_id'] );
				$ecardRecepient = sanitize_text_field( $_POST['recepient_email'] );
				foreach ( $woocommerce->cart->cart_contents as $sessionkey => $item ) {

					if ( $item['recipient_email_address'] == $ecardRecepient && $item['product_id'] == $ecardToUpdate && isset( $item['ecard_message'] ) ) {

						$fields_to_update = array( 'ecard_heading', 'ecard_message' );
						foreach ( $fields_to_update as $field ) {
							$fieldValue = strip_tags( $_POST[ $field ], '<br><p><div><br>' );
							$fieldValue = wp_kses_post( $fieldValue ); // Sanitise User Inputs
							$fieldValue = htmlentities( stripslashes( $fieldValue ) );
							$woocommerce->cart->cart_contents[ $sessionkey ][ $field ] = $fieldValue;
						}

						$woocommerce->cart->set_session();
						$response = array( 'status' => '1' );
						break;

					}
				}
			}

			echo json_encode( $response );
			exit;

		}

		function ecard_order_meta_handler( $item_id, $values, $cart_item_key ) {

			$orderMeta = $this->extraCartFields;
			$orderMeta[] = 'convert_into_ecard';
			foreach ( $orderMeta as $meta ) {
				if ( isset( $values[ $meta ] ) ) {
					wc_add_order_item_meta( $item_id, $meta, $values[ $meta ] );
				}
			}

		}

		function render_meta_on_cart_and_checkout( $cart_data, $cart_item = null ) {

			if ( empty( $cart_item['ecard_message'] ) ) {
				return $cart_data;
			}

			$custom_items = array();

			if ( ! empty( $cart_data ) ) {
				$custom_items = $cart_data;
			}
			$showOnCart = ( is_cart() && ( isset( $this->dboptions['ask_order_cart'] ) && $this->dboptions['ask_order_cart'] == 'true' ) ) ? true : false;
			$showOnCheckOut = ( is_checkout() && ( isset( $this->dboptions['ask_order_checkout'] ) && $this->dboptions['ask_order_checkout'] == 'true' ) ) ? true : false;
			if ( $showOnCart || $showOnCheckOut ) {

				if ( $showOnCart ) {
					if ( ! empty( $this->dboptions['cart_page_ecard_fields'] ) ) {
						$custom_cart_items = $this->dboptions['cart_page_ecard_fields'];
					} else {
						$custom_cart_items = array( 'recipient_email_address', 'ecard_delivery_date' );
					}
				}

				if ( $showOnCheckOut ) {
					if ( ! empty( $this->dboptions['checkout_page_ecard_fields'] ) ) {
						$custom_cart_items = $this->dboptions['checkout_page_ecard_fields'];
					} else {
						$custom_cart_items = array( 'recipient_email_address', 'ecard_delivery_date' );
					}
				}

				foreach ( $custom_cart_items as $field_value ) {

					if ( 'ecard_delivery_date' == $field_value ) {

						if ( isset( $cart_item[ $field_value ] ) && ! empty( $cart_item[ $field_value ] ) ) {
							$cart_item[ $field_value ] = date( apply_filters( 'wep_date_format', 'F, d Y' ), strtotime( $cart_item[ $field_value ] ) );
						} else {
							$cart_item[ $field_value ] = esc_html__( 'Today', 'woocommerce-ecards-pro' );
						}
					}

					if ( isset( $this->dboptions[ $field_value ] ) && ! empty( $cart_item[ $field_value ] ) ) {

						if ( 'ecard_message' == $field_value ) {

							$placeholders = array(
								'recipient_name' => $cart_item['recipient_name'],
								'sender_name' => $cart_item['sender_name'],
								'sender_email' => $cart_item['sender_email_address'],
								'site_url' => esc_url( site_url() ),
								'website_url' => esc_url( site_url() ),
								'website_name' => get_bloginfo( 'name' ),
							);
							foreach ( $placeholders as $p_key => $p_value ) {
								$cart_item[ $field_value ] = str_replace( '{' . $p_key . '}', $p_value, $cart_item[ $field_value ] );
							}
						}

						$custom_items[] = array(
							'name' => $this->dboptions[ $field_value ],
							'value' => html_entity_decode( $cart_item[ $field_value ] ),
						);

					}

					$product_url = get_permalink( $cart_item['product_id'] );

				}

				$ecard_id = $cart_item['product_id'];
				$ecard_receiver = $cart_item['recipient_email_address'];

				if ( is_cart() && isset( $this->dboptions['allow_update_cart'] ) && $this->dboptions['allow_update_cart'] == 'true' ) {

					$ecard_dynamic = $cart_item['product_id'] . '/' . $cart_item['recipient_email_address'] . '/' . $cart_item['recipient_name'];
					$updateEcard = esc_html__( 'Update eCard', 'woocommerce-ecards-pro' );
					$custom_items[] = array(
						'name' => $updateEcard,
						'value' => '<div class="ecard-update-link-container ecardupdate-container">
				<a href="javascript:void(0);" name="e-card-update"  class="e-card-update" data-ecard-id="' . esc_attr( $ecard_id ) . '" data-ecard-recepient="' . esc_attr( $ecard_receiver ) . '" id="update/' . esc_attr( $ecard_dynamic ) . '">' . $updateEcard . '</a></div>',
					);
				}

				if ( isset( $this->dboptions['ask_preview_information_cart'] ) ) {
					$onCartPage = $this->dboptions['ask_preview_information_cart'];
				}

				if ( isset( $this->dboptions['ask_preview_information_checkout'] ) ) {
					$onCheckoutPage = $this->dboptions['ask_preview_information_checkout'];
				}

				$displayPreview = false;
				if ( is_cart() && isset( $onCartPage ) && $onCartPage == 'true' ) {
					$displayPreview = true;
				} else if ( is_checkout() && isset( $onCheckoutPage ) && $onCheckoutPage == 'true' ) {
					$displayPreview = true;
					$ecard_dynamic = $cart_item['product_id'] . '/' . $cart_item['recipient_email_address'];
				}

				if ( $displayPreview ) {

					$previewEcard = esc_html__( 'Preview eCard', 'woocommerce-ecards-pro' );
					$custom_items[] = array(
						'name' => $previewEcard,
						'value' => '<div class="ecard-preview-link-container ecardpreview-container ' . esc_attr( $cart_item['product_id'] ) . '"><a href="javascript:void(0);" name="e-card-preview" class="e-card-preview" id="preview/' . esc_attr( $ecard_dynamic ) . '">' . $previewEcard . '</a></div>',
					);

				}
			}

			return $custom_items;

		}

		function save_extra_card_order_info( $cart_item_data, $product_id ) {

			foreach ( $this->extraCartFields as $field ) {

				if ( isset( $_REQUEST[ $field ] ) ) {
					$_REQUEST[ $field ] = strip_tags( $_REQUEST[ $field ], '<br><p><div><br>' );
					$_REQUEST[ $field ] = wp_kses_post( $_REQUEST[ $field ] );
					$cart_item_data[ $field ] = stripslashes( $_REQUEST[ $field ] );
					$cart_item_data['unique_key'] = md5( microtime() . rand() );
				}
			}
			$cart_item_data['ecard_delivered'] = false;
			$cart_item_data['ecard_delivery_time'] = '';
			if ( isset( $_REQUEST['sender_name'] ) && ! empty( $_REQUEST['sender_name'] ) ) {
				$cart_item_data['ecard_message'] = str_replace( '{sender_name}', $_REQUEST['sender_name'], $cart_item_data['ecard_message'] );
			}
			if ( isset( $_REQUEST['sender_email_address'] ) && ! empty( $_REQUEST['sender_email_address'] ) ) {
				$cart_item_data['ecard_message'] = str_replace( '{sender_email}', $_REQUEST['sender_email_address'], $cart_item_data['ecard_message'] );
			}

			if ( isset( $cart_item_data['ecard_message']) && ! empty( $_REQUEST['ecard_message'] ) ) {
			$cart_item_data['ecard_message'] = str_replace( '{site_url}', esc_url( home_url() ), $cart_item_data['ecard_message'] );
			$cart_item_data['ecard_message'] = str_replace( '{website_name}', get_bloginfo( 'name' ), $cart_item_data['ecard_message'] );
			}
			return $cart_item_data;

		}

		function apply_validations_to_extrafields( $validation, $product_id, $quantity ) {

			$ecard = esc_attr( get_post_meta( $product_id, 'convert_into_ecard', true ) );

			if ( $ecard == 'yes' ) {

				do_action( 'ecp_apply_validation' );

				if ( empty( $_REQUEST['recipient_name'] ) ) {
					wc_add_notice( esc_html__( 'Please enter recipient\'s full name.', 'woocommerce-ecards-pro' ), 'error' );
					$validation = false;
				}

				if ( empty( $_REQUEST['recipient_email_address'] ) ) {
					wc_add_notice( esc_html__( 'Please enter recipient\'s email address.', 'woocommerce-ecards-pro' ), 'error' );
					$validation = false;
				}

				if ( ! empty( $_REQUEST['recipient_email_address'] ) ) {

					$invalid_email = false;
					$all_emails_add = explode( ',', $_REQUEST['recipient_email_address'] );
					if ( is_array( $all_emails_add ) && ( count( $all_emails_add ) > 1  ) ) {
						foreach ( $all_emails_add as $address ) {
							if ( ! filter_var( $address, FILTER_VALIDATE_EMAIL ) ) {
								wc_add_notice( esc_html__( 'Please enter a valid recipient email address.', 'woocommerce-ecards-pro' ), 'error' );
								$validation = false;
								break;
							}
						}
					} else {
						if ( ! filter_var( $_REQUEST['recipient_email_address'], FILTER_VALIDATE_EMAIL ) ) {
							wc_add_notice( esc_html__( 'Please enter a valid recipient email address.', 'woocommerce-ecards-pro' ), 'error' );
							$validation = false;

						}
					}
						 
				}

				if ( isset( $_REQUEST['ecard_delivery_date'] ) && empty( $_REQUEST['ecard_delivery_date'] ) ) {
					wc_add_notice( esc_html__( 'Please enter ecard delivery date.', 'woocommerce-ecards-pro' ), 'error' );
					$validation = false;
				}

				if ( ! empty( $_REQUEST['ecard_delivery_date'] ) && ! (bool) strtotime( $_REQUEST['ecard_delivery_date'] ) ) {
					wc_add_notice( esc_html__( 'Please enter a valid ecard delivery date.', 'woocommerce-ecards-pro' ), 'error' );
					$validation = false;
				}

				if ( empty( $_REQUEST['ecard_heading'] ) ) {
					wc_add_notice( esc_html__( 'Please enter ecard heading.', 'woocommerce-ecards-pro' ), 'error' );
					$validation = false;
				}

				if ( empty( $_REQUEST['default_subject'] ) ) {
					wc_add_notice( esc_html__( 'Please enter ecard subject.', 'woocommerce-ecards-pro' ), 'error' );
					$validation = false;
				}

				if ( empty( $_REQUEST['ecard_message'] ) ) {
					wc_add_notice( esc_html__( 'Please enter ecard message.', 'woocommerce-ecards-pro' ), 'error' );
					$validation = false;
				}
				if ( ! empty( $_POST['contains_sender_information'] ) ) {
					if ( empty( $_REQUEST['sender_name'] ) ) {
						wc_add_notice( esc_html__( 'Please enter sender\'s name.', 'woocommerce-ecards-pro' ), 'error' );
						$validation = false;
					}
					if ( empty( $_REQUEST['sender_email_address'] ) ) {
						wc_add_notice( esc_html__( 'Please enter sender\'s email address.', 'woocommerce-ecards-pro' ), 'error' );
						$validation = false;
					}
				}
			}

			return $validation;
		}

		function is_product_added( $pid ) {

			global $woocommerce;
			$added = false;
			$items = $woocommerce->cart->get_cart();
			if ( $items ) {

				foreach ( $items as $item => $values ) {

					$_product = wc_get_product( $values['product_id'] );
					if ( $_product->get_id() == $pid ) {
						$added = true;
						break;
					}
				}
			}
			 return $added;

		}

		function get_preview_markup( $info ) {

			$ecardImage = wp_get_attachment_image_src( get_post_thumbnail_id( $info['ecard_id'] ), 'full' );
			$ecardForToday = array(
				'ecard_image' => $info['ecard_image'],
				'recipient' => '',
				'from' => '<b>From Email Address</b>',
				'ecard_message' => '',
			);

			$preview = $this->get_ecard_markup( $ecardForToday );

			$innercontent = '<div class="modal-body">
					<div class="ecard-real-preview">' . $preview . '</div>
				  </div>';

			$footerContent = $this->dboptions['ecard_footer'];
			$footerContent = str_replace( '{sitename}', get_bloginfo( 'name' ), $footerContent );

			$previewModal = '<div id="ecardpreview-container" class="fc-modal">
							  <div class="fc-modal-content">
				<div class="fc-modal-header">
					<span class="fc-modal-close"></span>
					<h3>' . esc_html__( 'eCard Preview', 'woocommerce-ecards-pro' ) . '</h3>
				  </div>
				  <div class="fc-modal-body">' . $innercontent . '</div>
				  <div class="fc-modal-footer"></div>
			  </div>
		  </div>';
			return $previewModal;

		}

		function add_exta_cart_fields() {

			global $post;

			if ( is_product() && $this->isEcard ) {

				   $extraCartFields = $this->extraCartFields;
				if ( $this->is_product_added( get_the_ID() ) ) {

					foreach ( $extraCartFields as $field ) {

						if ( isset( $_POST[ $field ] ) ) {
							unset( $_POST[ $field ] );
						}
					}

						$_POST['default_subject'] = get_post_meta( get_the_ID(), 'ecard_default_subject', true );
						$_POST['ecard_message'] = get_post_meta( get_the_ID(), 'ecard_default_email', true );

				}

					$ecard_sub = sanitize_text_field( get_post_meta( $post->ID, 'ecard_default_subject', true ) );
					$ecard_msg = esc_textarea( get_post_meta( $post->ID, 'ecard_default_email', true ) );
					$recipient_name = ( isset( $_POST['recipient_name'] ) ) ? sanitize_text_field( $_POST['recipient_name'] ) : '';
					$recipient_email_address = ( isset( $_POST['recipient_email_address'] ) ) ? sanitize_text_field( $_POST['recipient_email_address'] ) : '';
					$ecard_delivery_date = ( isset( $_POST['ecard_delivery_date'] ) ) ? sanitize_text_field( $_POST['ecard_delivery_date'] ) : '';
					$already_added = $this->is_product_added( $post->ID );
					$default_subject = ( isset( $_POST['default_subject'] ) || $already_added ) ? sanitize_text_field( $_POST['default_subject'] ) : $ecard_sub;

					$ecard_heading = ( isset( $_POST['ecard_heading'] ) ) ? sanitize_text_field( $_POST['ecard_heading'] ) : '';
					$ecard_message = ( isset( $_POST['ecard_message'] ) || $already_added ) ? sanitize_text_field( $_POST['ecard_message'] ) : $ecard_msg;
					$sender_name = ( isset( $_POST['sender_name'] ) ) ? sanitize_text_field( $_POST['sender_name'] ) : '';
					$sender_email_address = ( isset( $_POST['sender_email_address'] ) ) ? sanitize_text_field( $_POST['sender_email_address'] ) : '';

					$senderNameLabel = ( isset( $this->dboptions['sender_name'] ) && ! empty( $this->dboptions['sender_name'] ) ) ? $this->dboptions['sender_name'] : esc_html__( 'Sender Name', 'woocommerce-ecards-pro' );
					$senderEmailLabel = ( isset( $this->dboptions['sender_email_address'] ) && ! empty( $this->dboptions['sender_email_address'] ) ) ? $this->dboptions['sender_email_address'] : esc_html__( 'Sender Email', 'woocommerce-ecards-pro' );
					$copyToSelf = ( isset( $this->dboptions['copyself_label'] ) && ! empty( $this->dboptions['copyself_label'] ) ) ? $this->dboptions['copyself_label'] : esc_html__( 'Copy to Self', 'woocommerce-ecards-pro' );
					$sendersInfo = '';
				if ( isset( $this->dboptions['ask_sender_information'] ) && $this->dboptions['ask_sender_information'] == 'true' ) {
					$sendersInfo .= '<tr>
						  <td class="label"><label for="sender_name">' . $senderNameLabel . '&nbsp;<span>*</span></label></td>
						  <td class="ask_sender_information">
							  <input type="text" name="sender_name" id="sender_name" value="' . $sender_name . '" />
						  </td>
						</tr><tr>
						  <td class="label"><label for="sender_email_address">' . $senderEmailLabel . '&nbsp;<span>*</span></label></td>
						  <td class="value">
							  <input type="text" name="sender_email_address" id="sender_email_address" value="' . $sender_email_address . '" />
							  <input type="hidden" name="contains_sender_information" value="yes">
						  </td>
						</tr>';

				}
					$copy_to_self = '';
				if ( isset( $this->dboptions['copy_to_self'] ) && $this->dboptions['copy_to_self'] == 'true' ) {
					$copy_to_self .= '
					 <tr>
						  <td class="label"><label for="copy_to_self">' . $copyToSelf . ':</label></td>
						  <td class="value">
							  <input type="checkbox" name="copy_to_self" id="copy_to_self" value="true" />
						  </td>
						</tr>';
				}
					$previewEcard = '';
				if ( isset( $this->dboptions['ask_preview_information'] ) && $this->dboptions['ask_preview_information'] == 'true' ) {
					$previewEcard .= ' <tr>
							  <td class="label"><label for="color"></label></td>
							  <td class="value">
								  <a href="javascript:void(0);" data-target="ecardpreview-container" name="e-card-preiview" id="e-card-preiview" class="fc-modal-btn">Ecard Preview</a>
							  </td>
						  </tr>';
				}

					$sendEcardOnLabel = ( isset( $this->dboptions['ecard_delivery_date'] ) && ! empty( $this->dboptions['ecard_delivery_date'] ) ) ? $this->dboptions['ecard_delivery_date'] : esc_html__( 'Send Ecard On', 'woocommerce-ecards-pro' );

					$askDate = '';
				if ( isset( $this->dboptions['ask_date_information'] ) && $this->dboptions['ask_date_information'] == 'true' ) {
					$askDate = '<tr>
							  <td class="label"><label for="ecard_delivery_date" class="ecard_delivery_date_label">' . $sendEcardOnLabel . ' <span>*</span></label></td>
							  <td class="value">
								  <input type="text" name="ecard_delivery_date" id="ecard_delivery_date" value="' . $ecard_delivery_date . '" class="datepicker" id="ecard_delivery_date"/>
							  </td>
						  </tr> ';
				}

				   $recepientEmailLabel = ( isset( $this->dboptions['recipient_email_address'] ) && ! empty( $this->dboptions['recipient_email_address'] ) ) ? $this->dboptions['recipient_email_address'] : esc_html__( 'Recipient Email Address', 'woocommerce-ecards-pro' );
					$subjectLabel = ( isset( $this->dboptions['default_subject'] ) && ! empty( $this->dboptions['default_subject'] ) ) ? $this->dboptions['default_subject'] : esc_html__( 'eCard Subject', 'woocommerce-ecards-pro' );

					$headingLabel = ( isset( $this->dboptions['ecard_heading'] ) && ! empty( $this->dboptions['ecard_heading'] ) ) ? $this->dboptions['ecard_heading'] : esc_html__( 'eCard Heading', 'woocommerce-ecards-pro' );

					$messageLabel = ( isset( $this->dboptions['ecard_message'] ) && ! empty( $this->dboptions['ecard_message'] ) ) ? $this->dboptions['ecard_message'] : esc_html__( 'eCard Message', 'woocommerce-ecards-pro' );

					$recipient_name_label = ( isset( $this->dboptions['recipient_name'] ) && ! empty( $this->dboptions['recipient_name'] ) ) ? $this->dboptions['recipient_name'] : esc_html__( 'Recipient Name', 'woocommerce-ecards-pro' );
					$multiple_name_desc = $multiple_email_desc = '';
				if ( isset( $this->dboptions['enable_multiple_recepients'] ) && ( $this->dboptions['enable_multiple_recepients'] == 'true' ) ) {
					$multiple_name_desc = apply_filters( 'ecp_multiple_recepients_name_desc', esc_html__( 'You can enter comma seperated full names of multiple recepeints.', 'woocommerce-ecards-pro' ) );
					$multiple_email_desc = apply_filters( 'ecp_multiple_recepients_email_desc', esc_html__( 'You can enter comma seperated valid email addresses of multiple recepients. It must be in sequence with comma seperated names provided above.', 'woocommerce-ecards-pro' ) );
				}
					$ecardForm = '<table class="variations ecard-fields" cellspacing="0">
						  <tbody>
						    <tr>
							  <td class="label"><label for="recipient_name">' . $recipient_name_label . ' <span>*</span></label></td>
							  <td class="value">
								  <textarea name="recipient_name" id="recipient_name">' . $recipient_name . '</textarea>
								  <p class="multiple_name_desc">' . $multiple_name_desc . '</p>
							  </td>
						  </tr>
						    <tr>
							  <td class="label"><label for="recipient_email_address">' . $recepientEmailLabel . ' <span>*</span></label></td>
							  <td class="value">
							      <textarea name="recipient_email_address" id="recipient_email_address">' . $recipient_email_address . '</textarea>
							      <p class="multiple_email_desc">' . $multiple_email_desc . '</p>
							  </td>
						  </tr>  
						  ' . $askDate . '
						  <tr>
							  <td class="label"><label for="default_subject">' . $subjectLabel . ' <span>*</span></label></td>
							  <td class="value">
								  <textarea name="default_subject" id="default_subject">' . $default_subject . '</textarea>
							  </td>
						  </tr>
						  <tr>
							  <td class="label"><label for="ecard_heading">' . $headingLabel . ' <span>*</span></label></td>
							  <td class="value">
								  <input type="text" name="ecard_heading" id="ecard_heading" value="' . $ecard_heading . '" />
							  </td>
						  </tr>
							
						  <tr>
							  <td class="label"><label for="ecard_message">' . $messageLabel . ' <span>*</span></label></td>
							  <td class="value">
								  <textarea name="ecard_message" id="ecard_message">' . $ecard_message . '</textarea>
							  </td>
						  </tr>' . $sendersInfo . $copy_to_self . $previewEcard . '
						  </tbody>
					  </table><input type="hidden" name="convert_into_ecard" value="yes">';
					  $ecardForm = apply_filters( 'ecp_ecard_form', $ecardForm );
					  echo $ecardForm;
			}

		}

		function deliver_woo_ecards() {

			global $wpdb;
			$ecards = $this->get_all_orders();

			if ( ! empty( $ecards ) ) {

				foreach ( $ecards as $ecardkey => $ecardForToday ) {

					$to = trim( $ecardForToday['recipient'] );
					$recipient_name = $ecardForToday['recipient_name'];
					$all_recepients = explode( ',', $to );
					if ( is_array( $all_recepients ) && ( count( $all_recepients ) > 1 ) ) {

						$final_recepients = array();
						$all_recepient_names = explode( ',', $recipient_name );

						foreach ( $all_recepients as $key => $recepient ) {

							$temp = $ecardForToday;
							$temp['recipient'] = $recepient;
							$temp['recipient_name'] = $recipient_name;
							$temp['all_recipients_email'] = $to;
							$temp['recipient_total_count'] = count( $all_recepients );
							$final_recepients[] = $temp;

						}
					} else {
						
							$ecardForToday['all_recipients_email'] = $to;
							$ecardForToday['recipient_total_count'] = 1;
							$final_recepients[] = $ecardForToday;
					}
				}
			}

			if ( ! empty( $final_recepients ) ) {
				$this->do_ecards_delivery( $final_recepients );
			}

		}

		function deliver_ecard_email( $to, $fromEmail, $fromName, $subject, $message, $headers ) {

			$delivered = false;

			switch ( $this->dboptions['send_via_mandrill'] ) {
				case 'true':
					if ( ! empty( $this->dboptions['mandrill_key'] ) ) {

						$mandrillApi = $this->dboptions['mandrill_key'];
						$mandrill = new Mandrill( $mandrillApi );
						$message = array(
							'subject' => $subject,
							'from_email' => $fromEmail,
							'html' => $message,
							'from_name' => $fromName,
							'to' => array(
								array( 'email' => $to ),
							),
						);

						$result = $mandrill->messages->send( $message );
						if ( $result[0]['status'] == 'sent' ) {
							$delivered = true;
						} else {

							if ( wp_mail( $to, $subject, $message . 'via_fallback', $headers ) ) {
								$delivered = true;
							}
						}
					} else {
						if ( wp_mail( $to, $subject, $message, $headers ) ) {
							$delivered = true;
						}
					}
					break;
				default:
					if ( wp_mail( $to, $subject, $message, $headers ) ) {
						$delivered = true;
					}
			}

			return $delivered;

		}

		function getTextBetweenTags( $string, $tagname ) {

			$pattern = "/<$tagname ?.*>(.*)<\/$tagname>/";
			preg_match( $pattern, $string, $matches );
			return $matches[1];
		}

		function do_ecards_delivery( $ecards ) {

			$website_name = get_bloginfo( 'name' );
			$website_url = esc_url( home_url() );
			$website_url = str_replace( 'https://', '', $website_url );
			$website_url = str_replace( 'http://', '', $website_url );
			$website_url = str_replace( 'www.', '', $website_url );

			foreach ( $ecards as $ecardkey => $ecardForToday ) {

				$eid = $ecardForToday['ecard_id'];
				$orderMeta = get_post_meta( intval( $ecardForToday['order_id'] ), 'all_ecards_in_order', true );
				$totalSend = get_post_meta( $ecardForToday['order_id'], 'total_ecard_send', true );
				$to = trim( $ecardForToday['recipient'] );
				$all_recipients_email = trim( $ecardForToday['all_recipients_email'] );
				preg_match( '~>\K[^<>]*(?=<)~', $to, $match );
				if ( ! empty( $match[0] ) ) {
					$to = $match[0];
				}
				$fromEmail = $ecardForToday['from'];
				preg_match( '~>\K[^<>]*(?=<)~', $fromEmail, $match );
				if ( ! empty( $match[0] ) ) {
					$fromEmail = $match[0];
				}

				$fromName = $ecardForToday['sender_name'];
				if ( $fromName != '' ) {
					$sender_name = $fromName;
				} else {
					$sender_name = esc_html__( 'Someone', 'woocommerce-ecards-pro' );
				}
				$default_subject = $ecardForToday['default_subject'];
				$recipient_name = $ecardForToday['recipient_name'];
				if ( $default_subject == '' ) {
					$subject = ucwords( $sender_name ) . esc_html__( ' has sent you a card from ', 'woocommerce-ecards-pro' ) . $website_name;
					$subject = apply_filters( 'ecp_default_email_heading', $subject, $sender_name, $website_name );
				} else {
					$placeholders = array(
						'recipient_name' => $recipient_name,
						'sender_name' => $sender_name,
						'site_url' => $website_url,
						'website_url' => $website_url,
						'website_name' => $website_name,
					);
					foreach ( $placeholders as $p_key => $p_value ) {
						$default_subject = str_replace( '{' . $p_key . '}', $p_value, $default_subject );
					}

					$subject = $default_subject;
				}

				$fromName = get_bloginfo( 'name' );
				$message = $this->get_ecard_markup( $ecardForToday );
				$headers  = 'MIME-Version: 1.0' . "\r\n";
				$headers .= 'Content-type: text/html; charset=' . get_bloginfo( 'charset' ) . '' . "\r\n";
				$headers .= 'From: ' . $fromName . ' <' . $fromEmail . '>' . "\r\n";

				if ( $this->deliver_ecard_email( $to, $fromEmail, $fromName, $subject, $message, $headers ) ) {

					   $newCount = $totalSend + 1;
					   $new_order_meta = $orderMeta;

					foreach ( $orderMeta as $key => $ecards_in_order ) {

						 $plainEmail = $ecards_in_order['recipient_email_address'];
						if ( $ecardForToday['recipient_total_count'] > 1 ) {
							 $condition = ( $ecards_in_order['ecard_id'] == $eid && $plainEmail == $all_recipients_email ) ? true : false;
						} else {
							$condition = ( $ecards_in_order['ecard_id'] == $eid && $plainEmail == $to ) ? true : false;
						}

						if ( $condition ) {

							if ( $ecardForToday['recipient_total_count'] == $newCount || $newCount > $ecardForToday['recipient_total_count'] ) {
								$new_order_meta[ $key ]['ecard_send'] = true;
							}
							$new_order_meta[ $key ]['ecard_deliverery_time'] = time();
							update_post_meta( $ecardForToday['order_id'], 'total_ecard_send', $newCount );
							update_post_meta( $ecardForToday['order_id'], 'all_ecards_in_order', $new_order_meta );
							update_post_meta( $ecardForToday['order_id'], 'convert_into_ecard', 'yes' );

						}
					}

					if ( ! empty( $ecardForToday['copy_to_self'] ) && $ecardForToday['copy_to_self'] == 'true' ) {

						$newheaders  = 'MIME-Version: 1.0' . "\r\n";
						$newheaders .= 'Content-type: text/html; charset=' . get_bloginfo( 'charset' ) . '' . "\r\n";
						$newheaders .= 'From: ' . $fromName . ' <' . $fromEmail . '>' . "\r\n";
						$to = ( isset( $ecardForToday['sender_email_address'] ) ) ? $ecardForToday['sender_email_address'] : $fromEmail;
						$fromEmail = get_option( 'admin_email' );
						$this->deliver_ecard_email( $to, $fromEmail, $fromName, $subject, $message, $newheaders );

					}

					  $this->update_order_if_all_ecards_processed( $ecardForToday['order_id'] );

				}
			}

		}

		function update_order_if_all_ecards_processed( $order_id ) {

			$orderMeta = get_post_meta( $order_id, 'all_ecards_in_order', true );
			$all_ecards_delivered = true;
			foreach ( $orderMeta as $key => $ecards_in_order ) {

				if ( ! $ecards_in_order['ecard_send'] ) {
					$all_ecards_delivered = false;
					break;
				}
			}
			if ( $all_ecards_delivered ) {
				update_post_meta( $order_id, 'ecard_mail_send', 'true' );
			}

		}



		function is_ecard_already_send( $data ) {

			$alreadysend = false;
			$order_id = $data['order_id'];
			$receiver = $data['receiver'];
			$ecard_id = $data['ecard_id'];

			$orderMeta = get_post_meta( $order_id, 'all_ecards_in_order', true );
			foreach ( $orderMeta as $key => $ecards_in_order ) {

				if ( $ecards_in_order['recipient_email_address'] == $receiver &&
				$ecards_in_order['ecard_id'] == $ecard_id ) {
					if ( $ecards_in_order['ecard_send'] ) {
						$alreadysend = true;
					}
					break;
				}
			}
			return $alreadysend;

		}

		function get_all_orders() {

			$metaquery = array(
				'relation' => 'AND',
				array(
					'key' => 'convert_into_ecard',
					'compare' => 'EXISTS',
				),
				array(
					'key' => 'ecard_mail_send',
					'compare' => 'NOT EXISTS',
				),
			);

			$args = array(
				'post_status' => array( 'wc-completed' ),
				'post_type' => 'shop_order',
				'posts_per_page' => -1,
				'meta_query' => $metaquery,
			);

			$args = apply_filters( 'fc_wcep_get_orders_args', $args );

			$ecards = new WP_Query( $args );
			$today = date( 'Y-m-d' );
			$now = time();
			$ecards_for_today = array();

			while ( $ecards->have_posts() ) {

				$ecards->the_post();
				$order = new WC_Order( $ecards->post->ID );
				$allmeta = get_post_meta( $ecards->post->ID );
				$ecardsInOrder = maybe_unserialize( $allmeta['all_ecards_in_order'][0] );

				foreach ( $order->get_items() as $key => $lineItem ) {

					$ecardHeading = wc_get_order_item_meta( $key, 'ecard_heading' );

					$recepient_email_address = wc_get_order_item_meta( $key, 'recipient_email_address' );
					if ( empty( $recepient_email_address ) ) {
						continue;
					}

					$data = array(
						'order_id' => $ecards->post->ID,
						'receiver' => $recepient_email_address,
						'ecard_id' => $lineItem['product_id'],
					);

					$alreadysend = $this->is_ecard_already_send( $data );
					if ( $alreadysend ) {
						continue;
					}

					$sendEcard = false;

					if ( $now >= strtotime( $lineItem['ecard_delivery_date'] ) ) {
						$sendEcard = true;
					}

					if ( $sendEcard ) {

						$productID = $lineItem['product_id'];
						$ecardImage = wp_get_attachment_image_src( get_post_thumbnail_id( $productID ), 'full' );
						$currentEcard = array(
							'ecard_id' => $productID,
							'recipient_name' => $lineItem['ecard_heading'],
							'recipient' => $recepient_email_address,
							'from' => $allmeta['_billing_email'][0],
							'ecard_message' => $lineItem['ecard_message'],
							'ecard_image' => $ecardImage[0],
							'order_id' => $order->id,
							'default_subject' => $lineItem['default_subject'],
						);

						if ( isset( $lineItem['sender_name'] ) ) {
							$currentEcard['sender_name'] = $lineItem['sender_name'];
						} else {
							$currentEcard['sender_name'] = trim( $allmeta['_billing_first_name'][0] . ' ' . $allmeta['_billing_last_name'][0] );
						}

						if ( isset( $lineItem['sender_email_address'] ) ) {
							$currentEcard['from'] = $lineItem['sender_email_address'];
						}
						if ( isset( $lineItem['copy_to_self'] ) ) {
							$currentEcard['copy_to_self'] = $lineItem['copy_to_self'];
						}
						$ecards_for_today[] = $currentEcard;

					}
				}
			}

			wp_reset_query();

			return $ecards_for_today;

		}

		function send_ecards_to_users() {

			// #1 Nonce security verification
			// #2 Authorization of request verification
			$nonce = sanitize_text_field( $_POST['nonce'] );
			if ( wp_verify_nonce( $nonce, 'ecp-call-nonce' ) && current_user_can( 'administrator' ) ) {

				$userChoice = $_POST['user_choice'];
				$fromName = get_bloginfo( 'name' );
				if ( ! isset( $_POST['default_subject'] ) ) {
					$subject = apply_filters( 'ecard_mail_subject_users', ' Ecard Received - ' . $fromName );
				} else {
					$subject = $_POST['default_subject'];
				}

				$headers  = 'MIME-Version: 1.0' . "\r\n";
				$headers .= 'Content-type: text/html; charset=' . get_bloginfo( 'charset' ) . '' . "\r\n";
				$headers .= 'From: ' . $fromName . ' <' . get_bloginfo( 'admin_email' ) . '>' . "\r\n";
				$fromName = get_bloginfo( 'name' );

				if ( $userChoice == 'particular_email' ) {

					$ecardForToday = array(
						'ecard_image' => $_POST['ecard_img'],
						'recipient_name' => $_POST['ecard_title'],
						'from' => get_bloginfo( 'admin_email' ),
						'ecard_message' => $_POST['ecard_msg'],
					);
					
					$ecardForToday['recipient_name'] = str_replace( '{username}', esc_html__( 'Friend', 'woocommerce-ecards-pro' ), $ecardForToday['recipient_name'] );
					$ecardForToday['ecard_message'] = str_replace( '{sender_name}', esc_html__( 'Someone', 'woocommerce-ecards-pro' ), $ecardForToday['ecard_message'] );
					$ecardForToday['ecard_message'] = str_replace( '{recipient_email}', $_POST['particular_receiver'], $ecardForToday['ecard_message'] );
					$ecardForToday['ecard_message'] = str_replace( '{website_name}', get_bloginfo( 'name' ), $ecardForToday['ecard_message'] );
					
					$message = $this->get_ecard_markup( $ecardForToday );
					wp_mail( $_POST['particular_receiver'], $subject, $message, $headers );

				} else {

					if ( $userChoice == 'all' ) {
						 $args = array();
					} else if ( $userChoice == 'role_based' ) {
						 $args = array( 'role=' . $_POST['user_role'] );
					} else {
						 $users = $_POST['user_listing'];
						 $args = array(
							 'include'      => $users,
							 'orderby'      => 'nicename',
							 'order'        => 'ASC',
						 );

					}
					$site_users = get_users( $args );
					if ( ! empty( $site_users ) ) {
						foreach ( $site_users as $user ) {
							$uname = $user->display_name;
							if ( empty( $uname ) ) {
								$uname = $user->first_name;
							}
							$heading = str_replace( '{username}', $uname, $_POST['ecard_title'] );
							$ecardForToday = array(
								'ecard_image' => $_POST['ecard_img'],
								'recipient_name' => $heading,
								'from' => get_bloginfo( 'admin_email' ),
								'ecard_message' => $_POST['ecard_msg'],
							);
							$message = $this->get_ecard_markup( $ecardForToday );
							wp_mail( $user->user_email, $subject, $message, $headers );
						}
					}
				}
			}
			echo json_encode( array( 'status' => '1' ) );
			exit;

		}

		function get_ecard_markup( $ecardForToday ) {

			if ( ! empty( $ecardForToday ) && class_exists( 'FC_Email' ) ) {

				$email = new FC_Email();
				if ( isset( $ecardForToday['recipient_name'] ) && $ecardForToday['recipient_name'] != '' ) {
					$name = ucwords( $ecardForToday['recipient_name'] );
					$email->email_header( array( 'title' => $name ) );
				} else {
					$name = '{Recepient Name}';
					$email->email_header( array( 'title' => $name ) );

				}
				$title = '<div class="ecard_receipient_name">' . $name . '</div>';

				if ( ! empty( $this->dboptions['visit_site_label'] ) ) {
					$visit_site_label = $this->dboptions['visit_site_label'];
				} else {
					$visit_site_label = esc_html__( 'Visit Site', 'woocommerce-ecards-pro' );
				}

				$subject = ( ! empty( $ecardForToday['default_subject'] ) ) ? $ecardForToday['default_subject'] : 'You have got an eCard';
				$message = ( ! empty( $ecardForToday['ecard_message'] ) ) ? $ecardForToday['ecard_message'] : '';

				if ( ! empty( $ecardForToday['sender_name'] ) ? $ecardForToday['sender_name'] : '' ) {

					$message = str_replace( '{sender_name}', $ecardForToday['sender_name'], $message );
				}
				$message = str_replace( '{sender_email}', $ecardForToday['from'], $message );
				$message = str_replace( '{site_url}', esc_url( home_url() ), $message );
				$message = str_replace( '{website_name}', get_bloginfo( 'name' ), $message );

				$image = ( ! empty( $ecardForToday['ecard_image'] ) ) ? $ecardForToday['ecard_image'] : '';
				$recipient_name = ( ! empty( $ecardForToday['recipient_name'] ) ) ? $ecardForToday['recipient_name'] : '';
				$footer = ( ! empty( $this->dboptions['ecard_footer'] ) ) ? $this->dboptions['ecard_footer'] : '';

				$show_visitsite_lbl = apply_filters( 'ecp_display_sitelink', true );
				$content = array(
					'recepient_name' => $name,
					'title' => $title,
					'subject' => $subject,
					'message' => $message,
					'image' => $image,
					'link' => esc_url( site_url() ),
				);

				if ( $show_visitsite_lbl ) {
					$content['value'] = $visit_site_label;
				}

				$email->email_hero_image( $content );
				$email->email_1_col_text_btn( $content );
				$ecardmarkup = $email->email_content( array( 'extra_footer' => $this->dboptions['ecard_footer'] ) );
				$ecardmarkup = apply_filters( 'ecp_ecard_markup', $ecardmarkup, $content );
				$placeholders = array(
					'recipient_name' => $recipient_name,
					'ecard_image' => $image,
					'default_subject' => $subject,
					'ecard_message' => $message,
					'ecard_footer' => $footer,
					'sitename' => get_bloginfo( 'name' ),
					'website_name' => get_bloginfo( 'name' ),
					'website_url' => esc_url( home_url() ),
					'siteurl' => esc_url( home_url() ),
				);

				foreach ( $placeholders as $key => $value ) {
					$ecardmarkup = str_replace( '{' . $key . '}', $value, $ecardmarkup );
				}

				if ( isset( $ecardForToday['order_id'] ) ) {
					return apply_filters( 'wep_email_details', wp_unslash( $ecardmarkup ), $ecardForToday['order_id'], $ecardForToday['ecard_id'] );
				} else {
					return wp_unslash( $ecardmarkup );
				}
			}

		}

		/**
		 * Perform tasks on plugin deactivation.
		 */
		function plugin_deactivation_work() {

			wp_clear_scheduled_hook( 'deliver_woo_ecards' );
		}

		function hook_in_admin_footer() {

			$screen = get_current_screen();
			if ( $screen->id == 'woocommerce-ecards-pro_page_ecp_form_sendecards' ) {
				$info = array(
					'ecard_image' => '',
					'ecard_id' => null,
				);
				$ecardPreviewModal = $this->get_preview_markup( $info );
				echo '<div class="preview_for_admin">' . $ecardPreviewModal . '</div>';
			}
			
			if ( isset( $this->dboptions['display_column_for_ecard_product'] ) && ! empty( $this->dboptions['display_column_for_ecard_product'] ) ) {
				echo '<style>.wp-admin.post-type-product table.wp-list-table .column-name{width:20%!important;}
			      .wp-admin.post-type-product table.wp-list-table td.column-ecard_product_info{width:2%!important;}</style>';
			
			}

		}

		/**
		 * Perform tasks on plugin deactivation.
		 */

		function register_ecards_delivery_event() {

			$allowed_shedules = array( 'deliver_woo_ecards', 'hourly', 'twicedaily', 'daily' );
			$settings = maybe_unserialize( get_option( 'wp-ecards-pro' ) );
			$cron_interval = ( isset( $settings['ecard_cron'] ) && ! empty( $settings['ecard_cron'] ) ) ? $settings['ecard_cron'] : 'deliver_woo_ecards';
			$cron_interval = ( in_array( $cron_interval, $allowed_shedules ) ) ? $cron_interval : 'deliver_woo_ecards';
			if ( ! wp_next_scheduled( 'deliver_woo_ecards' ) ) {
				wp_schedule_event( time(), $cron_interval, 'deliver_woo_ecards' );
			}

		}


		function plugin_activation_work() {

			$default_data = get_option( 'wp-ecards-pro' );

			if ( empty( $default_data ) ) {

				$default_data = ECP_Fresh_Settings::get_fresh_settings();
				$default_data = apply_filters( 'ecp_setup_initials', $default_data );
				update_option( 'wp-ecards-pro', $default_data );

			}

			$this->register_ecards_delivery_event();

		}

		function load_frontend_resources() {

			if ( is_product() ) {

				wp_enqueue_script( 'jquery' );
				wp_enqueue_script( 'jquery-ui-core' );
				wp_enqueue_script( 'jquery-ui-datepicker' );
				wp_enqueue_style( 'jquery-ui-css', plugin_dir_url( __FILE__ ) . '/assets/css/jquery-ui.css' );
			}

		}

		function hook_in_admin_head() {
			?>
		
		<style>#covert-product-ecard h2,#covert-product-ecard span{background-color:#a46497!important;color:#fff;}#preview_current_ecard,#send_ecards_btn{font-size:14px;}p.ecard-extra-info{font-size: .92em!important;color: #888;font-weight: bold;}p.ecard-extra-info span{font-weight:normal;}.wecp-success{background-color: #73a724;color: #fff;width: 50px;text-align: center;}.wecp-notify{background-color: #ffba00;color: #fff;width: 50px;text-align: center;}</style>
				
			<?php
		}

		/**
		 * Define all constants.
		 */
		function _define_constants() {

			if ( ! defined( 'ECP_SLUG' ) ) {
				define( 'ECP_SLUG', 'ecp_view_overview' );
			}

			if ( ! defined( 'ECP_PREMIUM' ) ) {
				define( 'ECP_PREMIUM', 'true' );
			}

			if ( ! defined( 'ECP_VERSION' ) ) {
				define( 'ECP_VERSION', '2.0.1' );
			}

			if ( ! defined( 'ECP_FOLDER' ) ) {
				define( 'ECP_FOLDER', basename( dirname( __FILE__ ) ) );
			}

			if ( ! defined( 'ECP_DIR' ) ) {
				define( 'ECP_DIR', plugin_dir_path( __FILE__ ) );
			}

			if ( ! defined( 'ECP_TEMPLATES' ) ) {
				define( 'ECP_TEMPLATES', ECP_DIR . 'templates/' );
			}

			if ( ! defined( 'ECP_CORE_CLASSES' ) ) {
				define( 'ECP_CORE_CLASSES', ECP_DIR . 'core/' );
			}

			if ( ! defined( 'ECP_PLUGIN_CLASSES' ) ) {
				define( 'ECP_PLUGIN_CLASSES', ECP_DIR . 'classes/' );
			}

			if ( ! defined( 'ECP_MODEL' ) ) {
				define( 'ECP_MODEL', ECP_DIR . 'modules/' );
			}

			if ( ! defined( 'ECP_URL' ) ) {
				define( 'ECP_URL', plugin_dir_url( ECP_FOLDER ) . ECP_FOLDER . '/' );
			}

			if ( ! defined( 'ECP_TEMPLATES_URL' ) ) {
				define( 'ECP_TEMPLATES_URL', ECP_URL . 'templates/' );
			}

			if ( ! defined( 'ECP_TEMPLATES_DIR' ) ) {
				define( 'ECP_TEMPLATES_DIR', ECP_DIR . 'templates/' );
			}

			if ( ! defined( 'ECP_CSS' ) ) {
				define( 'ECP_CSS', ECP_URL . '/assets/css/' );
			}

			if ( ! defined( 'ECP_JS' ) ) {
				define( 'ECP_JS', ECP_URL . '/assets/js/' );
			}

			if ( ! defined( 'ECP_IMAGES' ) ) {
				define( 'ECP_IMAGES', ECP_URL . '/assets/images/' );
			}

			if ( ! defined( 'ECP_FONTS' ) ) {
				define( 'ECP_FONTS', ECP_URL . 'fonts/' );
			}

		}

	}

	new Woo_Ecards_Pro();
}
